# -*- coding: utf-8 -*-
import os
import sys
import datetime
import random

from tqdm import tqdm

from intranet.yandex_directory.src.yandex_directory.core.models.organization import (
    OrganizationMetaModel,
    OrganizationModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models.department import DepartmentModel
from intranet.yandex_directory.src.yandex_directory.common.commands.base import TransactionalCommand, Option
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    create_organization,
    create_user,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import lstring, utcnow


def read_dict(filename):
    with open(filename) as f:
        return [line.strip() for line in f.readlines()
                if line and not line.startswith('#')]


class Command(TransactionalCommand):
    name = 'create-fake-organizations'
    option_list = (
        Option('--number', '-n', dest='number', type=int),
        Option('--show-uids', '-u', dest='show_uids', type=bool),
        Option(
            '--shard',
            '-s',
            dest='shard',
            type=int,
            required=False,
            default=1,
            help='Shard number.',
        )
    )

    def run(self, shard, number, show_uids=False):
        # ~17 сотрудников
        # max_users = 3
        # max_deps = 4
        # max_depth = 2
        # ~ 528 сотрудников
        # max_users = 30
        # max_deps = 8
        # max_depth = 3
        # ~ 11013 сотрудников
        # max_users = 30
        # max_deps = 8
        # max_depth = 4


        if not os.path.exists('words'):
            print ('Пожалуйста, склонируйте репозиторий со словарями: '
                   'https://github.yandex-team.ru/yandex-directory/words')
            sys.exit(1)


        self.long_list_of_words = read_dict('words/Unabr.dict')
        self.names = read_dict('words/actor-givenname')
        self.surnames = read_dict('words/actor-surname')

        orgs = tqdm(list(range(number)))
        for org_idx in orgs:
            with self.meta_connection.begin_nested(), \
                 self.main_connection.begin_nested():
                # случайным образом выбираем параметры для генерации организации
                max_users = random.randint(1, 30)
                max_deps = random.randint(1, 8)
                max_depth = random.randint(1, 3)

                organization_label = self.generate_organization_label()
                organization = self.get_or_create_organization(organization_label)

                self.create_departments_and_users(
                    main_db_connection=self.main_connection,
                    meta_db_connection=self.meta_connection,
                    organization=organization,
                    parent_id=1,
                    max_users=max_users,
                    max_deps=max_deps,
                    max_depth=max_depth,
                )

                if show_uids:
                    print(organization['label'], organization['admin_uid'])

    def get_or_create_organization(self, organization_label):
        organizations = OrganizationMetaModel(self.meta_connection).find({'label': organization_label})

        if organizations:
            organization = OrganizationModel(self.main_connection).get(
                id=organizations[0]['id']
            )
        else:
            name = {
                'ru': organization_label,
                'en': organization_label,
            }
            admin_nickname = 'admin_of_%s' % organization_label
            admin_uid = self.generate_uid()
            # TODO: непонятно, какую зону используем для фейковых данных?
            created_organization_info = create_organization(
                meta_connection=self.meta_connection,
                main_connection=self.main_connection,
                name=name,
                domain_part=organization_label+'.yaserv.biz',
                language='ru',
                label=organization_label,
                admin_nickname=admin_nickname,
                admin_uid=admin_uid,
                admin_first_name='Admin',
                admin_last_name='Adminoff',
                admin_gender='male',
                admin_birthday='1990-05-03',
            )
            organization = created_organization_info['organization']
            organization['admin_uid'] = admin_uid

        return organization

    def create_departments_and_users(
            self,
            main_db_connection,
            meta_db_connection,
            organization,
            parent_id,
            max_users,
            max_deps,
            max_depth):
        """Создаем некоторое количество подотделов и сотрудников в отделе parent_id.
        Количество подотделов выбирается случайным образом от 1 до max_deps,
        Количество сотрудников в каждом отделе от 1 до max_users.
        Далее, если max_depth > 0, то рекрсивно создаем подотделы в каждом из
        созданных отделов.
        """
        if max_depth > 0:
            num_sub_deps = random.randint(1, max_deps)
            for i in range(num_sub_deps):
                sub_dep = self.create_department(
                    connection=main_db_connection,
                    org_id=organization['id'],
                    parent_id=parent_id,
                )
                self.create_departments_and_users(
                    main_db_connection,
                    meta_db_connection,
                    organization,
                    sub_dep['id'],
                    max_users,
                    max_deps,
                    max_depth=max_depth-1
                )

        num_users = random.randint(1, max_users)
        for i in range(num_users):
            self.create_user(
                main_db_connection=main_db_connection,
                meta_db_connection=meta_db_connection,
                organization=organization,
                department_id=parent_id,
            )

    def create_department(self, connection, org_id, parent_id):
        name = self.generate_department_name()
        dep = DepartmentModel(connection=connection).create(
            name=name,
            parent_id=parent_id,
            org_id=org_id,
        )
        return dep

    def create_user(self,
                    main_db_connection,
                    meta_db_connection,
                    organization,
                    department_id):

        nickname = random.choice(self.names)
        email = self.generate_email(organization)
        user_data = {
            'id': self.generate_uid(),
            'org_id': organization['id'],
            'department_id': department_id,
            'nickname': nickname,
            'email': email,
            'name': self.generate_name(),
            'gender': self.generate_gender(),
            'contacts': self.generate_contacts(nickname),
            'position': self.generate_position(),
            'about': self.generate_about(),
            'birthday': self.generate_birthday(),
        }
        create_user(
            org_id=organization['id'],
            user_data=user_data,
            nickname=nickname,
            meta_connection=meta_db_connection,
            main_connection=main_db_connection,
        )


    def random_word(self):
            return random.choice(self.long_list_of_words)


    def random_words(self, num=4):
        num_words = random.randint(1, num)
        name = [self.random_word() for i in range(num_words)]
        return  ' '.join(name)


    def generate_organization_label(self):
        return self.random_word()


    def generate_department_name(self):
        return lstring(self.random_words())


    def generate_uid(self):
        return random.randint(1, sys.maxsize)


    def generate_name(self):
        name = random.choice(self.names)
        lastname = random.choice(self.surnames)
        return {
            'first': lstring(name),
            'last': lstring(lastname),
        }


    def generate_email(self, organization):
        return '{0}@{1}.ws.yandex.ru'.format(
            random.choice(self.names),
            organization['label']
        )


    def generate_gender(self):
        if random.random() > 0.5:
            return 'male'
        else:
            return 'female'


    def generate_contacts(self, nickname):
        return [
            {
                'type': 'email',
                'label': {'ru': 'Email', 'en': 'Email'},
                'value': nickname,
            },
            {
                'type': 'skype',
                'label': {'ru': 'Skype', 'en': 'Skype'},
                'value': nickname,
            },
            {
                'type': 'twitter',
                'label': {'ru': 'Twitter', 'en': 'Twitter'},
                'value': nickname,
            }
        ]


    def generate_position(self):
        return lstring(self.random_words(2))


    def generate_about(self):
        return lstring(self.random_words(42))


    def generate_birthday(self):
        born_this_days_ago = random.randint(18 * 365, 50 * 365)
        return utcnow().date() - datetime.timedelta(born_this_days_ago)
