# -*- coding: utf-8 -*-
import re
import sys
import json
import os
import os.path
import commonmark
import flask_swagger
import inspect
import yaml

from intranet.yandex_directory.src.yandex_directory.common.commands.base import BaseCommand
from intranet.yandex_directory.src.yandex_directory.swagger import _schema_to_swagger
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.setup import initialize_routes
from jsonschema import Draft4Validator
from collections import defaultdict

from library.python import resource  # noqa


def read_asset(filename) -> bytes:
    return resource.resfs_read(filename)


def _validate_spec_or_die(doc):
    path_docs = os.environ.get("PATH_DOCS")
    is_it_autotests = (os.environ.get("ENVIRONMENT") == "autotests")

    if path_docs is None or is_it_autotests:
        path_docs = os.path.join(app.config['BASE_PATH'], 'docs')

    spec_path = os.path.join(
        path_docs,
        'swagger-spec-schema.json')

    schema = json.loads(read_asset(spec_path))

    errors = list(Draft4Validator(schema).iter_errors(doc))
    if errors:
        num_errors = len(errors)
        for idx, error in enumerate(errors):
            print(error)
            last_error = idx == num_errors - 1
            if not last_error:
                print('\n' + '=' * 40 + '\n')
        if num_errors > 1:
            print('{0} errors'.format(num_errors))
        else:
            print('{0} error'.format(num_errors))
        sys.exit(1)


def _process_doc(text):
    parser = commonmark.Parser()
    renderer = commonmark.HtmlRenderer()
    if isinstance(text, bytes):
        text = text.decode('utf-8')
    ast = parser.parse(text)
    return renderer.render(ast)


# Утащено из flask_swagger, так как нам нужно вставлять в докстринг
# схему из метода
def _parse_docstring(obj, process_doc, from_file_keyword, api_version):
    first_line, other_lines, swag = None, None, None
    full_doc = inspect.getdoc(obj)

    if full_doc:
        if from_file_keyword is not None:
            from_file = flask_swagger._find_from_file(full_doc, from_file_keyword)
            if from_file:
                full_doc_from_file = flask_swagger._doc_from_file(from_file)
                if full_doc_from_file:
                    full_doc = full_doc_from_file
        line_feed = full_doc.find('\n')
        if line_feed != -1:
            first_line = process_doc(full_doc[:line_feed])
            yaml_sep = full_doc[line_feed+1:].find('---')
            if yaml_sep != -1:
                other_lines = process_doc(full_doc[line_feed+1:line_feed+yaml_sep])
                swag = yaml.load(full_doc[line_feed+yaml_sep:], Loader=yaml.FullLoader)
            else:
                other_lines = process_doc(full_doc[line_feed+1:])
        else:
            first_line = full_doc

        if swag and hasattr(obj, '__schema__'):
            if 'parameters' in swag:
                schema = _schema_to_swagger(obj.__schema__, api_version)
                for item in swag['parameters']:
                    if item['in'] == 'body':
                        item['schema'] = schema
    return first_line, other_lines, swag


# Эта функция утащена из flask_swagger и запатчена так, чтобы
# понимать разные версии API Директории
# Первым параметром идёт номер версии API, для которой нужно сгенерить
# swagger спеку.
def swagger(api_version,
            app,
            ignored_prefix=None,
            process_doc=flask_swagger._sanitize,
            from_file_keyword=None,
            template=None):
    """
    Call this from an @app.route method like this
    @app.route('/spec.json')
    def spec():
       return jsonify(swagger(app))

    We go through all endpoints of the app searching for swagger endpoints
    We provide the minimum required data according to swagger specs
    Callers can and should add and override at will

    Arguments:
    app -- the flask app to inspect

    Keyword arguments:
    process_doc -- text sanitization method, the default simply replaces \n with <br>
    from_file_keyword -- how to specify a file to load doc from
    template -- The spec to start with and update as flask-swagger finds paths.
    """
    output = {
        "swagger": "2.0",
        "info": {
            "version": "0.0.0",
            "title": "Cool product name",
        }
    }
    paths = defaultdict(dict)
    definitions = defaultdict(dict)
    if template is not None:
        output.update(template)
        # check for template provided paths and definitions
        for k, v in list(output.get('paths', {}).items()):
            paths[k] = v
        for k, v in list(output.get('definitions', {}).items()):
            definitions[k] = v
    output["paths"] = paths
    output["definitions"] = definitions

    ignore_verbs = {"HEAD", "OPTIONS"}
    # technically only responses is non-optional
    optional_fields = ['tags', 'consumes', 'produces', 'schemes', 'security',
                       'deprecated', 'operationId', 'externalDocs']

    for rule in app.url_map.iter_rules():
        if ignored_prefix and rule.rule.startswith(ignored_prefix):
            continue
        endpoint = app.view_functions[rule.endpoint]
        methods = dict()
        for verb in rule.methods.difference(ignore_verbs):
            verb = verb.lower()
            if hasattr(endpoint, 'methods') and verb in list(map(str.lower, endpoint.methods)):
                # Определим метод, отвечающий за обработку HTTP метода
                # в той версии API, для которой сейчас генерим спеку
                if endpoint.view_class.__name__ in app.config['VIEW_CLASSES_WITHOUT_VERSION']:
                    method_name = endpoint.view_class._methods_map[(verb, 1)]
                else:
                    if not (verb, api_version) in endpoint.view_class._methods_map:
                        # если ручка не поддержиавет эту версию API, то ничего не делаем
                        continue
                    method_name = endpoint.view_class._methods_map[(verb, api_version)]

                methods[verb] = getattr(endpoint.view_class, method_name)
            else:
                methods[verb] = endpoint
        operations = dict()
        for verb, method in list(methods.items()):
            summary, description, swag = _parse_docstring(
                method,
                process_doc,
                from_file_keyword,
                api_version,
            )
            if swag is not None:  # we only add endpoints with swagger data in the docstrings
                defs = swag.get('definitions', [])
                defs = flask_swagger._extract_definitions(defs)
                params = swag.get('parameters', [])
                defs += flask_swagger._extract_definitions(params)
                responses = swag.get('responses', {})
                responses = {
                    str(key): value
                    for key, value in list(responses.items())
                }
                if responses is not None:
                    defs = defs + flask_swagger._extract_definitions(list(responses.values()))
                for definition in defs:
                    def_id = definition.pop('id')
                    if def_id is not None:
                        definitions[def_id].update(definition)
                operation = dict(
                    summary=summary,
                    description=description,
                    responses=responses
                )
                # parameters - swagger ui dislikes empty parameter lists
                if len(params) > 0:
                    operation['parameters'] = params
                # other optionals
                for key in optional_fields:
                    if key in swag:
                        operation[key] = swag.get(key)
                operations[verb] = operation

        if len(operations):
            rule = '/v{0}{1}'.format(api_version, rule)
            for arg in re.findall('(<([^<>]*:)?([^<>]*)>)', rule):
                rule = rule.replace(arg[0], '{%s}' % arg[2])
            paths[rule].update(operations)
    return output


class Command(BaseCommand):
    name = 'create-swagger-spec'

    def run(self):
        initialize_routes(app, internal_api=True)

        api_versions = app.config['SUPPORTED_API_VERSIONS']

        base_path = app.config['BASE_PATH']
        index_template = read_asset(os.path.join(base_path, 'docs', 'index.js')).decode()
        oauth_config = app.config['PLAYGROUND_OAUTH_CONFIG']

        for version in api_versions:
            doc = swagger(version, app, process_doc=_process_doc, ignored_prefix='/v<int:api_version>/')
            doc['info']['title'] = 'yandex-directory'

            with open('swagger.json', 'w') as swagger_file:
                swagger_file.write(json.dumps(doc, indent=4, sort_keys=True))

            _validate_spec_or_die(doc)

            new_content = index_template.replace(
                'SWAGGER_SPEC_PLACEHOLDER',
                json.dumps(doc)
            )
            new_content = new_content.replace(
                'SWAGGER_OAUTH_CONFIG',
                json.dumps(oauth_config)
            )
            filename = 'index-with-spec-{0}.js'.format(version)
            with open(filename, 'w') as index_js_file:
                index_js_file.write(new_content)

        # В файлике с версиями сохраним списочек, чтобы можно было нарисовать
        # переключалку
        with open('versions.js', 'w') as f:
            f.write('window.api_versions = [{0}];\n'.format(
                ', '.join(map(str, api_versions))
            ))
