# -*- coding: utf-8 -*-

from tqdm import tqdm

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.commands.base import BaseCommand
from intranet.yandex_directory.src.yandex_directory.common.commands.base import Option
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection,
    get_meta_connection,
    get_shard_numbers
)
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.common.utils import utcnow, format_date
from intranet.yandex_directory.src.yandex_directory.common.yt import utils as yt_utils
from intranet.yandex_directory.src.yandex_directory.core.features import (
    CAN_WORK_WITHOUT_OWNED_DOMAIN,
)
from intranet.yandex_directory.src.yandex_directory.core.features.utils import (
    set_feature_value_for_organization,
)
from intranet.yandex_directory.src.yandex_directory.core.models import (
    OrganizationModel,
    OrganizationMetaModel,
    UserModel,
    UserMetaModel,
)
from intranet.yandex_directory.src.yandex_directory.core.predicates import (
    And,
    Not,
)
from intranet.yandex_directory.src.yandex_directory.core.permission.rules import is_yandex_organization
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    only_attrs,
    get_user_data_from_blackbox_by_uid,
)

YANDEX_ALIAS_ID = '1'
LIGHT_ALIAS_ID = '5'
SOCIAL_ALIAS_ID = '6'


class Command(BaseCommand):
    """
    Включение фичи ЯОрг (can-work-without-owned-domains)
    Пока включаем фичу в организациях, у которых:
    1. Нет подтвержденного домена или есть только тех. домен
    2. Не включен расшренный коннект и не включен Трекер
    3. Не включена фича ЯОрг
    4. Нет социальных и лайт-аккаунтов (алиасы с id 5 и 6 в паспорте).
       При этом, если есть ещё алиас c id = 1, то включать фичу можно - этот аккаунт дорегестрирован.
    """
    name = 'enable-yaorg'
    option_list = (
        Option('--limit', '-l', dest='limit', type=int,
               required=False, help='Process only this number of org in each shard'),
        Option('--verbose', '-v', dest='verbose', action='store_true',
               required=False, help='Show orgs'),
        Option('--no-dry-run', '-n', dest='no_dry_run', action='store_true',
               required=False, help='Enable yaorg'),
    )

    def run(self, limit=None, verbose=False, no_dry_run=False):
        filters = {
            'subscription_plan': 'free',
        }
        rule = And(
            Not(is_yandex_organization),
            has_not_social_or_light_accounts,
        )
        table_path = '//home/yandex-connect/{}/analytics/enable-yaorg/{}'.format(
            app.config['ENVIRONMENT'],
            format_date(utcnow(), only_date=True),
        )

        with yt_utils.with_table(table_path, org_id='int64', admin_email='string') as add_row:
            shards = get_shard_numbers()

            for shard in shards:
                if shard == 4:
                    # Порталам пока не включаем
                    continue
                else:
                    if verbose:
                        print(('\n\nShard {0}: checking'.format(shard)))
                          
                    with get_main_connection(shard=shard) as main_connection, \
                         tqdm(total=limit) as total_enabled:
                        total = process(main_connection, total_enabled, no_dry_run, rule, add_row, **filters)
                        if verbose:
                            print(('\nShard {0}: feature enabled in {1} organizations'.format(shard, total)))

def process(main_connection, total_enabled, no_dry_run, rule, add_row, **filters):

    all_organizations = iterate_over_organizations(main_connection, **filters)

    for org in all_organizations:
        if total_enabled.total and total_enabled.n >= total_enabled.total:
            return total_enabled.n

        with get_meta_connection() as meta_connection, \
             get_main_connection(shard=main_connection.shard) as main_connection:
            org_id = org['id']
            with log.name_and_fields('enable-yaorg', org_id=org_id):
                cache = {}
                if rule.evaluate(
                    cache,
                    meta_connection=meta_connection,
                    main_connection=main_connection,
                    org_id=org_id,
                ):
                    admins = get_admins(meta_connection, main_connection, org_id)
                    if admins:
                        for admin_email in admins:
                            add_row(org_id=org_id, admin_email=admin_email)
                    else:
                        add_row(org_id=org_id, admin_email=None)

                    total_enabled.update(1)
                    if no_dry_run:
                        enable_yaorg(org_id)

    return total_enabled.n


def get_admins(meta_connection, main_connection, org_id):
    inner_admins = only_attrs(
        UserModel(main_connection) \
            .filter(org_id=org_id, role='admin', is_dismissed=False) \
            .fields('email'),
        'email',
    )
    outer_admins_uids = only_attrs(
        UserMetaModel(meta_connection) \
            .filter(org_id=org_id, user_type='outer_admin') \
            .fields('id'),
        'id',
    )
    outer_admins = []
    for uid in outer_admins_uids:
        user_info = get_user_data_from_blackbox_by_uid(uid)
        if not user_info:
            continue
        else:
            outer_admins.append(user_info['default_email'])
    return inner_admins + outer_admins


def iterate_over_organizations(main_connection, batch_size=100, **filters):
    max_org_id = 0

    query = OrganizationModel(main_connection) \
        .filter(id__gt=max_org_id, **filters) \
        .fields('id') \
        .order_by('id') \
        .limit(batch_size)

    while True:
        items = query.all()
        if not items:
            return

        for item in items:
            max_org_id = item['id']
            yield item

        query = query.filter(id__gt=max_org_id)


def get_orgs_count(main_connection, **filters):
    return OrganizationModel(main_connection).filter(**filters).count()


def enable_yaorg(org_id):
    with get_meta_connection(for_write=True) as meta_connection:
        if not OrganizationMetaModel(meta_connection).filter(id=org_id).one():
            log.info('Org already not exists')
            return
        set_feature_value_for_organization(
            meta_connection,
            org_id,
            CAN_WORK_WITHOUT_OWNED_DOMAIN,
            True,
        )


def has_not_social_or_light_accounts(meta_connection, org_id, **kwargs):
    outer_admins_uids = only_attrs(
        UserMetaModel(meta_connection) \
            .filter(org_id=org_id, user_type='outer_admin') \
            .fields('id'),
        'id',
    )
    # 'fields': {'aliases': [('1', 'test-fb'), ('6', 'test-111')]},
    for uid in outer_admins_uids:
        user_info = app.blackbox_instance.userinfo(uid=uid, userip='127.0.0.1')
        if not user_info.get('fields').get('aliases'):
            continue
        else:
            aliases_id = [alias[0] for alias in user_info.get('fields').get('aliases')]
            if YANDEX_ALIAS_ID not in aliases_id and \
                    (SOCIAL_ALIAS_ID in aliases_id or LIGHT_ALIAS_ID in aliases_id):
                return False
    return True
