# -*- coding: utf-8 -*-
import sys
import json
import os.path
import commonmark
import flask_swagger

from intranet.yandex_directory.src.yandex_directory.common.commands.base import BaseCommand
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.setup import is_internal_view


def get_schemas(cls):
    """Возвращает список из tuple типа:
    [
      (get_5, schema),
      (post, another_schema),
    ]
    """
    result = []

    # Составим список суффиксов, которые будем добавлять к get/post/put/delete
    # при поиске ручек во вьюхах.
    handle_suffixes = [
        '_{0}'.format(api_version)
        for api_version in app.config['SUPPORTED_API_VERSIONS']
    ]
    handle_suffixes.reverse()
    # Напоследок будем проверять существование метода без ручек
    handle_suffixes.append('')

    for method in sorted(cls.methods):
        for suffix in handle_suffixes:
            method_name = method.lower() + suffix
            func = getattr(cls, method_name, None)
            if func:
                schema = getattr(func, '__schema__', None)
                if schema:
                    result.append((method_name, schema))

    return result


def format_schema(schema, offset=4):
    """Возвращает схему в виде строки.

    При этом словарь с описанием схемы, прогнан через pretty-printer,
    и к каждой строке добавлен отступ в offset пробелов.
    """
    serialized = json.dumps(schema, indent=4, sort_keys=True)
    lines = serialized.split('\n')
    offset = ' ' * offset
    lines = (offset + line
             for line in lines)
    result = '\n'.join(lines)
    return result


def write_schema_to_file(handle, schema, f):
    f.write("""
{handle}
{header}

.. code::

{schema}
""".format(
    handle=handle,
    header='-' * len(handle), # подчеркивание заголовка
    schema=format_schema(schema),
))

class Command(BaseCommand):
    name = 'export-json-schemas'

    def run(self):
        with open('schemas.rst', 'w') as f:
            f.write("""===============
Публичные ручки
===============

""")
            for route, view_class in app.ROUTES:
                if not is_internal_view(view_class):
                    schemas = get_schemas(view_class)
                    if schemas:
                        f.write("""
{route}
{header}

""".format(route=route, header='=' * len(route)))

                        for method, schema in schemas:
                            write_schema_to_file(method, schema, f)
