# -*- coding: utf-8 -*-
import re
import json
import click

from itertools import islice
from collections import defaultdict
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from tqdm import tqdm

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.commands.base import AllShardsCommand, Option
from intranet.yandex_directory.src.yandex_directory.core.models import (
    DomainModel,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import find_domid, to_punycode
from intranet.yandex_directory.src.yandex_directory.passport.exceptions import AccountDisabled, PassportUnavailable


def get_domain_count(main_connection):
    return DomainModel(main_connection) \
        .filter(master=True) \
        .count()


def iterate_over_domains(main_connection, batch_size=100):
    max_org_id = 0
    query = DomainModel(main_connection) \
        .filter(master=True, owned=True, org_id__gt=max_org_id) \
        .fields('org_id', 'name') \
        .order_by('org_id') \
        .limit(batch_size)

    while True:
        items = query.all()
        if not items:
            return
        
        for item in items:
            max_org_id = item['org_id']
            yield item

        query = query.filter(org_id__gt=max_org_id)


class BadOrgName(RuntimeError):
    pass


def get_domain_organization_name(hosted_domains, name):
    """Ищет среди доменов отданных блекбоксом, домен с заданным именем.

       Параметр name должен быть в punicode.
    """
    # локально т.к. иначе циклический импорт
    if not hosted_domains:
        raise BadOrgName('No hosted_domains in Passport')

    punycode_name = to_punycode(name.lower())
    for domain in hosted_domains:
        if domain['domain'].lower() == punycode_name:
            options_as_str = domain['options']
            if options_as_str == '':
                raise BadOrgName('Domain options is empty string')
            try:
                options = json.loads(options_as_str)
            except ValueError:
                raise BadOrgName('Unable to decode json')
            
            return options.get('organization_name')

        
def get_correct_organization_name(domain):
    org_id = domain['org_id']
    return 'org_id:{}'.format(org_id)


def is_domain_bad(domain):
    domain_name = domain['name']
    hosted_domains = app.blackbox_instance.hosted_domains(domain=domain['name'])
    hosted_domains = hosted_domains['hosted_domains']
    
    try:
        organization_name = get_domain_organization_name(hosted_domains, domain['name'])
    except BadOrgName as e:
        return str(e)
    
    if organization_name is None:
        return 'Organization name is None'
    if organization_name == '':
        return 'Organization name is empty string'
    
    matched = re.match('^org_id:\d+$', organization_name)
    if not matched:
        return 'Organization name is not looks like "org_id:123456"'

    # Теперь проверим, что если даже формат записи правильный, то org_id
    # в ней соответствует тому, которму сейчас принадлежит домен.
    # Это нужно потому, что запись могла не обновиться при передаче домена.
    correct_name = get_correct_organization_name(domain)
    if organization_name != correct_name:
        return 'Wrong org_id'
    
    return False


def fix_domain(domain):
    domain_name = domain['name']

    with log.fields(domain=domain_name):
        log.warning('Setting correct organization name in passport')
        hosted_domains = app.blackbox_instance.hosted_domains(domain=domain_name)
        dom_id = find_domid(hosted_domains, domain_name)
        org_name = get_correct_organization_name(domain)
        app.passport.set_organization_name(dom_id, org_name)


def process_domains(main_connection, limit=None, dry_run=False, progress=False):
    results = defaultdict(list)
    all_domains = iterate_over_domains(main_connection)
    
    total = 0
    if limit:
        num_domains = limit
        domains = islice(all_domains, 0, limit)
    else:
        num_domains = get_domain_count(main_connection)
        domains = all_domains

    if progress:
        domains = tqdm(domains, total=num_domains)
        
    for domain in domains:
        error = is_domain_bad(domain)
        results[error].append(domain)
        total += 1

        if error and not dry_run and error != 'No hosted_domains in Passport':
            try:
                fix_domain(domain)
            except PassportUnavailable:
                results['Passport Unavalable'].append(domain)
            except AccountDisabled:
                results['Default UID disabled'].append(domain)

    return total, results


class Command(AllShardsCommand):
    """Одноразовая команда, нужна для того, чтобы устранить неправильные имена организаций в паспорте.
       https://st.yandex-team.ru/DIR-5968
    """
    name = 'normalize-org-names'
    option_list = (
        Option('--limit', '-l', dest='limit', type=int, required=False, help='Process only this number of domains in each shard'),
        Option('--skip-shards', '-s', dest='skip_shards', required=False, help='Skip these shards (comma-separated)'),
        Option('--verbose', dest='verbose', action='store_true', help='Show domains'),
        Option('--progress', dest='progress', action='store_true', help='Show progress bar'),
        Option('--no-dry-run', dest='no_dry_run', action='store_true', help='Fix organization name in Passport'),
    )


    def run(self, verbose=False, no_dry_run=False, limit=None, progress=False, skip_shards=None, *args, **kwargs):
        skip_shards = list(map(int, [_f for _f in (skip_shards or '').split(',') if _f]))
        if self.shard in skip_shards:
            click.echo('Пропускаю шард {}'.format(self.shard))
            return

        click.echo('Шард: {0}'.format(self.shard))
        total, results = process_domains(
            self.main_connection,
            dry_run=not no_dry_run,
            limit=limit,
            progress=progress,
        )
        click.echo('Всего проверено: {0}'.format(total))
        click.echo('')
        
        errors = list(results.items())
        errors.sort(key=lambda item: len(item[1]), reverse=True)
        errors = [item
                  for item in errors
                  if item[0] is not False]
        
        if errors:
            click.echo('Встреченные ошибки:')
            for error, domains in errors:
                click.echo('  {0}: {1}'.format(error, len(domains)))
                if verbose and error != 'No hosted_domains in Passport':
                    domain_names = sorted(d['name'] for d in domains)
                    click.echo('  Домены: {0}'.format(
                        ', '.join(domain_names)
                    ))


            if 'No hosted_domains in Passport' in results:
                click.echo('')
                click.echo('Ошибки "No hosted_domains in Passport" были проигнорированы, так как с ними сделать ничего нельзя.')
        else:
            click.echo('Ошибок нет.')

        click.echo('')
        click.echo('')
