# -*- coding: utf-8 -*-

from tqdm import tqdm
import sqlalchemy
from intranet.yandex_directory.src.yandex_directory.core.models import (
    GroupModel,
    DepartmentModel,
    OrganizationModel,
    OrganizationServiceModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models.domain import DomainNotFound
from intranet.yandex_directory.src.yandex_directory.common.commands.base import (
    AllShardsCommand,
    Option,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    only_ids,
    create_maillist,
)
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log


class Command(AllShardsCommand):
    """
    Заполняем uid рассылок для групп и отделов в базе
    """
    name = 'update-maillist-uids'
    option_list = (
        Option('--org-id', '-o', dest='org_id', type=int, required=False, help='Organization id'),
        Option('--force', '-f', dest='force_update', action='store_true', required=False),
        Option('--progress', '-p', action='store_true', required=False),
    )

    def run(self, org_id=None, force_update=False, progress=False):
        filter_data = {'label__isnull': False}
        if not force_update:
            filter_data['uid__isnull'] = True
        if org_id:
            org_ids = only_ids(OrganizationModel(self.main_connection).filter(id=org_id))
        else:
            org_ids = self.main_connection.execute("""
SELECT id FROM organizations
 WHERE EXISTS (
            SELECT 1
            FROM departments
            WHERE label is not null
            AND uid is null
            AND org_id = organizations.id
       )
       OR EXISTS (
            SELECT 1
            FROM groups
            WHERE label is not null
            AND uid is null
            AND org_id = organizations.id
       )
""").fetchall()
            org_ids = [row[0] for row in org_ids]

        if progress:
            org_ids = tqdm(org_ids)

        from intranet.yandex_directory.src.yandex_directory.passport.exceptions import DomainInvalidType
        for org_id in org_ids:
            with log.fields(org_id=org_id):
                try:
                    with self.main_connection.begin_nested():
                        fix_organization(self.meta_connection, self.main_connection, org_id, filter_data)
                except DomainInvalidType:
                    log.trace().error('Invalid domain type')
                except sqlalchemy.exc.IntegrityError:
                    log.trace().error('Integrity error')
                except Exception:
                    log.trace().error('Unhandled error')


def fix_organization(meta_connection, main_connection, org_id, filter_data):
    from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
        get_domains_from_db_or_domenator,
        DomainFilter,
        update_domains_in_db_or_domenator,
        DomainUpdateFilter,
        DomainUpdateData,
    )

    filter_data['org_id'] = org_id
    try:
        master_domain_info = get_domains_from_db_or_domenator(
            meta_connection=meta_connection,
            domain_filter=DomainFilter(org_id=org_id, master=True),
            main_connection=main_connection,
            one=True,
        )
        master_domain = None
        if master_domain_info:
            master_domain = master_domain_info['name']
        if master_domain in app.config['YANDEX_TEAM_ORG_DOMAINS']:
            return
    except DomainNotFound:
        master_domain = None

    new_maillists_enabled = OrganizationServiceModel(main_connection).is_service_enabled(org_id, 'maillist')

    # Проверим, есть ли домен в Паспорте, потому что иногда бывает так, что у нас он подтверждён, а в
    # паспорте его почему-то нет вовсе
    exists_in_passport = master_domain and app.blackbox_instance.hosted_domains(domain=master_domain)['hosted_domains']

    if not exists_in_passport:
        update_domains_in_db_or_domenator(
            meta_connection,
            DomainUpdateFilter(org_id=org_id, name=master_domain),
            DomainUpdateData(owned=False, master=False),
            main_connection=main_connection,
        )

    # Иногда бывает, что домен мастер, но не подтверждён:
    # https://catalog.ws.yandex-team.ru/organizations/550469/domains
    has_owned_domain = bool(get_domains_from_db_or_domenator(
        meta_connection=meta_connection,
        domain_filter=DomainFilter(org_id=org_id, owned=True),
        main_connection=main_connection,
        one=True,
    ))


    # Подчистим уиды в командах, которые были удалены
    main_connection.execute('UPDATE groups SET uid=NULL WHERE org_id=%(org_id)s AND removed = TRUE AND uid IS NOT NULL',
                            org_id=org_id)
    main_connection.execute('UPDATE departments SET uid=NULL WHERE org_id=%(org_id)s AND removed = TRUE AND uid IS NOT NULL',
                            org_id=org_id)

    for group in GroupModel(main_connection).filter(**filter_data).fields('label', 'id', 'uid'):
        try:
            if not group['uid']:
                if has_owned_domain:
                    uid = create_maillist(
                        main_connection,
                        org_id,
                        group['label'],
                        ignore_login_not_available=True,
                    )

                    GroupModel(main_connection) \
                        .filter(org_id=org_id, id=group['id']) \
                        .update(uid=uid)
                else:
                    # Если нет ни одного подтверждённого домена
                    GroupModel(main_connection) \
                        .filter(org_id=org_id, id=group['id']) \
                        .update(label=None)
        except DomainNotFound:
            log.trace().error('Domain not found')

    for department in DepartmentModel(main_connection).filter(**filter_data).fields('label', 'id', 'uid'):
        if not department['uid']:
            if has_owned_domain:
                uid = create_maillist(
                    main_connection,
                    org_id,
                    department['label'],
                    ignore_login_not_available=True,
                )

                DepartmentModel(main_connection) \
                    .filter(org_id=org_id, id=department['id']) \
                    .update(uid=uid)
            else:
                # Если нет ни одного подтверждённого домена
                DepartmentModel(main_connection) \
                    .filter(org_id=org_id, id=department['id']) \
                    .update(label=None)
