CREATE SCHEMA IF NOT EXISTS ydir;

SET SCHEMA 'ydir';



ALTER ROLE ydir SET search_path TO ydir, public;
CREATE EXTENSION IF NOT EXISTS ltree;


CREATE TABLE ydir.organizations (
    id BIGINT NOT NULL,
    label TEXT NOT NULL,
    name JSONB NOT NULL,
    robot_uid BIGINT NOT NULL,

    CONSTRAINT pk_organizations PRIMARY KEY (id)
);

CREATE UNIQUE INDEX uk_organizations_label ON ydir.organizations (label);


CREATE TABLE ydir.groups (
    org_id BIGINT NOT NULL,
    id BIGINT NOT NULL,
    name JSONB NOT NULL,
    type TEXT NOT NULL,
    description JSONB,

    CONSTRAINT pk_groups PRIMARY KEY (org_id, id),
    CONSTRAINT fk_groups_org_id
        FOREIGN KEY (org_id) REFERENCES organizations (id)
        ON DELETE CASCADE
);


CREATE TABLE ydir.departments (
    org_id BIGINT NOT NULL,
    id INT NOT NULL,
    parent_id INT,
    name JSONB NOT NULL,
    path ltree,
    description JSONB,
    heads_group_id BIGINT,

    CONSTRAINT pk_departments PRIMARY KEY (org_id, id),
    CONSTRAINT fk_departments_org_id
        FOREIGN KEY (org_id) REFERENCES organizations (id)
        ON DELETE CASCADE,
    CONSTRAINT fk_departments_org_id_parent_id
        FOREIGN KEY (org_id, parent_id) REFERENCES departments (org_id, id),
    CONSTRAINT fk_departments_org_id_heads_group_id
        FOREIGN KEY (org_id, heads_group_id) REFERENCES groups (org_id, id)
);

CREATE INDEX i_departments_path ON ydir.departments USING gist(path);


CREATE TABLE ydir.users (
    id BIGINT NOT NULL,
    org_id BIGINT NOT NULL,
    login TEXT NOT NULL,
    email TEXT NOT NULL,
    department_id INT NULL,
    name JSONB NOT NULL,
    gender TEXT NOT NULL,  -- male or female
    position JSONB,
    about JSONB,
    birthday DATE,
    contacts JSONB,

    CONSTRAINT pk_users PRIMARY KEY (id),
    CONSTRAINT fk_users_org_id
        FOREIGN KEY (org_id) REFERENCES organizations (id),
    CONSTRAINT fk_users_org_id_department_id
        FOREIGN KEY (org_id, department_id) REFERENCES departments (org_id, id)
);

CREATE UNIQUE INDEX uk_users_org_id_id ON ydir.users (org_id, id);


CREATE TABLE ydir.user_group_membership (
    org_id BIGINT NOT NULL,
    user_id BIGINT NOT NULL,
    group_id BIGINT NOT NULL,

    CONSTRAINT pk_user_group_membership PRIMARY KEY (org_id, user_id, group_id),
    CONSTRAINT fk_user_group_membership_user_id
        FOREIGN KEY (user_id) REFERENCES users (id)
        ON DELETE CASCADE,
    CONSTRAINT fk_user_group_membership_org_id_group_id
        FOREIGN KEY (org_id, group_id) REFERENCES groups (org_id, id)
        ON DELETE CASCADE
);


CREATE TABLE ydir.department_group_membership (
    org_id BIGINT NOT NULL,
    department_id INT NOT NULL,
    group_id BIGINT NOT NULL,

    CONSTRAINT pk_department_group_membership PRIMARY KEY (org_id, department_id, group_id),
    CONSTRAINT fk_department_group_membership_org_id_department_id
        FOREIGN KEY (org_id, department_id) REFERENCES departments (org_id, id)
        ON DELETE CASCADE,
    CONSTRAINT fk_department_group_membership_org_id_group_id
        FOREIGN KEY (org_id, group_id) REFERENCES groups (org_id, id)
        ON DELETE CASCADE
);


CREATE TABLE ydir.group_group_membership (
    org_id BIGINT NOT NULL,
    child_group_id BIGINT,
    parent_group_id BIGINT,

    CONSTRAINT pk_group_group_membership PRIMARY KEY (org_id, child_group_id, parent_group_id),
    CONSTRAINT fk_group_group_membership_org_id_child_group_id
        FOREIGN KEY (org_id, child_group_id) REFERENCES groups (org_id, id)
        ON DELETE CASCADE,
    CONSTRAINT fk_group_group_membership_org_id_parent_group_id
        FOREIGN KEY (org_id, parent_group_id) REFERENCES groups (org_id, id)
        ON DELETE CASCADE
);


CREATE TABLE ydir.resources (
    id TEXT NOT NULL,
    org_id BIGINT NOT NULL,
    service TEXT NOT NULL,  -- like "portal", "yamb" or OAuth app id

    CONSTRAINT pk_resources PRIMARY KEY (id),
    CONSTRAINT fk_resources_org_id
        FOREIGN KEY (org_id) REFERENCES organizations (id)
);

CREATE UNIQUE INDEX uk_resources_org_id_id ON ydir.resources (org_id, id);


CREATE TABLE ydir.resource_relations (
    org_id BIGINT NOT NULL,
    id BIGSERIAL NOT NULL,
    resource_id TEXT NOT NULL,
    user_id BIGINT,
    department_id INT,
    group_id BIGINT,
    name TEXT NOT NULL,

    CONSTRAINT pk_resource_relations PRIMARY KEY (org_id, id),
    CONSTRAINT fk_resource_relations_resource_id
        FOREIGN KEY (resource_id) REFERENCES resources (id)
        ON DELETE CASCADE,
    CONSTRAINT fk_resource_relations_org_id_user_id
        FOREIGN KEY (org_id, user_id) REFERENCES users (org_id, id)
        ON DELETE CASCADE,
    CONSTRAINT fk_resource_relations_org_id_department_id
        FOREIGN KEY (org_id, department_id) REFERENCES departments (org_id, id)
        ON DELETE CASCADE,
    CONSTRAINT fk_resource_relations_org_id_group_id
        FOREIGN KEY (org_id, group_id) REFERENCES groups (org_id, id)
        ON DELETE CASCADE,

    CONSTRAINT check_object_provided CHECK (
        user_id IS NOT NULL OR
        department_id IS NOT NULL OR
        group_id IS NOT NULL
    )
);

CREATE UNIQUE INDEX uk_resource_relations_resource_id_user_id_name
    ON resource_relations (resource_id, user_id, name);
CREATE UNIQUE INDEX uk_resource_relations_resource_id_department_id_name
    ON resource_relations (resource_id, department_id, name);
CREATE UNIQUE INDEX uk_resource_relations_resource_id_group_id_name
    ON resource_relations (resource_id, group_id, name);


CREATE INDEX i_users_firstru_idx ON users USING gin(
  to_tsvector('russian', name #>> '{{first,ru}}'));
CREATE INDEX i_users_lastru_idx ON users USING gin(
  to_tsvector('russian', name #>> '{{last,ru}}'));
CREATE INDEX i_users_nameen_idx ON users USING gin(
  to_tsvector('english', name #>> '{{first,en}}'));
CREATE INDEX i_users_nameru_idx ON users USING gin(
  to_tsvector('english', name #>> '{{last,en}}'));
CREATE INDEX departnentnameru_idx ON departments USING
  gin(to_tsvector('russian', name ->> 'ru'));
CREATE INDEX departnentnameen_idx ON departments USING
  gin(to_tsvector('english', name ->> 'en'));
CREATE INDEX groupnameru_idx ON groups USING
  gin(to_tsvector('russian', name ->> 'ru'));
CREATE INDEX groupnameen_idx ON groups USING
  gin(to_tsvector('english', name ->> 'en'));


CREATE TABLE ydir.actions (
    id SERIAL NOT NULL,
    org_id BIGINT NOT NULL,
    revision INT NOT NULL,
    name TEXT NOT NULL,  -- action name, for example user_add
    author_id BIGINT NOT NULL,
    object JSONB,
    object_type TEXT NOT NULL, -- user, group or department
    old_object JSONB,
    timestamp TIMESTAMP WITH TIME ZONE DEFAULT now(),

    CONSTRAINT pk_actions PRIMARY KEY (id)
);

CREATE INDEX i_actions_org_id_object_type ON ydir.actions (org_id, object_type);
CREATE UNIQUE INDEX uk_actions_org_id_revision ON ydir.actions (org_id, revision);


CREATE TABLE ydir.events (
    id SERIAL NOT NULL,
    org_id BIGINT NOT NULL,
    revision INT NOT NULL,
    name TEXT NOT NULL,  -- event name, for example user_moved
    content JSONB, -- json object with structure like {'after': {}, 'before': {}, 'diff': {}}
    object_type TEXT NOT NULL, -- user, group, department or resource
    object JSONB NOT NULL, -- object, for which this event was generated
    timestamp TIMESTAMP WITH TIME ZONE DEFAULT now(),

    CONSTRAINT pk_events PRIMARY KEY (id)
);

CREATE INDEX i_events_revision ON ydir.events (revision);
CREATE INDEX i_events_org_id_object_type ON ydir.events (org_id, object_type);
