# coding: utf-8

from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationModel


class Dependency(object):
    namespace = None

    def __init__(self, name):
        self.name = name

    def __hash__(self):
        return hash((self.namespace, self.name))

    def __eq__(self, other):
        return (self.namespace, self.name) == (other.namespace, other.name)

    def __neq__(self, other):
        return not self.__eq__(other)

    def enable(self, meta_connection, main_connection, org_id, author_id=None):
        raise NotImplemented()


class Service(Dependency):
    namespace = 'service'

    def enable(self, meta_connection, main_connection, org_id, author_id=None):
        from intranet.yandex_directory.src.yandex_directory.core.models.service import enable_service

        return enable_service(
            meta_connection,
            main_connection,
            org_id,
            self.name,
            author_id=author_id,
        )

    def __repr__(self):
        return '<Service ({})>'.format(self.name)


class Setting(Dependency):
    namespace = 'setting'

    def __init__(self, name, value):
        # имя настройки должно соответствовать столбцу в таблице organizations
        # column_name или column-name
        super(Setting, self).__init__(name)
        self.value = value

    def enable(self, meta_connection, main_connection, org_id, author_id=None):
        underscored_name = self.name.replace('-', '_').lower()
        if underscored_name in OrganizationModel.all_fields:
            OrganizationModel(main_connection).update(
                update_data={underscored_name: self.value},
                filter_data={'id': org_id}
            )
        else:
            raise RuntimeError('Unable enable service. Service "{0}" not found.'.format(self.name))

    def __repr__(self):
        return '<Setting ({}:{})>'.format(self.name, self.value)


def get_dependency_slug(dependency):
    return '{0}.{1}'.format(
        dependency.namespace,
        dependency.name,
    )


dependencies = {
    Service('wiki'): [
        Setting('shared-contacts', True),
        Service('staff'),
        Service('search'),
        Service('wiki-formatter'),
        Service('wiki-magicklinks'),
    ],
    Service('tracker'): [
        Setting('shared-contacts', True),
        Service('staff'),
        Service('search'),
    ],
    Service('yamb'): [
        Setting('shared-contacts', True),
        Service('search'),
        Service('botsman'),
    ],
    Service('calendar'): [
        Setting('shared-contacts', True)
    ],
}
