# -*- coding: utf-8 -*-
from collections import defaultdict
from copy import deepcopy

from intranet.yandex_directory.src.yandex_directory.core.utils import (
    get_common_parent,
    get_all_groups_for_department,
)
from intranet.yandex_directory.src.yandex_directory.core.models.department import DepartmentModel
from intranet.yandex_directory.src.yandex_directory.core.models.resource import ResourceModel
from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel
from intranet.yandex_directory.src.yandex_directory.core.events.utils import get_content_object
from intranet.yandex_directory.src.yandex_directory.common.models.types import (
    TYPE_DEPARTMENT,
    ROOT_DEPARTMENT_ID,
)
from intranet.yandex_directory.src.yandex_directory.core.events.resource import _save_resource_grant_changed
from intranet.yandex_directory.src.yandex_directory.core.events.group import _get_all_resources_for_group
from intranet.yandex_directory.src.yandex_directory.core.events import(
    event_department_department_added,
    event_department_department_deleted,
)
from intranet.yandex_directory.src.yandex_directory.core.events.tasks import UpdateMembersCountTask
import collections.abc
from functools import reduce


def _get_department_relations(connection, org_id, department):
    """
    Возвращает множество связей для департамента вида
    set((resource_id1, relation_name1),
        (resource_id2, relation_name3),
    )
    """
    id_name = set()
    if department['id'] == -1:
        return id_name
    filter_data = {
        'org_id': org_id,
        'department_id': department['id'],
    }
    resources = ResourceModel(connection).find(
        filter_data=filter_data,
        fields=['id', 'relations.*'],
    )

    def filter_relation(path_str):
        path = [int(x) for x in path_str.split('.')]

        def _filter(resource):
            filtered_relation = [relation for relation in resource['relations'] if relation['department_id'] in path]
            result = set(
                (resource['id'], x['name'])
                for x in filtered_relation
            )
            return result
        return _filter

    id_name_list = list(map(
        filter_relation(department['path']), resources))

    if resources:
        id_name = reduce(lambda result, x: result.union(x),
                                id_name_list)
    return id_name


def _resource_relation_change_for_departments(connection,
                                              org_id,
                                              revision,
                                              object_value,
                                              object_type,
                                              department_after,
                                              department_before):
    """
    Вычисляет изменения состава ресурса при перемещении пользователя или
    департамента в другой департамент, генерит событие resource_grant_changed
    для ресурсов текущего департамента и всех департаментов, выше по дереву.
    """

    # строим два множества вида
    # [(resource_id1, relation_name1), (resource_id2, relation_name3),]
    # для связей до изменения и после
    id_name_after = _get_department_relations(connection, org_id, department_after)
    id_name_before = _get_department_relations(connection, org_id, department_before)
    to_add = id_name_after.difference(id_name_before)
    to_add_dict = defaultdict(list)
    for resource_id, relation_name in to_add:
        to_add_dict[resource_id].append(relation_name)

    to_delete = id_name_before.difference(id_name_after)
    to_delete_dict = defaultdict(list)
    for resource_id, relation_name in to_delete:
        to_delete_dict[resource_id].append(relation_name)

    _save_resource_grant_changed(
        connection,
        org_id=org_id,
        revision=revision,
        object_value=object_value,
        object_type=object_type,
        id_name_to_add=to_add_dict,
        id_name_to_delete=to_delete_dict,
    )
    _save_resource_grant_changed_over_group(
        connection,
        org_id,
        revision,
        object_value,
        object_type,
        department_after,
        department_before
    )


def _get_resources_for_department_over_group(connection, org_id, department_id):
    """
    Отдаёт все ресурсы с которыми департамент связан опосредованно,
    через группу. Иерархия групп учитывается.
    """
    groups = get_all_groups_for_department(connection, org_id, department_id)
    resources_dict = {}
    for group in groups:
        resources_groups = _get_all_resources_for_group(
            connection,
            org_id,
            group,
        )
        for resource in resources_groups:
            resources_dict[resource['id']] = resource
    return resources_dict


def _save_resource_grant_changed_over_group(
        connection,
        org_id,
        revision,
        object_value,
        object_type,
        department_after,
        department_before):
    """
    Сохраняет событие resource_grant_changed для всех ресурсов
    с которыми департамент связан опосредованно, через группу
    """
    after_id = department_after['id']
    before_id = department_before['id']
    resources_before = _get_resources_for_department_over_group(
        connection,
        org_id,
        before_id
    )
    resources_after = _get_resources_for_department_over_group(
        connection,
        org_id,
        after_id
    )
    to_delete_dict = {id: [] for id in list(resources_before.keys())}
    to_add_dict = {id: [] for id in list(resources_after.keys())}
    _save_resource_grant_changed(
        connection,
        org_id=org_id,
        revision=revision,
        object_value=object_value,
        object_type=object_type,
        id_name_to_add=to_add_dict,
        id_name_to_delete=to_delete_dict,
    )


def _save_event_for_parents_departments(connection,
                                        department,
                                        revision,
                                        event_func,
                                        content,
                                        limit=None,
                                        save_for_current=True,
                                        is_directly=None):
    """
    Генерит заданное событие для всех предков до предка с id=limit
    Args:
        department (object): объект департамента
        revision (int): номер ревизии
        event_func (function): событие, которые будет создано
        content (object|function): дополнительная информация
                                   или функция принимающая отдел
                                   и возвращающая словарь.
                                   События сохраняются только если
                                   content is not None.
        limit (int): id депертамента на котором остановиться,
        обновить по всему дереву, включая корневой департамент
        save_for_current (bool): сгенерировать событие для текущего департамента
        is_directly (function): опциональная функция, которая должна
                                принимать отдел и возвращать True, если
                                действие приведшее к событию, было произведено
                                непосредственно над этим отделом. Или False,
                                если это не так.
                                    По умолчанию, функция будет считать, что действие
                                произведено непосредственно над отделом, который
                                в неё передан.
    """
    count = 0
    current_department = department
    path_length = len(department['path'])
    org_id = department['org_id']

    if isinstance(content, collections.abc.Callable):
        get_content = content
    else:
        base_content = content
        def get_content(obj):
            return deepcopy(base_content)

    if not isinstance(is_directly, collections.abc.Callable):
        def is_directly(obj):
            return obj['id'] == department['id']

    if save_for_current:
        content = get_content(department)
        if content is not None:
            content['directly'] = is_directly(department)
            event_func(
                connection,
                org_id,
                revision,
                current_department,
                TYPE_DEPARTMENT,
                content
            )

    department_model = DepartmentModel(connection)

    while ('parent_id' in current_department and
           current_department['id'] != limit and
           current_department['id'] != ROOT_DEPARTMENT_ID and
           current_department['parent_id'] and
           count < path_length):

        count += 1
        parent_object = department_model.get(
            current_department['parent_id'],
            org_id,
        )
        current_department = parent_object
        content = get_content(current_department)

        if content is not None:
            content['directly'] = is_directly(current_department)

            event_func(
                connection,
                org_id,
                revision,
                current_department,
                TYPE_DEPARTMENT,
                content
            )


def _update_group_cache(connection, org_id, department_id, directly=True):
    """
    Находит все группы департамента и обновляет для них разрернутый кеш
    пользователей
    """
    if not directly:  # обновляеем кеш только при непосредственном изменении
        return

    all_groups = get_all_groups_for_department(
        connection,
        org_id,
        department_id,
    )
    group_model = GroupModel(connection)

    for group in all_groups:
        group_model.update_leafs_cache_and_member_count(
            org_id=org_id,
            group_id=group['id'],
        )


def on_department_added(connection, org_id, revision, department,  **kwargs):
    """
    Обработчик события департамент добавлен
    """
    if department['id'] == ROOT_DEPARTMENT_ID or not department['parent_id']:
        return
    content = {
        'after': department,
    }
    parent_object = DepartmentModel(connection).get(
        department['parent_id'],
        department['org_id'],
    )

    _save_event_for_parents_departments(
        connection,
        parent_object,
        revision,
        event_department_department_added,
        content,
        ROOT_DEPARTMENT_ID
    )


def on_department_moved(connection,
                        org_id,
                        revision,
                        department,
                        object_type,
                        content,
                        **kwargs):
    """
    Обработчик события перемещения департамента(привязка в другому родителю)
    Args:
        org_id (int): id организации
        revision (int): номер ревизии
        department (object): объект после изменений
        content (object): по ключу ['diff']['parent_id'] изменения
        родительского депертамента (from_department_id, to_department_id)
    """
    if 'diff' not in content and 'parent_id' not in content['diff']:
        raise RuntimeError('Invalid data in department_moved listener')

    from_department_id, to_department_id = content['diff']['parent_id']

    department_model = DepartmentModel(connection)
    from_department = department_model.get(from_department_id, org_id)
    to_department = department_model.get(to_department_id, org_id)

    new_content = get_content_object(subject=department, directly=True)

    (
        common_parent_id,
        before_common_from_department,
        before_common_to_department
    ) = get_common_parent(
         from_department.get('path'),
         to_department.get('path')
    )

    _save_event_for_parents_departments(
        connection,
        to_department,
        revision,
        event_department_department_added,
        new_content,
        before_common_to_department,
    )
    _save_event_for_parents_departments(
        connection,
        from_department,
        revision,
        event_department_department_deleted,
        new_content,
        before_common_from_department
    )

    # важно пересчитывать сразу для обоих отделов, чтобы
    # изменение счетчиков произошло за одну операцию и было консистентным
    UpdateMembersCountTask(connection).delay(
        department_ids=[from_department_id, to_department_id],
        org_id=org_id,
        revision=revision,
    )

    _resource_relation_change_for_departments(
        connection,
        org_id,
        revision,
        department,
        TYPE_DEPARTMENT,
        to_department,
        from_department,
    )


def on_department_user_added(connection, org_id, revision, department, **kwargs):
    """
    В департамент добавился пользователь
    """
    _update_group_cache(
        connection,
        org_id,
        department['id'],
        kwargs['content']['directly'],
    )


def on_department_user_deleted(connection,
                               org_id,
                               revision,
                               department,
                               **kwargs):
    """
    Из департамента ушёл пользователь
    """
    _update_group_cache(
        connection,
        org_id,
        department['id'],
        kwargs['content']['directly'],
    )


def on_department_department_added(connection,
                                   org_id,
                                   revision,
                                   department,
                                   **kwargs):
    """
    Новый департамент в составе департамента
    """
    _update_group_cache(
        connection,
        org_id,
        department['id'],
        kwargs['content']['directly'],
    )


def on_department_department_deleted(connection,
                                     org_id,
                                     revision,
                                     department,
                                     **kwargs):
    """
    Какой-то департамент больше не входит в состав другого департамента
    """
    _update_group_cache(
        connection,
        org_id,
        department['id'],
        kwargs['content']['directly']
    )


def on_department_deleted(connection,
                          org_id,
                          revision,
                          department,
                          object_type,
                          content,
                          **kwargs):
    """
    Обработчик события "департамент удален"
    """
    content = {}
    _save_event_for_parents_departments(
        connection,
        department=department,
        revision=revision,
        event_func=event_department_department_deleted,
        content=content,
        limit=department['parent_id'],
        save_for_current=False,
    )
