# -*- coding: utf-8 -*-
from copy import deepcopy
from collections import defaultdict
from datetime import timedelta

from intranet.yandex_directory.src.yandex_directory.common.utils import utcnow
from intranet.yandex_directory.src.yandex_directory.core.models.resource import ResourceModel
from intranet.yandex_directory.src.yandex_directory.core.models.department import DepartmentModel
from intranet.yandex_directory.src.yandex_directory.core.models.group import (
    GroupModel,
    UserGroupMembership,
)
from intranet.yandex_directory.src.yandex_directory.core.models.user import UserModel
from intranet.yandex_directory.src.yandex_directory.core.models.service import (
    OrganizationServiceModel,
    update_license_cache_task,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import Ignore

from intranet.yandex_directory.src.yandex_directory.core.events.utils import (
    get_content_object,
    RESOURCE_CONTENT_STRUCTURE,
)
from intranet.yandex_directory.src.yandex_directory.common.models.types import (
    TYPE_GROUP,
    TYPE_USER,
    TYPE_DEPARTMENT,
    TYPE_RESOURCE,
    TYPE_SERVICE,
)
from intranet.yandex_directory.src.yandex_directory.core.events import (
    event_user_group_added,
    event_group_group_added,
    event_department_group_added,
    event_user_group_deleted,
    event_group_group_deleted,
    event_department_group_deleted,
    event_resource_grant_changed,
    event_group_membership_changed,
    event_service_license_changed,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import only_ids

TYPE_MODELS_DICT = {
    TYPE_USER: UserModel,
    TYPE_GROUP: GroupModel,
    TYPE_DEPARTMENT: DepartmentModel,
}

EVENT_FUNCS_BY_TYPE = {
    'department': event_department_group_added,
    'user': event_user_group_added,
    'group': event_group_group_added,
}


def _get_add_remove_members(members_before, members_after):
    """
    Args:
        members_before(list) - список мемберов до изменения
        members_after(list) - список мемберов после изменения
    Формируем список удаленных и добавленных сложных объектов вида
    {'type': 'user', 'id': 'user_id'},
    {'type': 'group', 'id': 'group_id'}.
    Формат:
    {
        members: {
            'add': {
                'users': [id1, id2, ...],
                'departments': [id1, id2, ...],
                'groups': [id1, id2, ...],
            },
            'remove': {
                'users': [id3, id4, ...],
                'departments': [id3, id4, ...],
                'groups': [id3, id4, ...],
            },
        },
    }
    """
    old_obj_dict = defaultdict(list)
    cur_obj_dict = defaultdict(list)

    # пытаемся сформировать словари вида:
    # old|cur_objects = {
    #     'users': [id1, id2, ...],
    #     'groups': [id2, id3, ...],
    #     'department': [id4, id5, ...]
    # }
    for obj in members_before:
        object_type = obj['type']+'s'
        value_id = obj['object']['id']
        old_obj_dict[object_type].append(value_id)
    for obj in members_after:
        object_type = obj['type']+'s'
        value_id = obj['object']['id']
        cur_obj_dict[object_type].append(value_id)

    # пытаемся создать объект вида:
    # diff_members = {
    #     'add': {
    #         'users': [id1, id2, id3, ...],
    #         'groups': [id1, id2, ...],
    #         'departments': [id3, id4, ...],
    #     },
    #     'remove': {
    #         'users': [id1, id2, id3, ...],
    #         'groups': [id1, id2, ...],
    #         'departments': [id3, id4, ...],
    #     }
    # }
    diff_members = {
        'add': defaultdict(list),
        'remove': defaultdict(list),
    }
    object_types = ['users', 'departments', 'groups']
    for object_type in object_types:
        old_obj_set = set(old_obj_dict[object_type])
        new_obj_set = set(cur_obj_dict[object_type])
        intersection = old_obj_set & new_obj_set
        to_add = new_obj_set - intersection
        to_remove = old_obj_set - intersection
        diff_members['add'][object_type] = list(to_add)
        diff_members['remove'][object_type] = list(to_remove)

    return diff_members


def _get_all_resources_for_parents(connection, org_id, group_id):
    """
    Отдаёт все ресурсы, связанные с предками группы
    """
    up_resources = {}
    parents = GroupModel(connection).get_parents(
        org_id=org_id,
        group_id=group_id
    )
    for parent in parents:
        resources = ResourceModel(connection).find(
            filter_data={
                'org_id': org_id,
                'group_id': parent['id'],
            },
            fields=[
                '*',
                'relations.*',
            ],
        )
        resources_dict = {x['id']: x for x in resources}
        up_resources.update(resources_dict)
        parent_resources = _get_all_resources_for_parents(
            connection,
            org_id,
            parent['id'],
        )
        up_resources.update(parent_resources)
    return up_resources


def _get_all_resources_for_group(connection, org_id, group):
    """
    Собирает и отдаёт все ресурсы для группы с учётом иерархической структуры
    Технический ресурс группы не отдается.
    """
    resource_group = ResourceModel(connection).find(
        filter_data={
            'org_id': org_id,
            'group_id': group['id'],
        },
        fields=[
            '*',
            'relations.*',
        ],
    )

    up_resources_dict = _get_all_resources_for_parents(
        connection,
        org_id,
        group['id']
    )
    resource_group.extend(list(up_resources_dict.values()))
    return resource_group


def _get_all_resources_for_group_with_tech(connection, org_id, group):
    """
    Собирает и отдаёт все ресурсы для группы с учётом иерархической структуры
    И технический ресурс группы.
    """
    resources = _get_all_resources_for_group(connection, org_id, group)
    technical_resource = ResourceModel(connection).find(
        filter_data={
            'org_id': org_id,
            'id': group['resource_id'],
        },
        fields=[
            '*',
            'relations.*',
        ],
    )
    resources.extend(technical_resource)
    return resources


def on_group_added(connection, org_id, revision, group, **kwargs):
    """
    Обработчик события: группа добавлена.
    """
    # Генерируем для всех members-группы события,
    # что они были добавлены в группу
    for member in group['members']:
        content = get_content_object(subject=group, directly=False)
        model = TYPE_MODELS_DICT[member['type']]
        member_type_id = member.get('id') or member.get('object', {}).get('id')
        member_key = member.get('type')+'_id'
        member_instance = model(connection).get(**{
            'org_id': org_id,
            member_key: member_type_id,
            'fields': ['*', 'email'],
        })
        # теперь для каждого члена группы нужно сгенерить событие
        # про то, что он был добавлен в группу
        event_func = EVENT_FUNCS_BY_TYPE[member['type']]
        event_func(
            connection,
            org_id=org_id,
            revision=revision,
            object_value=member_instance,
            object_type=member['type'],
            content=content,
        )


def on_group_membership_changed(connection, org_id, revision, group, content, **kwargs):
    """
    Обработчик события: изменились members в группе
    1. Генерировать события '<object_type>_group_added' или
    '<object_type>_group_deleted' для всех members.
    2. Удалять или добавлять resource_relation для members-групп
    из текущей группы.
    3. Смотреть в родительские группы и удалять или добавлять
    resource_relation для members-групп соотвествующие resource_relation-ы.
    """

    # content должен прилететь в виде:
    # content = {
    #     'subject': <group>,
    #     'diff': {
    #         'members': {
    #             'add': {
    #                 'users': [id1, id2, id3, ...],
    #                 'groups': [id1, id2, ...],
    #                 'departments': [id3, id4, ...],
    #             },
    #             'remove': {
    #                 'users': [id1, id2, id3, ...],
    #                 'groups': [id1, id2, ...],
    #                 'departments': [id3, id4, ...],
    #             }
    #         }
    #     }
    # }

    def generic_save_and_gen_event(
            object_type_model,
            object_type_id,
            object_type,
            event_func,
            additional_kwargs):

        filter_data = {
            'org_id': org_id,
            'id': object_type_id
        }
        filter_data.update(additional_kwargs)

        instance = object_type_model(connection).find(
            filter_data,
            one=True,
        )
        # В subject должна попадать сама группа, так как тут
        # мы генерим события типа:
        # пользователя добавили в группу
        # или
        # отдел добавили в группу
        #
        # поэтому объектом является пользователь а субъектом -
        # та группа, в которую его добавили.
        content = {
            'subject': group
        }
        event_func(
            connection,
            org_id=org_id,
            revision=revision,
            object_value=instance,
            object_type=object_type,
            content=content,
        )

    # 1. Генерируем события для members
    if 'members' not in content.get('diff', {}):
        raise RuntimeError('Invalid data in group_membership_changed listener')

    local_dict_events = {
        'add': {
            TYPE_USER: event_user_group_added,
            TYPE_GROUP: event_group_group_added,
            TYPE_DEPARTMENT: event_department_group_added,
        },
        'remove': {
            TYPE_USER: event_user_group_deleted,
            TYPE_GROUP: event_group_group_deleted,
            TYPE_DEPARTMENT: event_department_group_deleted,
        }
    }
    diff_members = content['diff']['members']
    ADDITIONAL_KWARGS = {
        # Для пользователей, в find модели надо передать is_dismissed=True
        # чтобы уволенные сотрудники тоже находились
        'user': {'is_dismissed': Ignore}
    }
    for operation in ['add', 'remove']:
        for object_type in [TYPE_USER, TYPE_GROUP, TYPE_DEPARTMENT]:
            operation_data = diff_members.get(operation, {})
            object_type_ids = operation_data.get(object_type + 's', [])
            for object_type_id in object_type_ids:
                generic_save_and_gen_event(
                    object_type_model=TYPE_MODELS_DICT[object_type],
                    object_type_id=object_type_id,
                    object_type=object_type,
                    event_func=local_dict_events[operation][object_type],
                    additional_kwargs=ADDITIONAL_KWARGS.get(object_type, {}),
                )

    # 2. Удаляем или добавляем события про изменение ресурса
    # Ищем ресурсы у родителя и всех родителей группы и генерируем события,
    # что эти ресурсы были изменены
    # Контент передаем балковый и одинаковый для каждого такого ресурса.
    all_linked_resources = _get_all_resources_for_group_with_tech(
        connection,
        org_id,
        group
    )

    # в bulk_content - раскрываем всех юзеров из групп и департаментов,
    # храним id-шники без relation_name
    bulk_content = deepcopy(RESOURCE_CONTENT_STRUCTURE)

    group_model = GroupModel(connection)
    user_model = UserModel(connection)

    for operation in ['add', 'remove']:
        opp_diff = diff_members.get(operation, {})
        for user_id in opp_diff.get('users', []):
            bulk_content['relations'][operation]['users'].append(user_id)
        for group_id in opp_diff.get('groups', []):
            inner_users = group_model.get_all_users(org_id, group_id)
            bulk_content['relations'][operation]['groups'].append(group_id)
            for user in inner_users:
                bulk_content['relations'][operation]['users'].append(user['id'])
        for dep_id in opp_diff.get('departments', []):
            inner_users = user_model.find(
                filter_data={'department_id': dep_id}
            )
            bulk_content['relations'][operation]['departments'].append(dep_id)
            for user in inner_users:
                bulk_content['relations'][operation]['users'].append(user['id'])

    # берем айдишники пользователей из кэша
    leafs = UserGroupMembership(connection).find(
        filter_data={
            'org_id': org_id,
            'group_id': group['id'],
        }
    )
    user_uids = [leaf['user_id'] for leaf in leafs]
    bulk_content['uids'] = user_uids

    org_licensed_services = OrganizationServiceModel(connection).get_org_services_with_licenses(org_id)
    all_linked_resources_id = set(only_ids(all_linked_resources))
    service_resource_ids = set(org_licensed_services.keys()).intersection(all_linked_resources_id)
    for resource in all_linked_resources:
        if resource['id'] in service_resource_ids:
            update_license_cache_task(connection, org_id)
            event_service_license_changed(
                connection,
                org_id=org_id,
                revision=revision,
                object_value=org_licensed_services[resource['id']],
                object_type=TYPE_SERVICE,
                content={},
            )
        else:
            event_resource_grant_changed(
                connection,
                org_id=org_id,
                revision=revision,
                object_value=resource,
                object_type=TYPE_RESOURCE,
                content=bulk_content,
                notify_at=utcnow()+timedelta(seconds=60),
            )


def on_group_deleted(connection, org_id, revision, group, **kwargs):
    """
    Обработчик события удаления группы.
    """
    # Удаляем всех участников из группы
    diff = {
        'members': _get_add_remove_members(group['members'], [])
    }
    content = get_content_object(diff=diff)
    event_group_membership_changed(
        connection,
        org_id=org_id,
        revision=revision,
        object_value=group,
        object_type=TYPE_GROUP,
        content=content,
    )
