# -*- coding: utf-8 -*-
from copy import deepcopy
from datetime import timedelta

from intranet.yandex_directory.src.yandex_directory.common.utils import utcnow
from intranet.yandex_directory.src.yandex_directory.core.models.user import UserModel
from intranet.yandex_directory.src.yandex_directory.core.models.resource import ResourceModel
from intranet.yandex_directory.src.yandex_directory.core.models.service import (
    OrganizationServiceModel,
    update_license_cache_task,
)
from intranet.yandex_directory.src.yandex_directory.core.events.utils import RESOURCE_CONTENT_STRUCTURE
from intranet.yandex_directory.src.yandex_directory.common.models.types import (
    TYPE_RESOURCE,
    TYPE_SERVICE,
)
from intranet.yandex_directory.src.yandex_directory.core.events import (
    event_resource_grant_changed,
    event_service_license_changed,
)


def _save_resource_grant_changed(connection,
                                 org_id,
                                 revision,
                                 object_value,
                                 object_type,
                                 id_name_to_add=None,
                                 id_name_to_delete=None):
    """
    Генерит события resource_grant_changed с заданным типом связи для
    каждого из ресурсов, перечисленных в id_name_to_add/id_name_to_delete
    Args:
        revision (int): номер ревизии
        object_value (object): объект, который вызвал изменения
        id_name_set_to_add (dict): связи на добавление -
        словарь вида resource_id: ['relation_name1', ...]
        id_name_set_to_delete (dict): связи на удаление - такой же формат
        id_name_set_to_delete = {
            'resource_id1': ['relation_name1', 'relation_name2', ...],
            'resource_id2': ['relation_name1', 'relation_name2', ...],
        }
    """

    if not id_name_to_add:
        id_name_to_add = {}
    if not id_name_to_delete:
        id_name_to_delete = {}

    union_set = set(id_name_to_add).union(id_name_to_delete)
    resources = {x: {'remove': [], 'add': []} for x in union_set}
    for key, value in id_name_to_add.items():
        resources[key]['add'] = value
    for key, value in id_name_to_delete.items():
        resources[key]['remove'] = value

    resource_model = ResourceModel(connection)
    user_model = UserModel(connection)
    org_licensed_services = OrganizationServiceModel(connection).get_org_services_with_licenses(org_id)

    for resource_id, operations in resources.items():
        if resource_id in list(org_licensed_services.keys()):
            update_license_cache_task(connection, org_id)
            event_service_license_changed(
                connection,
                org_id=org_id,
                revision=revision,
                object_value=org_licensed_services[resource_id],
                object_type=TYPE_SERVICE,
                content={},
            )
        else:
            content = deepcopy(RESOURCE_CONTENT_STRUCTURE)
            for operation, relation_names in operations.items():
                for name in relation_names:
                    add_relation = {
                        'object_id': object_value['id'],
                        'relation_name': name
                    }
                    content['relations'][operation][object_type + 's'].append(add_relation)

            resource = resource_model.get(
                id=resource_id,
                org_id=org_id,
            )
            # создаем поле uids в content по запросу yamb-a (alpinist@)
            content['uids'] = list(user_model.get_uids_by_resource(org_id, resource['service'], resource_id))

            event_resource_grant_changed(
                connection,
                org_id=org_id,
                revision=revision,
                object_value=resource,
                object_type=TYPE_RESOURCE,
                content=content,
                notify_at=utcnow()+timedelta(seconds=60)
            )


def event_resource_changed_for_each_resource(connection,
                                             org_id,
                                             revision,
                                             object_value,
                                             object_type,
                                             path=None):
    """
    Создает событие resource_changed для всех ресурсов объекта
    Args:
        path (string) - путь до корня, id департаментов, разделенные точкой
    """
    path = [int(x) for x in path.split('.')] if path else []
    filter_field = object_type + '_id'
    filter_data = {
        'org_id': org_id,
        filter_field: object_value['id'],
    }
    resources = ResourceModel(connection).find(
        filter_data=filter_data,
        fields=[
            'id',
            'relations.name',
            'relations.department_id',
        ]
    )
    # Фильтр отдает только нужные ресурсы, при этом в поле relations ВСЕ
    # связи этого ресурса, не отфильтрованные
    # Мы хотим знать тип(name) связи, выбираем только нужные.
    def filter_relation(resource):
        _filtered_relations = [r for r in resource['relations'] if r['department_id'] in path]
        return resource['id'], [f['name'] for f in _filtered_relations]

    id_names = dict(list(map(filter_relation, resources)))

    _save_resource_grant_changed(
        connection,
        org_id=org_id,
        revision=revision,
        object_value=object_value,
        object_type=object_type,
        id_name_to_add=id_names,
    )
