# -*- coding: utf-8 -*-
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.db import get_meta_connection
from intranet.yandex_directory.src.yandex_directory.common.utils import utcnow
from intranet.yandex_directory.src.yandex_directory.core.models import CallbackEventsModel
from intranet.yandex_directory.src.yandex_directory.core.task_queue import Task
from intranet.yandex_directory.src.yandex_directory.core.models import TaskModel
from intranet.yandex_directory.src.yandex_directory.core.task_queue.base import TERMINATE_STATES
from intranet.yandex_directory.src.settings import EVENT_NOTIFICATION_TASK_CHUNKS_COUNT, EVENT_NOTIFICATION_TASK_EVENT_IN_CHUNK_COUNT
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import default_log


# Вместо update_members_count
class UpdateMembersCountTask(Task):
    def do(self, department_ids, org_id, revision):
        from intranet.yandex_directory.src.yandex_directory.core.events.department import _save_event_for_parents_departments
        from intranet.yandex_directory.src.yandex_directory.core.events import event_department_property_changed
        from intranet.yandex_directory.src.yandex_directory.core.models.department import (
            DepartmentModel,
            is_ancestor,
        )

        department_model = DepartmentModel(self.main_connection)

        before = department_model.find(
            {'org_id': org_id, 'id': department_ids},
            fields=[
                'id',
                'path',
                'parents.id',
                'parents.members_count',
                'members_count',
            ]
        )
        # тут надо сохранить предыдущие значения счетчиков
        # department_id -> counter
        # эти значения будут использоваться для построения диффов
        # событий после обновления счетчиков
        previous_values = {}

        for dep in before:
            previous_values[dep['id']] = dep['members_count']
            previous_values.update(
                (parent['id'], parent['members_count'])
                for parent in dep['parents']
            )

        department_model.update_members_count(department_ids, org_id)

        def get_content(obj):
            # возвращает полный payload для события
            prev_count = previous_values.get(obj['id'])
            new_count = obj['members_count']

            if prev_count != new_count:
                return {
                    'diff': {
                        'members_count': [
                            prev_count,
                            new_count
                        ]
                    }
                }
            # если ничего не изменилось, то вернём None
            # и событие не сгенерится
            return None

        def is_directly(obj):
            return obj['id'] in department_ids

        # теперь из изначальных отделов надо убрать те, которые
        # являются предками других
        def is_not_ancestor_of_some_other(dep):
            if any(is_ancestor(dep, other_dep)
                   for other_dep in before):
                return False
            return True

        deps = list(filter(is_not_ancestor_of_some_other, before))

        for dep in deps:
            _save_event_for_parents_departments(
                self.main_connection,
                # В функцию надо передать новый объект в том виде,
                # в каком он теперь сохранился в базу.
                # Для этого достанем его из БД.
                department_model.get(dep['id'], org_id),
                revision,
                event_department_property_changed,
                content=get_content,
                is_directly=is_directly,
            )


class EventNotificationTask(Task):
    """
    Отдельно оповещаем в порядке появления событий каждого подписчика в организации.
    Если встретилось событие с временем оповещения в будущем, то отложим выполнение задачи до этого времени.
    """
    singleton = True
    default_priority = 1000  # События стараемся разослать в первую очередь, чтобы задержка была минимальной.
    lock_ttl = 10 * 60  # 10 минут

    def get_callbacks_for_process(self, org_id, callback, event_id_from):
        callback_events_model = CallbackEventsModel(self.main_connection)
        filter_fields = {
            'callback': callback,
            'org_id': org_id,
            'environment': app.config['ENVIRONMENT'],
            'done': False,
        }
        if event_id_from:
            filter_fields['event_id__gte'] = event_id_from

        return callback_events_model \
            .filter(**filter_fields) \
            .fields('id', 'event_id') \
            .order_by('event_id') \
            .limit(EVENT_NOTIFICATION_TASK_CHUNKS_COUNT * EVENT_NOTIFICATION_TASK_EVENT_IN_CHUNK_COUNT) \
            .all()

    def do(self, org_id, callback):
        metadata = self.get_metadata() or {}

        task_ids = metadata.get('task_ids', [])
        if task_ids:
            tasks = TaskModel(self.main_connection).filter(id__in=task_ids).fields('id', 'state').all()
        else:
            tasks = []

        active_tasks = [x for x in tasks if x['state'] not in TERMINATE_STATES]
        default_log.info('Active tasks %s' % active_tasks)

        event_id_from = metadata.get('event_id_from', None)
        default_log.info('Event id from %s' % event_id_from)
        callbacks = self.get_callbacks_for_process(org_id, callback, event_id_from)
        default_log.info('Count of callbacks: %s' % len(callbacks))

        if len(callbacks) == 0 and len(active_tasks) == 0:
            if event_id_from is None or not app.config['EVENT_NOTIFICATION_ENQUEUE_ON_END']:
                default_log.info('All callbacks sent, finish task')
                return
            else:
                default_log.info('Try find new callbacks')
                # пробуем еще раз сначала, т.к. что-то могло не отправится либо приехать новое
                self.update_metadata(task_ids=task_ids, event_id_from=None)
                self.defer(countdown=60)

        if len(callbacks) == 0:
            self.defer(countdown=60)

        for i in range(EVENT_NOTIFICATION_TASK_CHUNKS_COUNT - len(active_tasks)):
            index_from = i * EVENT_NOTIFICATION_TASK_EVENT_IN_CHUNK_COUNT
            index_to = min(index_from + EVENT_NOTIFICATION_TASK_EVENT_IN_CHUNK_COUNT, len(callbacks)) - 1
            if index_from > index_to:
                break

            task_id = EventNotificationChunkTask(self.main_connection).delay(
                org_id=org_id,
                callback=callback,
                event_id_from=callbacks[index_from]['event_id'],
                event_id_to=callbacks[index_to]['event_id'],
            ).task_id
            task_ids.append(task_id)

        self.update_metadata(task_ids=task_ids, event_id_from=callbacks[-1]['event_id'] + 1)
        self.defer(countdown=60)


class EventNotificationChunkTask(Task):
    """
    Отдельно оповещаем в порядке появления событий каждого подписчика в организации.
    Если встретилось событие с временем оповещения в будущем, то отложим выполнение задачи до этого времени.
    """
    singleton = False
    default_priority = 1001  # События стараемся разослать в первую очередь, чтобы задержка была минимальной. Приоритет выше, чем у EventNotificationTask
    lock_ttl = 10 * 60  # 10 минут

    def do(self, org_id, event_id_from, event_id_to, callback):
        """Основной метод, который отправляет все события организации для
           указанного потребителя.
        """
        from intranet.yandex_directory.src.yandex_directory.core.events.utils import notify_callback
        callback_events_model = CallbackEventsModel(self.main_connection)
        callbacks = callback_events_model \
            .filter(
                event_id__between=(event_id_from, event_id_to),
                callback=callback,
                org_id=org_id,
                environment=app.config['ENVIRONMENT'],
                done=False,
            ) \
            .fields('id', 'event_id', 'callback', 'settings', 'event.*') \
            .order_by('event_id') \
            .all()

        for item in callbacks:
            url_or_func_name = item['callback']
            event_id = item['event_id']

            with log.fields(event_id=event_id, callback=url_or_func_name):
                now = utcnow()
                event = item['event']
                callback_settings = item['settings']

                if event['notify_at'] > now:
                    log.debug('Defer for retry.')
                    self.defer(retry_at=event['notify_at'])
                log.debug('Notify callback')

                app.stats_aggregator.inc('notify_total_summ')

                done = True

                # В объектах тасков есть только коннект к шарду
                with get_meta_connection() as meta_connection:
                    result = notify_callback(
                        meta_connection,
                        self.main_connection,
                        url_or_func_name,
                        event,
                        callback_settings,
                    )
                status_code, response_content = result
                # Пока что сделал так, чтобы done всегда оставался
                # true, потому что если сервис постоянно отвечает ошибками,
                # то таск уходит в бесконечный цикл.
                #
                # # Только в случае двухсотого кода ответа, обработка
                # # вебхука считается удачной
                # if status_code < 200 or status_code >= 300:
                #     # TODO: Возможно в будущем, будем тут ещё и количество
                #     #       попыток считать, чтобы по ним можно было
                #     #       отключать неработающие вебхуки:
                #     #       https://st.yandex-team.ru/DIR-5096
                #     done = False

                seconds = (now - event['notify_at']).total_seconds()
                app.stats_aggregator.add_to_bucket('notify_delay', value=seconds)

                with log.fields(done=done,
                                last_response_at=now,
                                delivery_delay=seconds,
                                response_code=status_code,
                                response_content=response_content):
                    log.debug('Updating callback')

                callback_events_model \
                    .filter(id=item['id']) \
                    .update(
                        done=done,
                        last_response_at=now,
                        response_code=status_code,
                        response_content=response_content,
                    )
