# -*- coding: utf-8 -*-
from collections import defaultdict

from intranet.yandex_directory.src.yandex_directory.core.events import (
    event_department_user_added,
    event_department_user_deleted,
)
from intranet.yandex_directory.src.yandex_directory.core.events.department import (
    _resource_relation_change_for_departments,
    _save_event_for_parents_departments,
)
from intranet.yandex_directory.src.yandex_directory.core.events.resource import _save_resource_grant_changed
from intranet.yandex_directory.src.yandex_directory.core.events.utils import get_content_object
# from intranet.yandex_directory.src.yandex_directory.core.models.department import (
#     DepartmentModel,
#     is_ancestor,
# )
from intranet.yandex_directory.src.yandex_directory.core.models import (
    OrganizationServiceModel,
    DepartmentModel,
    ResourceRelationModel,
    ResourceModel,
)
from intranet.yandex_directory.src.yandex_directory.common.models.types import (
    TYPE_USER,
    ROOT_DEPARTMENT_ID,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import get_common_parent
from intranet.yandex_directory.src.yandex_directory.core.events.tasks import UpdateMembersCountTask


def _resource_relation_change_for_dismissed_user(connection,
                                                 org_id,
                                                 revision,
                                                 object_value,
                                                 object_type):
    """
    Вычисляет изменения состава ресурса при увольнении пользователя,
    генерит событие resource_grant_changed
    """

    filter_data = {
        'org_id': org_id,
        'user_id': object_value['id'],
    }
    relations = ResourceRelationModel(connection).find(
        filter_data=filter_data,
        fields=['resource_id', 'name'],
    )

    to_delete_dict = defaultdict(list)

    for relation in relations:
        resource_id = relation['resource_id']
        relation_name = relation['name']
        to_delete_dict[resource_id].append(relation_name)

    _save_resource_grant_changed(
        connection,
        org_id=org_id,
        revision=revision,
        object_value=object_value,
        object_type=object_type,
        id_name_to_delete=to_delete_dict,
    )


def on_user_moved(connection,
                  org_id,
                  revision,
                  user,
                  content,
                  **kwargs):
    """
    Обработчик события перемещения сотрудника в другой департамент
    Args:
        org_id (int): id организации
        revision (int): номер ревизии
        user (object): объект после изменений
        content (object): по ключу ['diff']['department_id'] изменения
        депертамента (from_department_id, to_department_id)
    """
    if 'diff' not in content and 'department_id' not in content['diff']:
        raise RuntimeError('Invalid data in user_moved listener')

    department_model = DepartmentModel(connection)
    from_department_id = content['diff']['department_id'][0]
    from_department = department_model.get(from_department_id, org_id, fields=['*', 'email'])
    to_department_id = content['diff']['department_id'][1]
    to_department = department_model.get(to_department_id, org_id, fields=['*', 'email'])

    new_content = get_content_object(subject=user)

    if from_department and to_department:
        (
            common_parent_id,
            before_common_from_department,
            before_common_to_department,
        ) = get_common_parent(
            from_department.get('path'),
            to_department.get('path'),
        )
        new_content.update({'directly': False})
        new_content['diff'] = {
            'members': {
                'remove': {},
                'add': {
                    'users': [user['id']],  # добавляем одного пользователя
                },
            },
        }
        _save_event_for_parents_departments(
            connection,
            to_department,
            revision,
            event_department_user_added,
            new_content,
            before_common_to_department,
        )
        new_content['diff'] = {
            'members': {
                'remove': {
                    'users': [user['id']],  # удаляем одного пользователя
                },
                'add': {},
            },
        }
        _save_event_for_parents_departments(
            connection,
            from_department,
            revision,
            event_department_user_deleted,
            new_content,
            before_common_from_department,
        )

        _resource_relation_change_for_departments(
            connection,
            org_id=org_id,
            revision=revision,
            object_value=user,
            object_type=TYPE_USER,
            department_after=to_department,
            department_before=from_department,
        )
        # теперь обновим счетчики сотрудников
        # важно пересчитывать сразу для обоих отделов, чтобы
        # изменение счетчиков произошло за одну операцию и было консистентным
        UpdateMembersCountTask(connection).delay(
            department_ids=[to_department_id, from_department_id],
            org_id=org_id,
            revision=revision,
        )


def on_user_dismissed(connection,
                      org_id,
                      revision,
                      user,
                      content,
                      **kwargs):

    diff = {
        'members': {
            'add': {},
            'remove': {
                'users': [user['id']],
            },
        },
    }
    new_content = get_content_object(subject=user, diff=diff)

    # если у юзера нет департамента, то не генерируем события о департаменте
    if not user['department_id']:
        return

    from_department_id = user['department_id']
    from_department = DepartmentModel(connection).get(from_department_id, org_id)

    _save_event_for_parents_departments(
        connection,
        from_department,
        revision,
        event_department_user_deleted,
        new_content,
        ROOT_DEPARTMENT_ID,
    )

    _resource_relation_change_for_departments(
        connection,
        org_id=org_id,
        revision=revision,
        object_value=user,
        object_type=TYPE_USER,
        department_after={
            'id': -1,
            'path': '-1',
        },  # отправляет id несуществующего департамента
        department_before=from_department,
    )

    _resource_relation_change_for_dismissed_user(
        connection,
        org_id=org_id,
        revision=revision,
        object_value=user,
        object_type=TYPE_USER,
    )

    UpdateMembersCountTask(connection).delay(
        department_ids=[from_department_id],
        org_id=org_id,
        revision=revision,
    )
