# -*- coding: utf-8 -*-
import os
import datetime
from intranet.yandex_directory.src.settings import CLOUD_ORGANIZATION_TYPE
from cachetools import TTLCache, cached
from threading import RLock
from collections import defaultdict
from functools import partial
from operator import itemgetter
from retrying import retry
from itertools import groupby

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection,
    get_meta_connection,
)
from requests.exceptions import (
    HTTPError,
    ReadTimeout,
)

from intranet.yandex_directory.src.yandex_directory.auth.utils import hide_auth_from_headers
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    import_from_string,
    Ignore,
    NotChanged,
    remove_not_changed_keys,
    json,
    hashabledict,
    utcnow,
    time_in_future,
)
from intranet.yandex_directory.src.yandex_directory.auth.scopes import (
    scope,
    check_scopes,
)
from intranet.yandex_directory.src.yandex_directory.core.events.tasks import EventNotificationTask
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask
from intranet.yandex_directory.src.yandex_directory.core.models.webhook import WebHookModel
from intranet.yandex_directory.src.yandex_directory.core.models.service import (
    OrganizationServiceModel,
    ServiceModel,
)
from intranet.yandex_directory.src.yandex_directory.common.models.base import Values
from intranet.yandex_directory.src.yandex_directory.core.models.event import (
    EventModel,
    LastProcessedRevisionModel,
    CallbackEventsModel,
    ProcessedEventsModel,
)

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log, default_log

from intranet.yandex_directory.src.yandex_directory.core.utils import (
    pmap,
    prepare_user,
    prepare_department,
    prepare_group,
    prepare_organization,
    prepare_service,
    prepare_domain,
    objects_map_by_id,
)
from intranet.yandex_directory.src.yandex_directory.common.models.types import (
    TYPE_USER,
    TYPE_DEPARTMENT,
    TYPE_GROUP,
    TYPE_RESOURCE,
    TYPE_ORGANIZATION,
    TYPE_SERVICE,
    TYPE_DOMAIN,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.retry import retry_http_errors
import collections.abc

from intranet.yandex_directory.src.yandex_directory.common import http_client

RESOURCE_CONTENT_STRUCTURE = {
    'relations': {
        'add': {
            'users': [],
            'departments': [],
            'groups': [],
        },
        'remove': {
            'users': [],
            'departments': [],
            'groups': [],
        },
    }
}


# кэш для webhook подписок на события про организацию
# нам на каждый запрос приходится ходить в базу за получением подписчиков
# чтобы этого не делать, мы можем доставать информацию из этого кэша
if os.environ.get('ENVIRONMENT') == 'autotests':
    webhooks_cache = None
else:
    webhooks_cache = TTLCache(
        maxsize=app.config['WEBHOOKS_CACHE_SIZE'],
        ttl=app.config['WEBHOOKS_CACHE_TTL'],
    )
webhooks_lock = RLock()


PREPARE_MAP = {
    TYPE_USER: partial(prepare_user, api_version=1),
    TYPE_DEPARTMENT: partial(prepare_department, api_version=1),
    TYPE_GROUP: partial(prepare_group, api_version=1),
    TYPE_RESOURCE: lambda connection, x: x,
    TYPE_ORGANIZATION: prepare_organization,
    TYPE_SERVICE: prepare_service,
    TYPE_DOMAIN: prepare_domain,
}

@cached(webhooks_cache,
        key=lambda *args, **kwargs: args[1],
        lock=webhooks_lock)
def get_web_hooks(meta_connection, event_name):
    """
    Список подписчиков для оповещения по web hook-ам
    :param meta_connection:
    :param event_name: событие
    :type event_name: str
    """
    # Опять из-за циклической зависимости приходится
    # делать импорт внутри
    model = WebHookModel(meta_connection)

    environment = app.config['ENVIRONMENT']
    # Так как окружения для стендов состоят из двух частей, типа:
    # testing.dir-5988, а вебхуки привязываются только к окружениям
    # testing, qa и internal-prod, то тут надо взять только первую
    # часть значения, разделённого точкой.
    environment = environment.split('.', 1)[0]

    base_query = model.filter(environment=environment)
    # подписка на любое событие
    any_event = base_query.filter(event_names=[]).all()
    # подписка на конкретное событие
    by_event = base_query.filter(event_names__contains=event_name).all()

    listeners = []
    for web_hook in any_event + by_event:
        url = web_hook['url']

        item = {
            'callback': url,
            'settings': {
                'expand_content': web_hook['expand_content']
            }
        }
        if web_hook['fields_filter']:
            item['filter'] = web_hook['fields_filter']

        if web_hook['service_id']:
            service_id = web_hook['service_id']
            item['service_id'] = service_id

            # Так же, нужно проставить отдельный признак, если сервису
            # позволено работать с любой организацией
            service = ServiceModel(meta_connection).get(service_id)
            # Этот признак потом понадобится, чтобы определить
            # нужно ли доставлять событие этому сервису или он не имеет
            # права работать с данными организации
            item['can_work_with_any_organization'] = \
                check_scopes(service['scopes'], [scope.work_with_any_organization])

        listeners.append(item)
    return listeners


def call_callback(main_connection, callback, data):
    """
    Вызов функции как колбека события
    """
    log_fields = {
        'event': data.get('event'),
        'callable_object': callback.__name__,
    }
    with log.fields(**log_fields):
        log.debug('Sending notification to callback')
        try:
            callback(main_connection, **data)
        except Exception:
            log.trace().error('Some exception when sending data to callback')
            raise


@retry(stop_max_attempt_number=3,
       wait_incrementing_increment=100,
       retry_on_exception=retry_http_errors('notify'))
def http_post_with_retries(url, data, headers, timeout, verify=None):
    """Эта функция делает почти то же самое, что requests.post, но
       все пятисотки, а так же 423 (Locked) и 429 (Too many requests)
       коды ретраит до 3 раз с интервалом в 100ms.
    """

    return http_client.request(
        'post',
        url,
        data=data,
        headers=headers,
        timeout=timeout,
        verify=verify,
    )


def http_post_with_handled_errors(url, data, headers, timeout, verify=None):
    """Эта функция не выкидывает исключений при HTTP ошибках, а возвращает
       status_code и ответ.
       Некоторые ошибки ретраятся до 3 раз, за счёт того, что внутри используется
       http_post_with_retries.
    """
    try:
        response = http_post_with_retries(
            url,
            data=data,
            headers=headers,
            timeout=timeout,
            verify=verify,
        )
        return response.status_code, response.content
    except HTTPError as exception:
        log.trace().error('HTTP error')
        if getattr(exception, 'response', None) is not None:
            status_code = exception.response.status_code
            content = exception.response.content
            return status_code, content
        else:
            return 500, ''
    except ReadTimeout:
        log.trace().error('HTTP timeout')
        return 542, 'read timeout'


def send_data_to_webhook(url, data, custom_headers, verify=None):
    """
    Отправка нотифайки как колбека события
    """
    from intranet.yandex_directory.src.yandex_directory.auth import tvm

    # Подготовим заголовки для вебхука.
    headers = {
        'Content-Type': 'application/json',
    }

    # Если для вебхука есть tvm_client_id, то его тикет должен
    # быть тикет в tvm.tickets. Так используем же его!
    if url in tvm.tickets:
        headers['X-Ya-Service-Ticket'] = tvm.tickets[url]

    if 'org_id' in data:
        headers['X-Org-ID'] = str(data['org_id'])

    # Заголовки могут быть переопределены для каждого вебхука
    # поэтому обновим словарь заголовков данными из настроек
    # вебхука.
    if custom_headers:
        headers.update(custom_headers)

    # заменяем obj на object (не ломаем апи нотифаек для потребителей)
    notification_data = {
        'event': data['event'],
        'object': data['obj'],
        'content': data.get('content'),
        'revision': data['revision'],
        'org_id': data['org_id'],
    }
    log_fields = {
        'payload': notification_data,
        'url': url,
        'headers': hide_auth_from_headers(headers),
        'org_id': data['org_id'],
    }
    log_fields.update(notification_data)

    with log.fields(**log_fields):
        log.debug('Sending event to webhook')

        if verify is None:
            verify = app.config['YANDEX_ALL_CA']

        response_log_fields = {}

        try:
            timeout = app.config['DEFAULT_NOTIFICATION_TIMEOUT']
            status_code, response_content = http_post_with_handled_errors(
                url,
                data=json.dumps(notification_data),
                headers=headers,
                timeout=timeout,
                verify=verify,
            )
            response_log_fields['response_code'] = status_code
            response_log_fields['response'] = response_content
        except Exception:
            with log.fields(**response_log_fields):
                log.trace().error('Exception while sending event')
            app.stats_aggregator.inc('notify_send_data_to_webhook_errors_summ')
            raise

        with log.fields(**response_log_fields):
            log.debug('Event has been sent')

        return status_code, response_content


def get_content_object(subject=None, before=None, diff=None, extra=None, directly=True):
    """
    Возвращает объект для записи в поле events.content.
    Args:
        subject (object): объект (группа, департамент, пользователь), который вызвал изменение и сгенерировал событие.
        before (object): объект до изменения
        diff (dict): словарь из полей объекта, которые были изменены. Формат:
            diff = {
                'param_field1': [delete_value11, add_value12, ...],
                'param_field2': [delete_value21, add_value22, ...],
                ...
            }
        extra (dict): в content[extra] можно передавать что угодно, в зависимости от желаний потребителя.
        directly (bool): если события вызываются для родителей, то directly=False,
        если напрямую для объекта, то directly=True.
    Returns:
        dict: content
    """
    content = {}
    if subject:
        content.update({'subject': subject})
    if before:
        content.update({'before': before})
    if diff:
        content.update({'diff': diff})
    if extra:
        content.update(extra)
    content.update({'directly': directly})
    return content


def notify_callback(meta_connection, main_connection, callback, event, settings=None):
    """
    Оповещение подписчика о событии.
    Эта функция возвращает результат даже если сервер вернул HTTP ошибку.
    В качестве результата, возвращается HTTP код и контент ответа.
    Если callback, это функция, то HTTP код будет 200

    :param main_connection:
    :param callback: подписчик (url или путь к функции)
    :param event: событие (EventModel)
    :param settings: дополнительные настройки
    """
    event_name = event['name']
    value = event['object']
    content = event['content']
    revision = event['revision']
    org_id = event['org_id']

    settings = settings or {}

    object_type = event_name.split('_')[0]
    prepare_function = PREPARE_MAP.get(
        object_type,
        lambda connection, item: item
    )
    if object_type == 'organization' and value is None:
        value = {'id': org_id}
    value = prepare_function(main_connection, value)
    data = {
        'event': event_name,
        'revision': revision,
        'obj': value,
        'org_id': org_id,
    }
    if isinstance(callback, str):
        if not callback.startswith('http'):
            callback = import_from_string(callback, callback)
    elif not isinstance(callback, collections.abc.Callable):
        raise RuntimeError('Callback {0} is not callable'.format(callback))

    if settings.get('expand_content', False):
        data['content'] = content

    if isinstance(callback, collections.abc.Callable):
        data['content'] = content
        try:
            call_callback(main_connection, callback, data)
        except DuplicatedTask:
            default_log.info('Duplicated task found')

        # для каллбэков-функций код ответа всегда 200 и тела ответа нет
        return 200, None
    else:
        verify = settings.get('verify', None)
        custom_headers = settings.get('headers', {})
        return send_data_to_webhook(callback, data, custom_headers, verify)


def is_service_should_receive_the_event(main_connection,
                                        org_id,
                                        org_type,
                                        service_id,
                                        can_work_with_any_organization,
                                        event_name):
    # Если вебхук привязан к сервису,
    # и событие про конкретную организацию, то
    # его надо оправлять только в том случае, если
    # сервис включён для этой организации или имеет возможность
    # работать с любыми данными

    # Для облачных организаций при этом другая логика - события
    # для них получают только включенные сервисы
    if can_work_with_any_organization and org_type != CLOUD_ORGANIZATION_TYPE:
        return True

    # Получим информацию про подключенность сервиса
    org_services = get_organization_services(main_connection, org_id)
    service = org_services.get(service_id, None)
    with log.fields(service=service):
        service_enabled = service and service['enabled']
        # Этот признак нам нужен для того, чтобы понимать, что
        # сервис когда-то был подключен, и событие про отключение ему надо отправить
        service_was_disabled = service and not service['enabled']

        if not service_enabled:
            # События service_disabled мы отправляем только если сервис
            # когда-то был подключен. При чём события про отключение других
            # сервисов тоже будем отправлять. Это будет происходить всегда.
            # Если сервисам будет напряжно, то что-нибудь придумаем, например
            # будем отправлять события только про них самих, или только в течении
            # какого-то времени после отключения.
            if event_name == 'service_disabled' and service_was_disabled:
                return True
            else:
                log.debug('Skipped because service is not enabled')
                return False
        # если сервис еще не готов он должен получать только события о своем включении
        return service['ready'] or event_name == 'service_enabled'


def get_organization_services(main_connection, org_id):
    services = OrganizationServiceModel(main_connection).find(
        {
            'org_id': org_id,
            'enabled': Ignore,
        },
        fields=['enabled', 'ready', 'service_id'],
    )
    return objects_map_by_id(services, key='service_id')


def create_callback_tasks(main_connection, new_events):
    """
    Создание задачи на оповещение о событии
    :param main_connection:
    :param new_events: { (org_id, callback): [{event_id: id события, settings: настройки для callback}] }
    :type new_events: dict
    :return:
    """
    environment = app.config['ENVIRONMENT']

    for key, events in new_events.items():
        org_id, callback = key
        with log.fields(callback=callback):
            for event in events:
                with log.fields(event_id=event['event_id']):
                    already_exists = CallbackEventsModel(main_connection) \
                                         .filter(
                                             callback=callback,
                                             event_id=event['event_id'],
                                             settings=event['settings'],
                                             environment=environment,
                                         ).count() > 0
                    if not already_exists:
                        log.debug('Adding callback event')
                        CallbackEventsModel(main_connection).create(
                            callback=callback,
                            event_id=event['event_id'],
                            settings=event['settings'],
                            environment=environment,
                        )
                    else:
                        log.debug('Callback event was already added')
            try:
                log.debug('Creating notification task')
                EventNotificationTask(main_connection).delay(org_id=org_id, callback=callback)
            except DuplicatedTask as exc:
                with log.fields(existing_task_id=exc.existing_task_id):
                    log.warning('Task for this callback and org_id is already in the queue')


def get_all_oldschool_subscriptions():
    # Эту функцию удобно мокать в тестах, если тебе кажется, что
    # код можно упростить, заинлайнив её, не делай этого :)
    return app.config['SUBSCRIPTIONS']


def get_oldschool_subscriptions(event_name):
    # Эту функцию удобно мокать в тестах, если тебе кажется, что
    # код можно упростить, заинлайнив её, не делай этого :)
    return get_all_oldschool_subscriptions().get(event_name, [])


def get_callbacks_for_events(meta_connection, main_connection, events, org_type):
    """Сюда на вход подаются события только тех организаций, которые
       относятся к текущему окружению. А раньше могли передаваться
       любые события, и они фильтровались по ходу дела.
    """
    new_events = defaultdict(set)
    for event in events:
        event_name = event['name']
        org_id = event['org_id']
        event_id = event['id']
        object_value = event['object']

        with log.fields(event=event_name, event_id=event_id, org_type=org_type):
            log.debug('Creating notifications tasks')
            # Раньше тут было так:
            # web_hook_subscriptions = get_web_hooks(meta_connection, event_name)
            # subscriptions = web_hook_subscriptions
            # subscriptions += get_oldschool_subscriptions(event_name)
            # операция += делала append к списку вебхуков которые надо оповестить
            # и постоянно растила этот "закэшированный" список.
            # Иммютабельность, бессердечная ты ссука! А точнее, твоё отсутствие!
            #
            # Правильно делать так:
            subscriptions = get_web_hooks(meta_connection, event_name) \
                          + get_oldschool_subscriptions(event_name)

            if org_type == CLOUD_ORGANIZATION_TYPE:
                # Не нужно оповещать неявных подписчиков о
                # событиях облачных организаций
                subscriptions = get_web_hooks(meta_connection, event_name)

            for listener in subscriptions:
                callback = listener['callback']

                with log.fields(callback=callback):
                    settings = listener.get('settings', {})
                    settings['headers'] = listener.get('headers', {})

                    send = True
                    if 'service_id' in listener and org_id:
                        # Если вебхук привязан к сервису,
                        # и событие про конкретную организацию, то
                        # его надо оправлять только в том случае, если
                        # сервис включён для этой организации.
                        send = is_service_should_receive_the_event(
                            main_connection,
                            org_id,
                            org_type,
                            listener['service_id'],
                            listener.get('can_work_with_any_organization'),
                            event_name
                        )

                    if 'filter' in listener:
                        # Все значения из фильтра должны содержаться в объекте
                        # если хоть одно не совпадает, то событие не будет отправлено
                        for key, value in listener['filter'].items():
                            if object_value.get(key) != value:
                                send = False
                                log.debug('Skipped because of filter')
                                break

                    if send:
                        key = (org_id, callback,)
                        new_events[key].add(hashabledict({
                            'event_id': event_id,
                            'settings': settings,
                        }))
    return new_events


def _get_continuous_events(events, last_processed_event_id, skip_head_hole=False):
    """
    Получаем первый блок непрерывных событий.
    :param events: события для оповещения, возможно с дырами
    :param last_processed_event_id: id последнего обработанного события
    :param skip_head_hole: флаг, обозначающий, что надо проигнорировать
                           last_processed_event_id и просто взять все события до
                           первой "дыры"
    """
    result = []
    if not events:
        return []
    prev_event_id = last_processed_event_id
    events = sorted(events, key=lambda e: e['id'])

    if skip_head_hole:
        prev_event_id = events[0]['id']-1

    for event in events:
        current_event_id = event['id']
        if current_event_id == prev_event_id + 1:
            prev_event_id = event['id']
            result.append(event)
        else:
            break
    return result


def select_not_processed_revisions(main_connection, environment, revision_limit=1000, since=None):
    """Возвращает ревизии организаций которые ещё не были обработаны.
       Данные возвращаются не все, так как их может накопиться много.
       Поэтому мы ограничиваем их количество параметром revision_limit.

       Эта функция всегда отдаёт tuple вида:
       (org_id, org_type, revision, difference)

       Здесь difference - разница между last_processed_revision и ревизией.
       по difference удобно считать есть ли дыра.

       since - время когда была последняя обработка событий. Нужно для того, чтобы
       не перелопачивать всю табличку actions

       Данные отсортированы по org_id, revision.
    """
    if since:
        filter_by_timestamp = 'actions.timestamp > %(since)s AND '
    else:
        filter_by_timestamp = ''

    query = """
SELECT actions.org_id,
       organizations.organization_type as org_type,
       actions.revision,
      (actions.revision - last_processed_revision.revision) as difference
  FROM actions
 INNER JOIN last_processed_revision
         ON actions.org_id = last_processed_revision.org_id
 INNER JOIN organizations on organizations.id = actions.org_id
 WHERE {filter_by_timestamp}
       actions.revision > last_processed_revision.revision
   AND last_processed_revision.environment = %(environment)s
 ORDER BY org_id, revision
 LIMIT %(limit)s
    """.format(filter_by_timestamp=filter_by_timestamp)

    results = main_connection.execute(
        query,
        environment=environment,
        limit=revision_limit,
        since=since,
    )
    return results.fetchall()


def group_revisions_by_organizations(revision_tuples):
    """Преобразует туплы, отданные функцией select_not_processed_revisions
       в словарь org_id -> (org_type, [revision, delta]).
       Предполагается, что данные уже отсортированы по org_id.
    """
    grouped_data = groupby(revision_tuples, lambda elem: (elem[0], elem[1]))
    return {
        item[0][0]: (item[0][1], [tupl[2:] for tupl in item[1]])
        for item in grouped_data
    }


class Hole(object):
    def __init__(self, revision, delta):
        self.revision = revision
        self.delta = delta


def get_continuous_revisions(revisions):
    """Получает на вход список туплов (revision, delta).
       Отдаёт список ревизий у которого delta начинается с 0 и монотонно возрастает.
       Например, для [(42, 1), (43, 2), (47, 6)]
       ответом будет [(42, 1), (43, 2)], так как дальше начинается "дыра".

       Если входная последовательность сразу начинается с "дыры", то
       функция возвращает объект Hole(revision, delta).
    """
    result = []
    if revisions:
        # дыры нет
        first_item_delta = revisions[0][1]
        if first_item_delta == 1:
            prev_delta = first_item_delta - 1
            for item in revisions:
                item_delta = item[1]
                if item_delta != prev_delta + 1:
                    break
                result.append(item)
                prev_delta = item_delta
        # если данные начинаются с "дыры"
        else:
            result = Hole(revisions[0][0], revisions[0][1])
    return result


def process_organization_revisions(org_id, org_type, org_revisions, wait_tills, environment, shard, now):
    # Тут мы получаем новый коннект, так как эта функция будет выполняться во много потоков
    # и в каждом потоке должен использоваться свой коннект
    with log.fields(org_id=org_id, environment=environment), \
         get_main_connection(shard=shard, for_write=True) as main_connection_in_thread, \
         get_meta_connection() as meta_connection:

        log.debug('Checking which events can be delivered to subscribers')
        try:
            batch = get_continuous_revisions(org_revisions)
            # если тут None, то last_procesed
            last_processed_revision = NotChanged
            wait_till = NotChanged

            if isinstance(batch, Hole):
                # Нашли дыру
                hole = batch
                # сами batch сбросим в [] чтобы в этот раз ничего не обрабатывать
                batch = []
                with log.fields(hole_starting_revision=hole.revision,
                                hole_size=hole.delta):
                    log.warning('Found a hole')

                    wait_till = wait_tills.get(org_id)

                    if wait_till is None:
                        # a) для организации wait_till не установлен – значит мы обнаружили
                        # новую дыру и надо wait_till установить на now() + TTL;
                        ttl = app.config['EVENT_HOLE_WAIT_TTL']
                        wait_till = time_in_future(seconds=ttl)
                        with log.fields(wait_till=wait_till):
                            log.warning('Pausing event processing because of a hole')

                    elif wait_till > now:
                        # б) wait_till установлен и он ещё в будущем – значит пока ничего не делаем,
                        #    просто ждём пока зарастёт дыра.
                        with log.fields(wait_till=wait_till):
                            log.warning('Still waiting for a hole to disappear')
                    else:
                        # в) wait_till установлен и уже в прошлом – значит "дыру" пора пропустить и приступить к обработке
                        #    последующих событий. Поэтому проставим wait_till в None,
                        #    а последнюю обработанную ревизию на конец "дыры",
                        #    чтобы в следующий раз подхватилась новая пачка событий
                        last_processed_revision = org_revisions[0][0] - 1
                        with log.fields(wait_till=wait_till):
                            log.warning('Timeout expired, leaving hole as is and proceeding to a last revision.')
                        wait_till = None
            else:
                # Если найдена пачка ревизий о которой надо разослать оповещения, то
                # надо установить last_processed_revision в максимальную ревизию,
                # а wait_till сбросить в None
                last_processed_revision = batch[-1][0]
                wait_till = None
                previous_wait_till = wait_tills.get(org_id)
                if previous_wait_till:
                    with log.fields(previous_wait_till=previous_wait_till):
                        log.warning('Hole disappeared')

            # Теперь займёмся постановкой тасков на рассылку событий
            log_fields = {}

            if batch:
                batch_revisions = list(map(itemgetter(0), batch))
                min_revision = min(batch_revisions)
                max_revision = max(batch_revisions)
                log_fields['min_revision'] = min_revision
                log_fields['max_revision'] = max_revision

            with log.fields(**log_fields):
                if batch:
                    events = EventModel(main_connection_in_thread) \
                                 .filter(
                                     org_id=org_id,
                                     revision__between=(
                                         min_revision,
                                         max_revision
                                     )) \
                                 .fields('id', 'name', 'org_id', 'object', 'notify_at', 'environment') \
                                 .order_by('id') \
                                 .all()
                    # запомним обработанные события
                    event_ids = [e['id'] for e in events]
                    with log.fields(events=event_ids):
                        log.debug('Saving processed events')

                    processed_events = [
                        {
                            'event_id': e['id'],
                            'environment': environment,
                        }
                        for e in events
                    ]
                    ProcessedEventsModel(main_connection_in_thread).bulk_create(
                        processed_events,
                        strategy=Values(on_conflict=Values.do_nothing),
                    )
                    # for e in events:
                    #     with log.fields(event_id=e['id']):
                    #         log.debug('Saving event')
                    #         ProcessedEventsModel(main_connection_in_thread).create(
                    #             event_id=e['id'],
                    #             environment=environment,
                    #         )


                    log.debug('Getting callbacks for events')
                    new_events = get_callbacks_for_events(meta_connection, main_connection_in_thread, events, org_type)

                    log.debug('Creating callback tasks')
                    create_callback_tasks(main_connection_in_thread, new_events)

                # И наконец подновим данные в таблице обработанных событий
                log.debug('Setting last processed revision')
                # Ревизия не всегда меняется, поэтому
                # значение может быть NotChanged
                update_fields = dict(
                    revision=last_processed_revision,
                    wait_till=wait_till,
                    updated_at=now,
                )
                update_fields = remove_not_changed_keys(update_fields)
                if update_fields:
                    LastProcessedRevisionModel(main_connection_in_thread) \
                        .filter(org_id=org_id, environment=environment) \
                        .update(**update_fields)
        except Exception:
            log.trace().error('Unable to process revisions for organization')

        log.info('Processing is done')



def notify_about_new_events(meta_connection, main_connection):
    """
    Эта функция работает так:

    1. Выбирает все новые ревизии появившиеся для организаций спустя зафиксированные last_processed_revision.
    2. Группирует их по организациям.
    3. Для каждой организации находит пачку ревизий, события о которые можно отправить. в этой пачке не должно быть "дыры".
       Допустим, если на шаге 2 для организации были собраны ревизии: 1 2 3 4 8 9, то в "пачку"
       войдут только ревизии 1 2 3 4.
    4. Если для организации найдена "дыра", то этот факт фиксируется в специальном флаге hole_found.
       "дырой" считаем ситуацию, когда, например, last_processed_revision = 4, а на шаге 2 мы получили список
       событий с ревизиями 8 9. в этой ситуации возможны три варианта развития событий:
       a) для организации wait_till не установлен – значит мы обнаружили новую дыру и надо wait_till установить
          на now() + ttl;
       б) wait_till установлен и он ещё в будущем – значит пока ничего не делаем, просто ждём пока зарастёт дыра;
       в) wait_till установлен и уже в прошлом – значит "дыру" пора пропустить и приступить к обработке
          последующих событий. поэтому проставим wait_till в none,
          а последнюю обработанную ревизию на конец "дыры",
          чтобы в следующий раз подхватилась новая пачка событий.
    5. Для всех найденных "пачек" запускается таск на доставку событий для этой организации.
    6. Для всех организаций по которым создали таски, обновляем записи про last_processed_revision.
       При этом, если установлен флаг hole_found, то для этой организации так же будет проставлено время
       wait_till. Оно будет указывать на момент в будущем, после которого мы решим, что "дыра" уже никогда
       не зарастёт, и продолжим отправлять события для этой организации.
    """

    revision_limit = app.config['NOTIFY_BATCH_SIZE']
    pool_size = app.config['NOTIFY_THREAD_POOL_SIZE']
    environment = app.config['ENVIRONMENT']
    now = utcnow()

    last_processing_was_at = main_connection.execute(
        'SELECT max(updated_at) FROM last_processed_revision'
    ).fetchall()[0][0]

    revisions = select_not_processed_revisions(
        main_connection,
        environment,
        revision_limit=revision_limit,
        # Когда разгребаем большую пачку действий, которые могли нагенериться
        # несколько часов назад, надо уметь отсутпить на эти несколько часов назад
        since=last_processing_was_at - datetime.timedelta(minutes=60 * 4),
    )
    revisions = group_revisions_by_organizations(revisions)
    org_ids = list(revisions.keys())


    if org_ids:
        items = LastProcessedRevisionModel(main_connection) \
                .filter(
                    environment=environment,
                    org_id=org_ids,
                ) \
                .fields('org_id', 'wait_till') \
                .all()
    else:
        items = []

    with log.fields(org_ids=org_ids):
        log.info('Processing events for organizations')

    wait_tills = {
        item['org_id']: item['wait_till']
        for item in items
    }

    input_data = (
        {
            'org_id': org_id,
            'org_type': org_type,
            'org_revisions': org_revisions,
            'wait_tills': wait_tills,
            'environment': environment,
            'shard': main_connection.shard,
            'now': now,
        }
        for org_id, (org_type,  org_revisions) in revisions.items()
    )
    pmap(process_organization_revisions, input_data, pool_size=pool_size)

    log.info('Revision processing is done')
