# -*- coding: utf-8 -*-
from intranet.yandex_directory.src.yandex_directory.common.exceptions import APIError


# Это исключение было перенесено сюда из billing.exceptions.
# Возможно его стоит объединить с OrganizationBillingException
class BillingException(Exception):
    def __init__(self, status_code, message, **params):
        self.status_code = status_code
        self.message = message
        self.params = params


class OrganizationBillingException(APIError):
    code = 'unknown_billing_error'
    status_code = 503
    message = 'Unknown organization billing exception'


class OrganizationIsWithoutContract(OrganizationBillingException):
    # вызывается при попытке действий, которые могут
    # быть осуществлены только с организациями подключенными к Биллингу
    status_code = 422
    code = 'organization_is_without_contract'
    message = 'Organization is without contract'
    description = 'У организации не заполнена платёжная информация.'


class OrganizationHasBillingInfo(OrganizationBillingException):
    status_code = 403
    code = 'organization_has_billing_info'
    message = 'Organization has billing info'
    description = 'У организации заполнена платёжная информация.'


class OrganizationIsNotPartner(APIError):
    # вызывается при попытке действий, которые могут
    # быть осуществлены только с организациями подключенными к Биллингу
    status_code = 422
    code = 'organization_is_not_partner'
    message = 'Organization is not partner'
    description = 'Это действие доступно только для "партнёрских" организаций.'


class OrganizationIsOnFreePlan(OrganizationBillingException):
    # вызывается при попытке действий, которые могут
    # быть осуществлены только с платными организациями
    code = 'organization_is_on_free_plan'
    description = 'Действие доступно только для организаций на платном тарифе.'


class OrganizationHasDebt(OrganizationBillingException):
    # вызывается при попытке действий, которые могут
    # быть осуществлены только с организациями без задолженности
    status_code = 422
    code = 'organization_has_debt'
    message = 'Organization has debt'
    description = 'Операция невозможна, так как у организации есть задолженность.'


class OrganizationAlreadyHasContract(OrganizationBillingException):
    # вызывается при попытке действий, которые могут
    # быть осуществлены только с организациями не заведенными в Биллинге
    status_code = 422
    code = 'organization_already_has_contract'
    message = 'Organization already has contract'
    description = 'У организации уже есть заполенная платёжная информация.'


class TooManyPersonsFromBilling(OrganizationBillingException):
    status_code = 422
    code = 'too_many_person_ids_from_billing'
    message = 'Got incorrect data from billing - please, contact support'
    description = 'Получили некорректные данные из биллинга - обратитесь в поддержку'


class BillingInvalidField(BillingException):
    code = 'invalid_{field}'
    status_code = 422
    message = 'Field "{field}" has invalid value'

    def __init__(self, field):
        self.code = self.code.format(field=field)
        self.params = {'field': field}


class BillingMissingField(BillingException):
    code = 'required_field'

    def __init__(self, field):
        super(BillingMissingField, self).__init__(
            422,
            'Please, provide field "{field}"',
            field=field,
        )


class BillingUidNotFoundInPassport(BillingException):
    code = 'user_uid_not_found_in_passport'

    def __init__(self, uid):
        super(BillingUidNotFoundInPassport, self).__init__(
            404,
            'Passport account with ID "{uid}" not found',
            uid=uid,
        )


class BillingUnknownError(BillingException):
    code = 'unknown'

    def __init__(self):
        super(BillingUnknownError, self).__init__(
            503,
            'Unknown billing error',
        )


class BillingClientIdNotFound(BillingException):
    code = 'client_id_not_found'

    def __init__(self, client_id):
        super(BillingClientIdNotFound, self).__init__(
            422,
            'Billing client ID {} not found'.format(client_id),
        )


class BillingClientIdMismatch(BillingException):
    code = 'client_id_mismatch'

    def __init__(self):
        super(BillingClientIdMismatch, self).__init__(
            422,
            'Billing client id association mismatch',
        )


class OrganizationDeleted(APIError):
    with_trace = False
    status_code = 404
    code = 'organization_deleted'
    log_level = 'WARNING'
    message = 'Organization was deleted'
    description = 'Организация была удалена'


class OrganizationUnknown(APIError):
    status_code = 404
    log_level = 'WARNING'
    with_trace = False
    code = 'unknown_organization'
    message = 'Unknown organization'
    description = 'Организация не найдена'


class PromocodeInvalidException(APIError):
    status_code = 422
    code = 'invalid_promocode'
    message = 'Invalid promo code'
    description = 'Промокод не верен.'


class PromocodeExpiredException(APIError):
    status_code = 422
    code = 'expired_promocode'
    message = 'Expired promo code'
    description = 'Срок действия промокода истёк.'


class EmailMigrationException(APIError):
    code = 'unknown_email_migration_error'
    status_code = 503
    message = 'Unknown emails migration exception'


class MigrationFileTooLarge(EmailMigrationException):
    status_code = 422
    code = 'migration_file_is_too_large'
    message = 'File size is more then 1 Mb'
    description = 'Файл с данными для миграции почты больше 1 мегабайта.'


class RequiredFieldsMissed(EmailMigrationException):
    code = 'required_fields_missed'
    status_code = 422
    message = '{fields} are required fields'
    description = 'В файле отсутствуют требуемые поля.'

    def __init__(self, required_fields):
        super(RequiredFieldsMissed, self).__init__(fields=required_fields)


class MigrationFileParsingError(EmailMigrationException):
    status_code = 422
    code = 'migration_file_parsing_error'
    message = 'Migration file has invalid format'
    description = 'Неверный формат данных в описании ящиков.'


class DomainExistsInPassport(APIError):
    code = 'domain_exists_in_passport'
    status_code = 422
    message = 'Domain exists in Passport'
    log_level = 'WARNING'
    description = 'Домен не может быть добавлен, так как он уже есть в Паспорте. Обратитесь в саппорт.'


class DomainIsAlias(APIError):
    code = 'domain_is_alias'
    status_code = 422
    message = 'Domain is alias in passport'
    log_level = 'WARNING'
    description = 'Домен не может быть добавлен, так как он уже является алиасом.'


class DomainHasAccounts(APIError):
    code = 'domain_has_accounts'
    status_code = 422
    message = 'Domain has accounts in passport'
    log_level = 'WARNING'
    description = 'Домен не может быть добавлен, так как он уже есть в Паспорте и на нём есть аккаунты.'


class DomainHasAliases(APIError):
    code = 'domain_has_aliases'
    status_code = 422
    message = 'Domain has aliases in passport'
    log_level = 'WARNING'
    description = 'Домен не может быть добавлен, так как у него есть алиасы'


class UnableToBlockServiceResponsible(APIError):
    """
    Ответственный всегда должен иметь доступ к организации.
    """
    code = 'cannot_block_service_owner'
    status_code = 422
    message = 'Person, responsible for any service, must always have access to organization'
    log_level = 'WARNING'
    description = 'Нельзя заблокировать ответственного за сервис'


class UnableToDismissServiceResponsible(APIError):
    """
    Ответственный всегда должен иметь доступ к организации.
    """
    code = 'cannot_dismiss_service_owner'
    status_code = 422
    message = 'Person, responsible for any service, must always have access to organization'
    log_level = 'WARNING'
    description = 'Нельзя уволить ответственного за сервис'


class ExternalIDAlreadyUsed(APIError):
    status_code = 409
    code = 'some_user_has_this_external_id'
    message = 'Some user already exists with external_id'
    description = 'Сотрудник с таким external_id уже существует.'


class RelationIsAlreadyExists(APIError):
    status_code = 409
    code = 'relation_already_exists'
    message = 'Relation already exists'
    description = 'Связь уже сущствует.'


class RelationWithResourceAlreadyExists(APIError):
    status_code = 409
    code = 'relation_with_resource_already_exists'
    message = 'User already has relation with this resource'
    description = 'У сотрудника уже есть доступ к этому объекту. Сначала удалите старый доступ.'


class InvalidDepartmentId(APIError):
    status_code = 422
    code = 'invalid_department_id'
    message = 'Invalid department id was passed'
    description = 'Был передан неверный id департамента'


class NoActivePerson(OrganizationBillingException):
    # Не удалось найти активный договор у плательщика
    status_code = 422
    code = 'no_active_person_found'
    message = 'No valid payment method found, please, refill payment information'
    description = 'Не найден подходящий платежный способ, пожалуйста, заполните платежную информацию'


class ValidationQueryParametersError(APIError):
    code = 'validation_query_parameters_error'
    status_code = 422
    message = 'Query parameters "{query_params}" are not valid'
    description = 'Параметры запроса "{query_params}" не валидные'

    def __init__(self, query_params):
        super(ValidationQueryParametersError, self).__init__(query_params=query_params)


class CloudValidationError(APIError):
    status_code = 422
    code = 'schema_validation_error'
    description = 'Schema validation error'


class CloudUnavailableError(APIError):
    status_code = 422
    code = 'cloud_unavailable_error'
    description = 'Cloud internal error'
