# -*- coding: utf-8 -*-
import os
from threading import RLock

from cachetools import TTLCache, cached

from intranet.yandex_directory.src.yandex_directory.common.exceptions import FeatureNotFound
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from .rules import rules

NOT_DEFINED = object()

if os.environ.get('ENVIRONMENT') == 'autotests':
    feature_cache = None
else:
    feature_cache = TTLCache(
        maxsize=1000,
        ttl=60*60,
    )
feature_lock = RLock()

if os.environ.get('ENVIRONMENT') == 'autotests':
    org_feature_value_cache = None
else:
    org_feature_value_cache = TTLCache(
        maxsize=1000,
        ttl=5*60,
    )
org_feature_value_lock = RLock()


@cached(feature_cache, lock=feature_lock)
def get_feature_by_slug(meta_connection, feature_slug):
    query = """
        SELECT id, enabled_default
        FROM features
        WHERE slug = %(feature_slug)s
    """
    return meta_connection.execute(query, feature_slug=feature_slug).fetchone()


def is_feature_enabled_for_organization(meta_connection, feature_id, org_id):
    query = """
        SELECT enabled
        FROM org_features
        WHERE org_id = %(org_id)s
        AND feature_id = %(feature_id)s
    """
    feature = meta_connection.execute(query, org_id=org_id, feature_id=feature_id).fetchone()
    if feature:
        return feature[0]
    return NOT_DEFINED


@cached(org_feature_value_cache, lock=org_feature_value_lock)
def is_feature_enabled(meta_connection, org_id, feature_slug):
    # Возвращает значение фичи для данной организации, либо дефолтное значение фичи
    feature = get_feature_by_slug(meta_connection, feature_slug)
    if not feature:
        raise FeatureNotFound

    feature_id, enabled_default = feature
    feature_org = is_feature_enabled_for_organization(meta_connection, feature_id, org_id)
    return enabled_default if feature_org is NOT_DEFINED else feature_org


def get_organization_features_info(meta_connection, org_id):
    from intranet.yandex_directory.src.yandex_directory.core.utils import (
        objects_map_by_id,
    )
    query = """
        SELECT
            f.slug,
            COALESCE(of.enabled, f.enabled_default) as enabled,
            f.enabled_default as default,
		    COALESCE(f.description, '') as description
        from features as f
        left join org_features as of on f.id=of.feature_id and of.org_id = %(org_id)s
    """
    features = list(map(dict, meta_connection.execute(query, org_id=org_id).fetchall()))
    features = objects_map_by_id(features, 'slug', remove_key=True)

    return features


def set_feature_value_for_organization(meta_connection, org_id, feature_slug, enabled):
    from intranet.yandex_directory.src.yandex_directory.core.features import USE_DOMENATOR, SSO_AVAILABLE
    from intranet.yandex_directory.src.yandex_directory.common.db import (
        get_main_connection,
        get_shard,
    )
    from intranet.yandex_directory.src.yandex_directory.core.utils import (
        get_organization_admin_uid,
    )
    from intranet.yandex_directory.src.yandex_directory import app
    from ...sso.tasks import DisableSsoTask

    with log.fields(org_id=org_id, feature=feature_slug, enabled=enabled):
        log.info('Trying to set feature value for organization')
        feature = get_feature_by_slug(meta_connection, feature_slug)
        if not feature:
            log.warning('Feature not found')
            raise FeatureNotFound
        query = """
             INSERT INTO org_features(org_id, feature_id, enabled)
             SELECT '{org_id}', {feature_id}, {enabled}
             ON CONFLICT (org_id, feature_id) DO UPDATE
                 SET enabled={enabled}
        """.format(
            org_id=org_id,
            feature_id=feature['id'],
            enabled=enabled,
        )
        try:
            meta_connection.execute(query)
            log.info('Feature value has been set')
        except Exception:
            log.trace().error('Feature value has not been set')
            raise

        if enabled and feature_slug == USE_DOMENATOR:
            shard = get_shard(meta_connection, org_id)
            with get_main_connection(shard) as main_connection:
                admin_id = get_organization_admin_uid(main_connection, org_id)

            app.domenator.sync_connect(org_id=org_id, admin_id=admin_id)

        if not enabled and feature_slug == SSO_AVAILABLE:
            shard = get_shard(meta_connection, org_id)
            with get_main_connection(shard=shard, for_write=True) as main_connection:
                DisableSsoTask(main_connection).delay(org_id=org_id)


def get_features_to_enable(meta_connection,
                           main_connection,
                           org_id=None,
                           organization=None):
    from intranet.yandex_directory.src.yandex_directory.core.models.organization import OrganizationModel
    cache = {}
    features = []

    if not (org_id or organization):
        raise RuntimeError('Either org_id or organization should be given.')

    if organization is None:
        organization = OrganizationModel(main_connection).get(org_id, fields=['source', 'organization_type'])

    for feature, rule in list(rules.items()):
        if rule.evaluate(
                cache,
                meta_connection=meta_connection,
                main_connection=main_connection,
                organization=organization,
        ):
            features.append(feature)

    # Чтобы было удобнее смотреть на фичи глазами
    features.sort()
    return features


def enable_features(meta_connection,
                    main_connection,
                    organization):
    features = get_features_to_enable(meta_connection,
                                      main_connection,
                                      organization=organization)
    for feature in features:
        set_feature_value_for_organization(meta_connection,
                                           organization['id'],
                                           feature,
                                           enabled=True)


def get_enabled_organizations_features(meta_connection, shard, org_id=None):
    """
    Получаем список id включенных фич для заданной организации
    Если org_id не передан - возвращаем списки для всех организаций
    """
    query = '''
        SELECT o.org_id, array_agg(o.feature_id) as feature_ids
          FROM org_features o
        INNER JOIN organizations ON o.org_id = organizations.id
        INNER JOIN features f ON f.id = o.feature_id
        WHERE organizations.shard = %(shard)s
          AND o.enabled = TRUE
          AND f.enabled_default = FALSE
          {org_filter}
        GROUP BY org_id
    '''.format(
        org_filter=' AND o.org_id = {}'.format(org_id) if org_id else ''
    )
    result = meta_connection.execute(query, shard=shard).fetchall()
    return list(map(dict, result))


def is_multiorg_feature_enabled_for_user(meta_connection, uid) -> bool:
    """
    Проверяем, что пользователь портальный
    """
    from intranet.yandex_directory.src.yandex_directory.core.utils import is_outer_uid
    is_outer = is_outer_uid(uid)
    return is_outer
