# -*- coding: utf-8 -*-
import json
import requests

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth import tvm
from intranet.yandex_directory.src.yandex_directory.common.utils import log_service_response
from .exceptions import (
    IDMB2BException,
    IDMB2BConflictRequestError,
)
from ...common import http_client


class IDMB2BApiClient(object):
    def __init__(self):
        self.host = app.config['IDM_B2B_HOST']

    def get_roles(self, org_id, author_id, system=None, resource_id=None, role_type=None, lang=None, role_id=None,
                  path=None, link_type=None, uid=None, state=None):
        """
        https://wiki.yandex-team.ru/intranet/idm/API/public/#roles
        Пример возвращаемого значения:
        [
          {
            "review_date": null,
            "human_state": "\\u041e\\u0448\\u0438\\u0431\\u043a\\u0430",
            "with_robots": true,
            "human": "\\u0420\\u043e\\u043b\\u044c: group-1",
            "human_short": "group-1",
            "id": 1,
            "group": null,
            "system": {
              "use_mini_form": false,
              "description": "\\u0422\\u0435\\u0441\\u0442\\u043e\\u0432\\u0430\\u044f \\u0441\\u0438\\u0441\\u0442\\u0435\\u043c\\u0430 \\u0432 b2b",
              "group_policy": "unaware",
              "is_broken": false,
              "is_active": true,
              "name": "\\u0422\\u0435\\u0441\\u0442\\u043e\\u0432\\u0430\\u044f \\u0441\\u0438\\u0441\\u0442\\u0435\\u043c\\u0430 \\u0432 b2b",
              "endpoint_long_timeout": 60,
              "state": "\\u0410\\u043a\\u0442\\u0438\\u0432\\u043d\\u0430",
              "is_sox": false,
              "resource_uri": "",
              "slug": "testsystem-b2b",
              "endpoint_timeout": 60
            },
            "state": "failed",
            "fields_data": {
              "resource_id": "12345"
            },
            "node": {
              "is_auto_updated": false,
              "is_key": false,
              "set": null,
              "description": "",
              "value_path": "/group-1/",
              "system": {
                "use_mini_form": false,
                "description": "\\u0422\\u0435\\u0441\\u0442\\u043e\\u0432\\u0430\\u044f \\u0441\\u0438\\u0441\\u0442\\u0435\\u043c\\u0430 \\u0432 b2b",
                "group_policy": "unaware",
                "is_broken": false,
                "is_active": true,
                "name": "\\u0422\\u0435\\u0441\\u0442\\u043e\\u0432\\u0430\\u044f \\u0441\\u0438\\u0441\\u0442\\u0435\\u043c\\u0430 \\u0432 b2b",
                "endpoint_long_timeout": 60,
                "state": "\\u0410\\u043a\\u0442\\u0438\\u0432\\u043d\\u0430",
                "is_sox": false,
                "resource_uri": "",
                "slug": "testsystem-b2b",
                "endpoint_timeout": 60
              },
              "slug": "group-1",
              "slug_path": "/role/group-1/",
              "state": "active",
              "human": "\\u0420\\u043e\\u043b\\u044c: group-1",
              "human_short": "group-1",
              "is_public": true,
              "resource_uri": "",
              "data": {
                "role": "group-1"
              },
              "id": 10,
              "unique_id": "",
              "name": "group-1"
            },
            "updated": "2019-08-15T13:58:17.092117+00:00",
            "added": "2019-08-15T13:58:14.649302+00:00",
            "granted_at": null,
            "parent": null,
            "system_specific": null,
            "is_active": false,
            "with_external": true,
            "user": {
              "username": "yndx-idm-cnt-robot",
              "fired_at": null,
              "is_active": true,
              "sex": null,
              "full_name": "yndx-idm-cnt-robot",
              "position": "",
              "resource_uri": "",
              "type": "user",
              "email": "",
              "date_joined": "2019-08-15T12:23:37.393361+00:00"
            },
            "with_inheritance": true,
            "is_public": null,
            "data": {
              "role": "group-1"
            },
            "ttl_date": null,
            "last_request_comment": "Test",
            "review_at": null,
            "review_days": null,
            "organization": 1234567890,
            "ttl_days": null,
            "expire_at": null,
            "resource_uri": "/api/b2b/roles/1/"
          }
        ]
        """
        params = {'limit': 1000}
        if system is not None:
            params['system'] = system
        if resource_id is not None:
            params['fields_data'] = json.dumps({'resource_id': str(resource_id)})
        if role_type is not None:
            params['type'] = role_type
        if role_id is not None:
            role_ids = role_id if isinstance(role_id, (list, tuple)) else [role_id]
            params['id'] = ','.join(map(str, role_ids))
        if path is not None:
            params['path'] = path
        if link_type is not None:
            params['user_type'] = link_type
        if uid is not None:
            uids = uid if isinstance(uid, (list, tuple)) else [uid]
            params['user'] = ','.join(map(str, uids))
        if state is not None:
            states = state if isinstance(state, (list, tuple)) else [state]
            params['state'] = ','.join(states)

        url = self.host + '/api/b2b/roles/'

        roles = []
        while True:
            response = self._request('get', url, org_id, author_id, lang, params=params)
            roles += response['objects']

            url = response.get('meta', {}).get('next')

            if not url:
                break

        return roles

    def revoke_roles(self, org_id, author_id, roles, lang=None):
        """
        Балковый отзыв ролей по id
        https://wiki.yandex-team.ru/intranet/idm/API/public/#otzyvneskolkixrolejj
        Пример возвращаемого значения:
        {"errors": 0, "successes": 8}
        """

        url = self.host + '/api/b2b/roles/'
        roles = roles if isinstance(roles, (list, tuple)) else [roles]
        payload = {
            'id': ','.join(map(str, roles))
        }
        return self._request('delete', url, org_id, author_id, lang=lang, json=payload)

    def request_role(self, org_id, author_id, system, path, resource_id, uid, user_type=None, lang=None):
        """
        https://wiki.yandex-team.ru/intranet/idm/API/public/#zaprosroli
        """
        params = {
            'user': str(uid),
            'system': system,
            'path': path,
            'user_type': user_type or 'user',
            'fields_data': {'resource_id': str(resource_id)},
        }
        return self._request('post', '/api/b2b/rolerequests/', org_id, author_id, lang, json=params)

    def batch_request_roles(self, org_id, author_id, system, roles, lang=None):
        params = []
        for role in roles:
            params.append({
                'method': 'POST',
                'path': 'rolerequests/',
                'body': {
                    'user': role['uid'],
                    'user_type': role.get('user_type', 'user'),
                    'system': system,
                    'path': role['path'],
                    'fields_data': {
                        'resource_id': str(role['resource_id'])
                    }
                },
            })

        raw_response = self._request('post', '/api/b2b/batch/', org_id, author_id, lang, json=params)
        responses = raw_response['responses']

        return [
            response['body']
            for response in responses
        ]

    def _request(self, method, url, org_id, author_id, lang, **kwargs):
        if not url.startswith('http'):
            url = self.host + '/' + url.lstrip('/')

        headers = {
            'X-Ya-Service-Ticket': tvm.tickets['idm_b2b'],
            'X-UID': str(author_id),
            'X-ORG-ID': str(org_id),
        }

        if lang:
            headers['Accept-Language'] = lang
        headers.update(kwargs.pop('headers', {}))
        kwargs['headers'] = headers

        response = http_client.request(method, url, **kwargs)

        log_service_response('idm_b2b', method.upper(), url, kwargs, response)
        if response.status_code == 409:
            response_data = response.json()
            if response_data['error_code'] == 'CONFLICT':
                raise IDMB2BConflictRequestError()
        try:
            response.raise_for_status()
        except requests.exceptions.HTTPError as exc:
            raise IDMB2BException(original_exception=exc)
        return response.json()
