# -*- coding: utf-8 -*-

"""
Клиент для IDM
https://wiki.yandex-team.ru/Intranet/idm/API/
"""
from collections import defaultdict

from retrying import retry

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.caching.utils import make_key
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    url_join,
    log_service_response,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.retry import retry_http_errors
from intranet.yandex_directory.src.yandex_directory.common import http_client


class ADMIN_ROLES:
    admin = 'admin'
    support = 'support'
    assessor = 'assessor'
    bizdev = 'bizdev'


class IDMInternalApiClient(object):
    def __init__(self):
        # Если вдруг протухнет OAuth токен, с которым robot-connect ходит в IDM
        # нужно перегенерить его:
        # https://oauth.yandex-team.ru/authorize?response_type=token&client_id=b9fa533f1fc64782a1e4e9d559e1952f
        # и положить в секрет qloud

        self.api_url = app.config['IDM_API']
        self.token = app.config['IDM_TOKEN']
        self.connect_admin_slug = app.config['CONNECT_ADMIN_SLUG']
        self.headers = {
            "Authorization": "OAuth {}".format(self.token),
            "Content-type": "application/json",
        }

    def get_all_active_roles(self, login=None):
        return self._get_all_active_roles(login)

    @retry(wait_exponential_multiplier=500,
           wait_exponential_max=10000,
           stop_max_attempt_number=10,
           retry_on_exception=retry_http_errors('idm'))
    def _get_all_active_roles(self, login=None):
        # https://wiki.yandex-team.ru/intranet/idm/api/public/#spisokrolejjgetprefix/roles/
        key = make_key(
            prefix='admin_roles',
            key_data=(),
        )
        roles = app.cache.get(key)
        if not roles:
            url = url_join(self.api_url, 'api/v1/roles/')
            response = self._make_get_request(
                url,
                system=self.connect_admin_slug,
                type='active',
                ownership='personal',
                user=login,
                limit=1000,
            )
            roles = defaultdict(list)
            for item in response.get('objects', []):
                username = item['user']['username']
                role = item['data']['role']
                roles[username].append(role)
            app.cache.set(key, roles, app.config['IDM_CACHE_TTL'])
        return roles

    def _make_get_request(self, url, **kwargs):
        response = http_client.request('get', url, headers=self.headers, params=kwargs)
        log_service_response('idm', 'GET', url, kwargs, response)
        response.raise_for_status()
        json_response = response.json()
        return json_response
