# -*- coding: utf-8 -*-
import random
from copy import copy
import string
from collections import (
    defaultdict,
    Counter,
)

from transliterate import slugify

from intranet.yandex_directory.src.yandex_directory.common.utils import force_text
from intranet.yandex_directory.src.yandex_directory.core.importer.exception import (
    DuplicateUserNickname,
    UnknownUserDepartment,
    InvalidHeadId,
    NoToplevelDepartments,
    DuplicateDepartmentId,
    InvalidParentId,
    ToManyConnectedComponent,
    OrganizationNotFound,
)
from intranet.yandex_directory.src.yandex_directory.core.models.department import DepartmentModel
from intranet.yandex_directory.src.yandex_directory.core.models.user import UserModel
from intranet.yandex_directory.src.yandex_directory.core.models.organization import OrganizationModel
from intranet.yandex_directory.src.yandex_directory.common.models.types import ROOT_DEPARTMENT_ID
from intranet.yandex_directory.src.yandex_directory.core.utils import create_user, get_organization_admin_uid

ROOT_DEPARTMENT = object()

PASSWORD_CHARS = string.ascii_letters + string.digits
PASSWORD_LENGTH = 8


def generate_password(length):
    return ''.join(random.choice(PASSWORD_CHARS) for _ in range(length))


def generate_nickname_if_need(users):
    """
    Генириуем ник для пользователей без ников
    "Фамиля Имя" -> "familiya-imya"
    :param users:
    :type users: dict
    :rtype: dict
    """
    users_copy = copy(users)
    user_nicknames = set(u['nickname'] for u in users if 'nickname' in u)
    for user in [u for u in users_copy if 'nickname' not in u]:
        first = force_text(user['name']['first'])
        last = force_text(user['name']['last'])
        nickname = nickname_candidate = slugify(
            '{first} {last}'.format(first=first, last=last)
        )
        i = 1
        while nickname in user_nicknames:
            nickname = nickname_candidate + str(i)
            i += 1
        user_nicknames.add(nickname)

        user['nickname'] = nickname
    return users_copy


class BaseImporter(object):
    def __init__(self, meta_connection, main_connection, org_id, passwd_length=PASSWORD_LENGTH):
        self.org_id = org_id
        self.meta_connection = meta_connection
        self.main_connection = main_connection
        self.raw_data = None
        self.departments = []
        self.users = []
        self.exits_user_nicknames = [u['nickname'] for u in UserModel(self.main_connection).find(
            filter_data={
                'org_id': org_id,
                'is_dismissed': [True, False]
            }
        )]
        self.passwd_length = passwd_length

        if not OrganizationModel(self.main_connection).get(self.org_id):
            raise OrganizationNotFound(org_id)

        self.admin_uid = get_organization_admin_uid(self.main_connection, self.org_id)

    def load(self, data):
        """
        Загрузка данных из источника
        """
        raise NotImplementedError

    def departments_to_graph(self):
        # создаем из списка отделов граф отделов в виде списка смежности
        adjacency_list = defaultdict(set)
        for dep in self.departments:
            # если у отдела нет родителя, то укажем ему как родителя
            # фейковый объект "головной отдел"
            parent_id = dep.get('parent', ROOT_DEPARTMENT)

            adjacency_list[parent_id].add(dep['id'])
            adjacency_list[dep['id']].add(parent_id)

        return adjacency_list

    def validate_departments(self):
        if not self.departments:
            return

        user_ids = set(u['id'] for u in self.users)
        head_ids = set(d['head_id'] for d in self.departments if d.get('head_id'))

        unknown_head_ids = head_ids - user_ids
        if unknown_head_ids:
            raise InvalidHeadId(unknown_head_ids)

        dep_ids = [d['id'] for d in self.departments]

        dep_id, count = Counter(dep_ids).most_common(1)[0]
        # есть id отдела встретившийся более одного раза
        if count > 1:
            raise DuplicateDepartmentId(dep_id)

        # указан не существующий родительский отдел
        parent_ids = set(d['parent'] for d in self.departments if d.get('parent'))
        invalid_parent_ids = parent_ids - set(dep_ids)
        if invalid_parent_ids:
            raise InvalidParentId(invalid_parent_ids)

        adj_list = self.departments_to_graph()
        if ROOT_DEPARTMENT not in adj_list:
            raise NoToplevelDepartments

        if len(dep_ids) < 3:
            # в нашем случае только граф из 3 и более отделов
            # может содержать более 1 компонеты свзяности
            return

        def bfs(adj_list):
            # обход графа
            queue = list()
            visited = set()

            v = list(list(adj_list.values())[0])[0]
            queue.append(v)
            # обход в ширину
            while queue:
                v = queue.pop()
                for w in adj_list[v]:
                    if w in visited:
                        continue
                    queue.append(w)
                    visited.add(w)
            return visited

        visited = bfs(adj_list)
        # если из 1 узла не обошли всех, то
        # число компонет свзяности графа отделов > 1
        if set(dep_ids) - visited:
            raise ToManyConnectedComponent

    def validate_users(self):
        dep_ids = set(d['id'] for d in self.departments if 'id' in d)
        user_dep_ids = set(u['department'] for u in self.users if 'department' in u)
        user_nicknames = [u['nickname'] for u in self.users if 'nickname' in u]
        user_nicknames = self.exits_user_nicknames + user_nicknames
        unknown_dep_ids = user_dep_ids - dep_ids
        if unknown_dep_ids:
            raise UnknownUserDepartment(unknown_dep_ids.pop())

        if user_nicknames:
            nickname, count = Counter(user_nicknames).most_common(1)[0]
            # есть nickname  встретившийся более одного раза
            if count > 1:
                raise DuplicateUserNickname(nickname)

    def _generate_nickname_if_need(self):
        """
        Генириуем ник для пользователей без ников
        "Фамиля Имя" -> "familiya-imya"
        """
        self.users = generate_nickname_if_need(self.users)

    def _create_departments(self):
        """
        Создание отделов
        """
        with self.main_connection.begin_nested():
            for department in self.departments:
                new_department = DepartmentModel(self.main_connection).create(
                    org_id=self.org_id,
                    name={'ru': department['name'], 'en': ''},
                    external_id=department['id'],
                )
                department['internal_id'] = new_department['id']

            departments_dict = {d['id']: d for d in self.departments}
            for department in self.departments:
                internal_dep_id = department['internal_id']
                parent = department.get('parent')
                if parent:
                    parent_id = departments_dict[parent]['internal_id']
                else:
                    parent_id = ROOT_DEPARTMENT_ID

                DepartmentModel(self.main_connection).update_one(
                    org_id=self.org_id,
                    id=internal_dep_id,
                    data={
                        'parent_id  ': parent_id,
                    }
                )

    def _fill_department_head_id(self):
        """
        Проставляем руководителей отделов
        """
        users_dict = {u['id']: u for u in self.users}
        for department in self.departments:
            head_id = department.get('head_id')
            if not head_id:
                continue
            internal_head_user_id = users_dict[head_id]['internal_id']
            internal_department_id = department['internal_id']
            with self.main_connection.begin_nested():
                DepartmentModel(self.main_connection).update_one(
                    id=internal_department_id,
                    org_id=self.org_id,
                    data={
                        'head_id': internal_head_user_id,
                    }
                )

    def _create_users(self):
        """
        Создаем пользователей
        """
        organization = OrganizationModel(self.main_connection).get(self.org_id)

        departments_dict = {d['id']: d for d in self.departments}
        for user in self.users:
            department_id = departments_dict.get(user.get('department'), {}).get('internal_id', ROOT_DEPARTMENT_ID)
            password = generate_password(self.passwd_length)
            name = {k: dict(ru=v) for k, v in user['name'].items()}
            user_data = {
                'org_id': self.org_id,
                'department_id': department_id,
                'nickname': user['nickname'],
                'name': name,
                'gender': user.get('gender', 'male'),
                'contacts':  user.get('contacts'),
                'position': user.get('position'),
                'about': user.get('about'),
                'birthday': user.get('birthday'),
                'external_id': user['id'],
            }
            with self.main_connection.begin_nested():
                new_user = create_user(
                    meta_connection=self.meta_connection,
                    main_connection=self.main_connection,
                    org_id=organization['id'],
                    user_data=user_data,
                    nickname=user['nickname'],
                    password=password,
                )
            user['password'] = password
            user['internal_id'] = new_user['user']['id']

    def start_import(self):
        """
        Пошаговый импорт
        """
        self.validate_users()
        self.validate_departments()

        self._generate_nickname_if_need()
        self._create_departments()
        self._create_users()
        self._fill_department_head_id()
        return {
            'users': self.users,
            'departments': self.departments,
        }
