# -*- coding: utf-8 -*-
import json
from intranet.yandex_directory.src.yandex_directory.common import schemas

from intranet.yandex_directory.src.yandex_directory.common.utils import (
    validate_data_by_schema,
    parse_birth_date
)

from intranet.yandex_directory.src.yandex_directory.core.importer import BaseImporter
from intranet.yandex_directory.src.yandex_directory.core.importer.exception import NotValidJson
from intranet.yandex_directory.src.yandex_directory.core.models.user import (
    CONTACT_TYPES,
    GENDER_TYPES
)

# https://github.yandex-team.ru/gist/art/a4823aae79878dc864cf
SCHEMA = {
    "$schema": "http://json-schema.org/draft-04/schema#",
    "title": "Import file",
    "type": "object",
    "properties": {
        "departments": {
            "type": "array",
            "items": {
                "type": "object",
                "properties": {
                    "head_id": schemas.STRING_OR_INTEGER,
                    "parent": schemas.STRING_OR_INTEGER,
                    "id": schemas.STRING_OR_INTEGER,
                    "name": schemas.STRING,
                },
                "required": [
                    "id",
                    "name",
                ]
            }
        },
        "users": {
            "type": "array",
            "items": {
                "type": "object",
                "properties": {
                    "id": schemas.STRING_OR_INTEGER,
                    "name": {
                        "type": "object",
                        "properties": {
                            "first": schemas.STRING,
                            "last": schemas.STRING,
                            "middle": schemas.STRING,
                        },
                        "required": [
                            "first",
                            "last"
                        ]
                    },
                    "nickname": schemas.STRING,
                    "position": schemas.STRING,
                    "about": schemas.STRING,
                    "gender": {
                        "enum": GENDER_TYPES
                    },
                    "birthday": schemas.DATE,
                    "contacts": {
                        "type": "array",
                        "items": {
                            "type": "object",
                            "properties": {
                                "type": {
                                    "enum": CONTACT_TYPES
                                },
                                "value": schemas.STRING,
                                "main": schemas.BOOLEAN,
                            },
                            "required": [
                                "type",
                                "value",
                            ]
                        }
                    }
                },
                "required": [
                    "id",
                    "name",
                ]
            }

        }
    },
}


class JsonImporter(BaseImporter):
    def load(self, data):
        """
        Загрузка фалов для импорта из строки, файла или словаря
        :param data:
        :type data: file|str|dict
        :return:
        """
        if isinstance(data, str):
            try:
                self.data = json.loads(data)
            except:
                raise ValueError('Not json')
        elif isinstance(data, dict):
            self.data = data
        elif isinstance(data, file):
            try:
                self.data = json.load(data)
            except ValueError as e:
                raise ValueError('Not json file')
        else:
            raise ValueError('Invalid param data, type {}'.format(type(data)))

        errors = validate_data_by_schema(self.data, SCHEMA)
        if errors:
            raise NotValidJson(errors)

        self.departments = self.data.get('departments', [])
        self.users = self.data.get('users', [])

        self.normalize_data()

    def normalize_data(self):
        """
        Нормализуем данные по отделам и пользователям
        Приводим все id к строке, а даты к datetime.date
        :return:
        """
        for dep in self.departments:
            dep['id'] = str(dep['id'])
            if 'parent' in dep:
                dep['parent'] = str(dep['parent'])
            if 'head_id' in dep:
                dep['head_id'] = str(dep['head_id'])

        for user in self.users:
            user['id'] = str(user['id'])
            if 'parent' in user:
                user['parent'] = str(user['parent'])
            if 'department' in user:
                user['department'] = str(user['department'])
            if 'birthday' in user:
                user['birthday'] = parse_birth_date(user['birthday'])
