# -*- coding: utf-8 -*-
from requests.exceptions import HTTPError
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory.common.utils import (
    get_user_id_from_passport_by_login,
)
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_meta_connection,
)
from intranet.yandex_directory.src.yandex_directory.core.mail_migration.exception import ExistingPassportAccountError
from intranet.yandex_directory.src.yandex_directory.core.models import (
    UserModel,
)
from intranet.yandex_directory.src.yandex_directory.core.task_queue import (
    Task,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    create_user,
    build_email,
    ROOT_DEPARTMENT_ID,
)
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.core.actions import action_user_add


class CreateAccountTask(Task):
    """
    Создание нового аккаунта в паспорте и в директории
    """
    need_rollback = True
    sensitive_params = ['password', 'old_password']
    singleton_params = ['nickname', 'org_id']

    def do(self, nickname, password, org_id, email, old_password, first_name, last_name):
        """
        email и old_password не используются непосредственно при создании аккаунта,
        но будут нужны в задаче по созданию сборщика почты.
        Если не передать это тут, в задаче по созданию сборщиков придётся парсить миграционный файл
        и матчить его с успешно завершёнными задачами по созданию аккаунта.
        А так они удобно будут хранится в params каждой задачи на создание аккаунта
        :param nickname: новый никнейм в коннекте
        :param password: новый пароль в коннекте
        :param org_id: org_id
        :param email: ящик, с которого будем собирать почту
        :param old_password: пароль от ящика, с которого будем собирать почту
        :param first_name: имя
        :param last_name: фамилия
        :return:
        """
        with get_meta_connection(for_write=True) as meta_connection, \
             log.fields(org_id=org_id, nickname=nickname):
            # Проверяем, есть ли в метаданных account_id.
            # Если есть - успешно завершаемся, возвращая account_id
            metadata = self.get_metadata()
            if metadata and metadata.get('account_id'):
                log.debug('Account already exists')
                return metadata['account_id']
            else:
                try:
                    log.debug('Starting account creating')
                    account = create_account(
                        meta_connection,
                        self.main_connection,
                        nickname,
                        password,
                        org_id,
                        first_name,
                        last_name,
                    )
                    action_user_add(
                        self.main_connection,
                        org_id=org_id,
                        author_id=None,
                        object_value=account['user']
                    )
                    log.debug('Account created')
                    return account['user']['id']
                except Exception:
                    log.trace().error('Account creation error')
                    raise

    def rollback(self, **kwargs):
        # Удаляем аккаунты из паспорта и директории, если они создались
        directory_user_id = get_user_id_from_directory(
            self.main_connection,
            kwargs['nickname'],
            kwargs['org_id'],
        )
        if directory_user_id:
            UserModel(self.main_connection).dismiss(
                kwargs['org_id'],
                user_id=directory_user_id,
                author_id=None,
            )
        else:
            passport_user_id = get_user_id_from_passport(
                self.main_connection,
                kwargs['nickname'],
                kwargs['org_id'],
            )
            if passport_user_id:
                delete_account_from_passport(passport_user_id)

    def on_dependency_success(self):
        """
        При завершении таска-зависимости со статусом Success разлочиваем таск
        """
        self.resume()

    def on_dependency_fail(self, dependency_info):
        """
        При завершении таска-зависимости со статусом, отличным от Success
        отменяем задачу
        """
        self.cancel()


class SetAccountConsistencyTask(Task):
    """
    Устранение неконсистентности между аккаунтами в паспорте и директории.
    Эту задачу надо запускать перед задачей по созданию аккаунта,
    если нет уверенности, что аккаунт находится в одинаковом состоянии
    (существует или не существует) в паспорте и директории
    """
    def do(self, nickname, org_id):
        # Пытаемся найти аккаунт в Паспорте и в Директории
        passport_user_id = get_user_id_from_passport(
            self.main_connection,
            nickname,
            org_id,
        )
        directory_user_id = get_user_id_from_directory(
            self.main_connection,
            nickname,
            org_id,
        )
        # Аккаунта нет ни в паспорте, ни в директории
        # Ничего не делаем, успешно завершаемся
        if not passport_user_id and not directory_user_id:
            return
        # Аккаунт есть в паспорте, но его нет в директории
        # Завершаемся с ошибкой, так как аккаунтом в паспорте уже могли мользоваться
        elif not directory_user_id:
            raise ExistingPassportAccountError()
        # Аккаунт есть в директории, но его нет в паспорте
        # Увольняем пользователя в директории
        elif not passport_user_id:
            UserModel(self.main_connection).dismiss(
                org_id=org_id,
                user_id=directory_user_id,
                author_id=None,
                skip_passport=True,
            )
        # Аккаунт есть и в дериктории, и в паспорте
        # Если id совпадают - всё ок. Записываем в метадата родительского таска, что аккаунт уже существует.
        # Если нет - завершаемся с ошибкой
        else:
            if int(passport_user_id) != int(directory_user_id):
                raise AssertionError(
                    'Account has different ids in Passport ({passport_user_id}) ' \
                    'and in Directory ({directory_user_id})'.format(
                        passport_user_id=passport_user_id,
                        directory_user_id=directory_user_id,
                    )
                )
            dependent_tasks = self.get_dependents()
            for dependent in dependent_tasks:
                dependent.set_metadata(
                    dict(account_id=directory_user_id),
                )


def get_user_id_from_passport(main_connection, nickname, org_id):
    user_email = build_email(
        main_connection,
        nickname,
        org_id,
    )
    return get_user_id_from_passport_by_login(user_email)


def get_user_id_from_directory(main_connection, nickname, org_id):
    user_email = build_email(
        main_connection,
        nickname,
        org_id,
    )
    directory_user = UserModel(main_connection).filter(
        email=user_email,
        org_id=org_id,
        is_dismissed=False,
    ).fields('id').one()
    return directory_user['id'] if directory_user else None


def create_account(meta_connection, main_connection, nickname,
                   password, org_id, first_name=None, last_name=None):
    capitalized_nickname = nickname.capitalize()
    user_data = {
        'name': {
            'first': {'ru': first_name or capitalized_nickname},
            'last': {'ru': last_name or capitalized_nickname},
        },
        'gender': None,
        'department_id': ROOT_DEPARTMENT_ID,
    }
    return create_user(
        meta_connection,
        main_connection,
        org_id=org_id,
        user_data=user_data,
        nickname=nickname,
        password=password,
        ignore_login_not_available=True,
    )


def delete_account_from_passport(user_id):
    with log.fields(user_id=user_id):
        log.info('Unblocking account')
        try:
            app.passport.unblock_user(user_id)
        except HTTPError as exc:
            # Если пытаемся разблокировать пользователя, которого нет, паспорт возвращает 404
            # Игнорируем это
            if exc.response.status_code == 404:
                log.warning('Unknown UID (unblock user)')
            else:
                log.trace().error('HTTPError when unblock user')
                raise

        log.info('Deleting Passport account')
        app.passport.account_delete(user_id)
