# -*- coding: utf-8 -*-

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory.core.mail_migration.exception import (
    NoCollectorID,
    MailBoxesCreatingError,
    CollectorsCreatingError,
)
from intranet.yandex_directory.src.yandex_directory.core.mail_migration.mailbox.tasks import CreateMailBoxesTask
from intranet.yandex_directory.src.yandex_directory.core.mail_migration.metatask.tasks import (
    MetaTask,
)
from intranet.yandex_directory.src.yandex_directory.core.mail_migration.account.tasks import CreateAccountTask
from intranet.yandex_directory.src.yandex_directory.core.task_queue import (
    Task,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import unpickle
from intranet.yandex_directory.src.yandex_directory.core.yarm import (
    check_server,
    create,
)
from intranet.yandex_directory.src.yandex_directory.core.yarm.exceptions import (
    YarmDuplicateError,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import get_user_data_from_blackbox_by_uid


class CreateMailCollectorsTask(MetaTask):
    """
    Этот таск начинает выполняться, когда завершится CreateMailBoxesTask.
    Если CreateMailBoxesTask завершился с ошибкой, этот таск тоже завершается с ошибкой.
    Иначе он берет успешно завершившиеся подтаски типа CreateAccountTask
    таска CreateMailBoxesTask и для каждой создает таск CreateMailCollectorTask
    После этого уходит в режим ожидания до того, как все подтаски завершатся.
    Если все подтаски завершились со статусом, отличным от Success,
    главный таск завершается со статусом Failed.
    Иначе - со статусом Success 
    """
    main_dependency_fail_exception = MailBoxesCreatingError
    all_subtasks_fail_exception = CollectorsCreatingError
    main_dependency_cls = CreateMailBoxesTask
    main_dependency_subtask_cls = CreateAccountTask

    def _create_subtasks(self, tasks, host=None, port=None, imap=True, ssl=True, no_delete_msgs=True, sync_abook=True, mark_archive_read=True, org_id=None):
        subtask_ids = []
        for task in tasks:
            user_id = unpickle(task['result'])
            user_from_bb = get_user_data_from_blackbox_by_uid(user_id)
            if not user_from_bb or not user_from_bb.get('login'):
                raise ValueError('No user found in blackbox (user_id={user_id})'.format(user_id=user_id))
            task = CreateMailCollectorTask(self.main_connection).delay(
                user_id=user_id,
                email=task['params'].get('email'),
                password=task['params'].get('old_password'),
                user=user_from_bb['login'],
                org_id=org_id,
                host=host,
                port=port,
                imap=imap,
                ssl=ssl,
                no_delete_msgs=no_delete_msgs,
                sync_abook=sync_abook,
                mark_archive_read=mark_archive_read,
            )
            subtask_ids.append(task.task_id)
        return subtask_ids


class CreateMailCollectorTask(Task):
    """
    Создание сборщика почты
    """
    sensitive_params = ['password']

    def do(self, user_id, email, password, user, org_id, host, port, imap=True, ssl=True, no_delete_msgs=True, sync_abook=True, mark_archive_read=True):
        """
        :param user_id: id пользователя в коннекте
        :param email: ящик, с которого будем переносить почту
        :param password: пароль от этого ящика
        :param user: логин пользователя в паспорте
        :param org_id: org_id
        :param host: хост
        :param port: порт
        :param imap: использовать imap иначе pop3
        :param ssl: нужно используя SSL  (по умолчанию Да)
        :param no_delete_msgs: флаг того, что письма с удаленного сервера удалять не надо  (по умолчанию Да)
        :param sync_abook: флаг того, что для сборщика нужно выполнить процедуру сбора контактов  (по умолчанию Да)
        :param mark_archive_read: флаг того, что письма пришедшие в исходный ящик до того, как был создан сборщик, нужно при сборе пометить прочитанными (по умолчанию Да)
        :return:
        """
        with log.fields(org_id=org_id, user_id=user_id):
            log.debug('Creating mail collector')
            try:
                check_server(email, password, host, port, user_id, imap, ssl)
                response = create(
                    uid=user_id,
                    login=email,
                    password=password,
                    user=user,
                    server=host,
                    port=port,
                    imap=imap,
                    ssl=ssl,
                    no_delete_msgs=no_delete_msgs,
                    sync_abook=sync_abook,
                    mark_archive_read=mark_archive_read,
                )
                pop_id = response.get('popid')
                if pop_id:
                    with log.fields(pop_id=pop_id):
                        log.debug('Mail collector created')
                        return pop_id
                else:
                    raise NoCollectorID()
            except YarmDuplicateError:
                self.set_metadata(dict(message='Collector already exists'))
                return
            except Exception:
                log.trace().error('Collector creating error')
                raise
