# -*- coding: utf-8 -*-

from intranet.yandex_directory.src.yandex_directory.core.mail_migration.exception import (
    UnknownMetataskType,
)
from intranet.yandex_directory.src.yandex_directory.core.models import TaskModel
from intranet.yandex_directory.src.yandex_directory.core.task_queue import (
    Task,
    TASK_STATES,
)
from intranet.yandex_directory.src.yandex_directory.core.task_queue.base import TERMINATE_STATES
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import Suspend

# Зависимость, с которой таск создаётся изначально
# для CreateMailCollectorsTask - это CreateMailBoxesTask
# для WaitingForMigrationsTask - это CreateMailCollectorsTask
MAIN_DEPENDENCIES_COUNT = 1


class MetaTask(Task):
    """
    Это общий класс для двух типов тасков - CreateMailCollectorsTask и WaitingForMigrationsTask.
    Метатаск изначально создаётся с одной зависимостью.
    CreateMailCollectorsTask -  с CreateMailBoxesTask.
    WaitingForMigrationsTask - с CreateMailCollectorsTask.
    Метатаск ждёт, пока основная зависимость выполнится,
    а потом запускается сам. Берет все успешно выполненные подтаски своей основной зависимости
    и для каждого из них создаёт новый таск. Все новые таски добавляет к себе в зависимости
    и уходит в режим ожидания, пока все его новые подтаски не завершатся
    """

    # Количество попыток 1, так как это метатаск, который запускает другие таски.
    # Каждый маленький таск имееет 3 попытки.
    # В главном таске попытки не нужны, т.к. в противном случает
    # к нему в зависимости будут добавляться новые таски со старыми параметрами,
    # и потом сложно будет разобраться, какие подтаски относятся к какой попытке
    tries = 1

    # исключение, которое надо бросить, если сфейлился основной таск-зависимость
    main_dependency_fail_exception = None
    # исключение, которое надо бросить, если сфейлились все подтаски (кроме основной)
    all_subtasks_fail_exception = None
    # класс основного таска-зависимости
    main_dependency_cls = None
    # класс подтаска основного таска-зависимости
    main_dependency_subtask_cls = None

    def __init__(self,
                 main_connection,
                 queue=None,
                 task_id=None,
                 parent_task_id=None,
                 depends_on=None,
                 priority=None,
                 **kwargs):
        if not self.main_dependency_fail_exception \
                or not self.all_subtasks_fail_exception \
                or not self.main_dependency_cls:
            raise UnknownMetataskType()
        super(MetaTask, self).__init__(
            main_connection,
            queue,
            task_id,
            parent_task_id,
            depends_on,
            priority,
            **kwargs
        )

    def _is_ready(self):
        # Метод решает, запустить ли выполнение основного таска,
        # завершиться или подождать ещё
        task = TaskModel(self.main_connection).get(self.task_id)
        dependencies_count = task['dependencies_count']

        # есть ли подтаски (кроме основной зависимости)
        has_subtasks = dependencies_count > MAIN_DEPENDENCIES_COUNT
        # все подтаски кроме основной зависимости завершились с ошибкой
        all_subtasks_are_failed = task['failed_dependencies_count'] >= \
                                  dependencies_count - MAIN_DEPENDENCIES_COUNT

        ready = False
        if dependencies_count > 0:
            if dependencies_count == MAIN_DEPENDENCIES_COUNT:
                if task['failed_dependencies_count'] == dependencies_count:
                    # Сфейлился основной таск-зависимость
                    # Не начинаем выполнение главного таска, завершаемся с ошибкой
                    raise self.main_dependency_fail_exception()
            if has_subtasks:
                if all_subtasks_are_failed:
                    # все таски которых ждали - сфейлились
                    # поэтому главный таск завершается с ошибкой
                    raise self.all_subtasks_fail_exception()

            # дополнительно убедимся, что все подтаски завершились
            if not self.is_all_dependencies_completed():
                raise Suspend()
            if has_subtasks:
                # главный таск завершается успешно
                ready = True
        else:
            # таск взят в работу раньше времени,
            # dependencies_count == 0
            # ожидаем
            raise Suspend()
        return ready

    def do(self, **kwargs):
        if self._is_ready():
            return

        # Сначала таск должен дождаться выполнения основной зависимости,
        # а затем насоздавать ещё подзадач для создания ящиков или настройки сборщиков
        dependencies = TaskModel(self.main_connection).get_dependencies(
            self.task_id,
            self.main_dependency_cls,
        )
        if not dependencies:
            raise RuntimeError('Unable to find {}'.format(
                self.main_dependency_cls.get_task_name().split('.')[-1])
            )

        main_dependency_task = dependencies[0]
        if main_dependency_task.get('state') not in TERMINATE_STATES:
            raise Suspend()

        # Сначала, проверим, может мы уже насоздавали подтаски
        deps_count = len(
            TaskModel(self.main_connection)
            .get_dependencies(self.task_id)
        )

        # Если зависимости всего одна, это значит, что мы зависим только от
        # от таска по созданию учёток и надо сделать подзадачи по
        # настройке сборщиков
        if deps_count == 1:
            # Берём из базы все успешные поддаски основного подтаска, на которые надо завести
            # подтаски для этого таска
            main_dependency_subtasks = TaskModel(self.main_connection).get_dependencies(
                task_id=main_dependency_task['id'],
                state=TASK_STATES.success,
                task_name=self.main_dependency_subtask_cls,
            )

            subtasks = self._create_subtasks(main_dependency_subtasks, **kwargs)
            if subtasks:
                self.wait_for(subtasks)

    def _create_subtasks(self, tasks, **kwargs):
        raise NotImplementedError()
