# -*- coding: utf-8 -*-

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory.core.mail_migration.collector.tasks import (
    CreateMailCollectorsTask,
    CreateMailCollectorTask,
)
from intranet.yandex_directory.src.yandex_directory.core.mail_migration.mailbox.tasks import CreateMailBoxesTask
from intranet.yandex_directory.src.yandex_directory.core.models import TaskModel
from intranet.yandex_directory.src.yandex_directory.core.task_queue import (
    Task,
    TASK_STATES,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import unpickle
from intranet.yandex_directory.src.yandex_directory.core.mail_migration.exception import (
    MailMigrationError,
    CollectorDeletingError,
)
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import Suspend
from intranet.yandex_directory.src.yandex_directory.core.yarm import delete


class MailMigrationTask(Task):
    """
    Запуск миграции ящиков

    Создаёт задачу по созданию ящиков.
    Создаёт задачу по настройке сборщиков, которая зависит от задачи по сборке ящиков
    Создаёт задачу по сборке почты, которая зависит от задача по настройке сборщиков
    Добавляет все три задачи к себе в зависимости и ожидает, пока они выполнятся
    Завершается успешно, если все три задачи выполнились успешно
    """
    def _is_ready(self):
        ready = False
        task = TaskModel(self.main_connection).get(self.task_id)
        dependencies_count = task['dependencies_count']
        # Если есть таски, которых надо ждать
        if dependencies_count > 0:
            # и все они завершились
            if self.is_all_dependencies_completed():
                # и среди них есть фейлы
                if task['failed_dependencies_count'] > 0:
                    # главный таск завершается с ошибкой
                    raise MailMigrationError()
                # фелов нет, главный таск завершается успешно
                else:
                    ready = True
            else:
                # ещё не все зависимости выполнились, ждём дальше
                raise Suspend()
        # зависимостей ещё нет, можно выполнять главный таск
        return ready

    def do(self, migration_file_id, org_id, host, port, imap=True, ssl=True, no_delete_msgs=True, sync_abook=True, mark_archive_read=True):
        if self._is_ready():
            return

        with log.fields(org_id=org_id):
            log.debug('Start mail migration')
            create_mailbox_task = CreateMailBoxesTask(
                self.main_connection,
                parent_task_id=self.task_id,
            ).delay(
                migration_file_id=migration_file_id,
                org_id=org_id,
            )

            create_collector_task = CreateMailCollectorsTask(
                self.main_connection,
                depends_on=[create_mailbox_task.task_id],
                parent_task_id=self.task_id,
            ).delay(
                org_id=org_id,
                host=host,
                port=port,
                imap=imap,
                ssl=ssl,
                no_delete_msgs=no_delete_msgs,
                sync_abook=sync_abook,
                mark_archive_read=mark_archive_read,
            )

            # Временно отключим таски по ожидания миграции DIR-4706
            # waiting_for_migrations_task = WaitingForMigrationsTask(
            #     self.main_connection,
            #     depends_on=[create_collector_task.task_id],
            #     parent_task_id=self.task_id,
            # ).delay()

            self.wait_for(
                create_mailbox_task,
                create_collector_task,
                # waiting_for_migrations_task,
            )


class DeleteCollectorTask(Task):
    """
    Удаление сборщика почты
    """
    def do(self, uid, popid, org_id):
        log.debug('Starting deletion of collector')
        delete(uid, popid)


class DeleteCollectorsTask(Task):
    """
    Удаление сборщиков почты запускается когда успешно или с ошибкой завершается таск MailMigrationTask
    """

    tries = 1

    def _is_ready(self):
        ready = False
        task = TaskModel(self.main_connection).get(self.task_id)
        dependencies_count = task['dependencies_count']

        # один из подтасков кроме основной зависимости завершился с ошибкой
        any_subtask_is_failed = task['failed_dependencies_count'] > 1

        # если подтасков нет - ожидаем
        if dependencies_count == 0:
            raise Suspend()
        # если есть один подтаск (основной, MailMigrationTask)
        # и если он ещё не завершился - ожидаем
        elif dependencies_count == 1:
            if not self.is_all_dependencies_completed():
                raise Suspend()
        # если есть подтаски кроме основного
        else:
            # если все они завершились
            if self.is_all_dependencies_completed():
                #  и хотя бы один с ошибкой
                if any_subtask_is_failed:
                    # главный таск завершается с ошибкой
                    raise CollectorDeletingError()
                else:
                    # если все завершились успешно - главный таск завершается успешно
                    ready = True
            # если есть незавершённые подтаски - ожидаем
            else:
                raise Suspend()
        return ready

    def do(self, org_id):
        if self._is_ready():
            return

        migration_task = TaskModel(self.main_connection).get_dependencies(
            self.task_id,
            MailMigrationTask,
        )
        if not migration_task:
            raise RuntimeError('Unable to find {}'.format(
                MailMigrationTask.get_task_name().split('.')[-1])
            )
        collectors_task = TaskModel(self.main_connection).get_dependencies(
            migration_task[0]['id'],
            CreateMailCollectorsTask,
        )
        if not collectors_task:
            raise RuntimeError('Unable to find {}'.format(
                CreateMailCollectorsTask.get_task_name().split('.')[-1])
            )

        # Если подзадач на удаление сборщиков ещё нет, то надо их завести
        dependencies = TaskModel(self.main_connection).get_dependencies(
            self.task_id,
            DeleteCollectorTask,
        )
        if not dependencies:
            collectors_subtasks = TaskModel(self.main_connection).get_dependencies(
                task_id=collectors_task[0]['id'],
                task_name=CreateMailCollectorTask.get_task_name(),
                state=TASK_STATES.success,
            )
            subtasks = []
            for task in collectors_subtasks:
                # в каждом таске CreateMailCollectorTask в поле result есть popid,
                # а в params - user_id. Эти параметры нужны для удаления сборщика почты.
                # unpickle делаем для того, чтобы получить popid как число, так как воркер кладёт
                # его в result как unicode
                delete_collector_task = DeleteCollectorTask(self.main_connection).delay(
                    uid=task.get('params').get('user_id'),
                    popid=unpickle(task.get('result')),
                    org_id=org_id,
                )
                subtasks.append(delete_collector_task)
            # если не нашлось ни одного сборщика - таск успешно завершается
            if not subtasks:
                return
            # иначе создаются подзадачи на удаление сборщиков и таск засыпает
            self.wait_for(subtasks)
