# -*- coding: utf-8 -*-
from intranet.yandex_directory.src.yandex_directory.core.mail_migration.account.tasks import (
    CreateAccountTask,
    SetAccountConsistencyTask,
)
from intranet.yandex_directory.src.yandex_directory.core.models import TaskModel
from intranet.yandex_directory.src.yandex_directory.core.task_queue import Task, TASK_STATES
from intranet.yandex_directory.src.yandex_directory.core.task_queue.base import TERMINATE_STATES


class MAIL_MIGRATION_STATES:
    in_progress = 'in-progress'
    success = 'success'
    failed = 'failed'
    pending = 'pending'
    # Во время таска миграции должны создаваться подтаски. Если таск завершился, но у него нет подтасков, то что-то пошло не так.
    # Обычно это бывает, когда подтаски сломались во время создания. Раньше выводился pending, но по факту здесь мы уже ничего не ожидаем.
    undefined = 'undefined'


def create_account(main_connection, nickname, password, org_id, email, old_password,
                   first_name=None, last_name=None):
    """
    Создание аккаунта
    """
    nickname = nickname.lower()
    set_consistency_task = SetAccountConsistencyTask(main_connection).delay(
        nickname=nickname,
        org_id=org_id,
    )
    return CreateAccountTask(
        main_connection,
        depends_on=[set_consistency_task.task_id]
    ).delay(
        nickname=nickname,
        password=password,
        org_id=org_id,
        email=email,
        old_password=old_password,
        first_name=first_name,
        last_name=last_name,
    ).task_id


def start_migration(main_connection, migration_file_id, org_id, host, port, imap, ssl, no_delete_msgs, sync_abook, mark_archive_read,):
    """
    Запуск миграции почты.
    После завершения задачи по миграции запускается задача
    по удалению созданных миграторов
    :return: id задачи на миграцию ящиков
    :rtype: uuid
    """
    # этот импорт тут, чтобы избежать циклический зависимостей
    from intranet.yandex_directory.src.yandex_directory.core.mail_migration import (
        MailMigrationTask,
        DeleteCollectorsTask,
    )
    mail_migration_task = MailMigrationTask(main_connection).delay(
        migration_file_id=str(migration_file_id),
        org_id=org_id,
        host=host,
        port=port,
        imap=imap,
        ssl=ssl,
        no_delete_msgs=no_delete_msgs,
        sync_abook=sync_abook,
        mark_archive_read=mark_archive_read,
    )
    # Временно отключим удаление сборщиков DIR-4706
    # DeleteCollectorsTask(
    #     main_connection,
    #     depends_on=[mail_migration_task.task_id],
    # ).delay()
    return mail_migration_task.task_id


class MailMigration(object):

    def __init__(self, main_connection, mail_migration_task_id=None, org_id=None):
        if not mail_migration_task_id and not org_id:
            raise RuntimeError('At least one parameter must be passed (mail_migration_task_id or org_id)')
        self.main_connection = main_connection
        self.migration_task_id = mail_migration_task_id
        self.org_id = org_id

    def get_migration_progress(self):
        # Получение прогресса переноса почты
        migration_task = self.get_migration_task()
        if not migration_task:
            return None
        migration_subtasks = TaskModel(self.main_connection).get_dependencies(
            task_id=migration_task['id'],
        )

        if not migration_subtasks:
            if migration_task['state'] in TERMINATE_STATES:
                return [
                    {'stage': 'accounts-creating', 'state': MAIL_MIGRATION_STATES.undefined},
                    {'stage': 'collectors-creating', 'state': MAIL_MIGRATION_STATES.undefined},
                ]
            else:
                return [
                    {'stage': 'accounts-creating', 'state': MAIL_MIGRATION_STATES.pending},
                    {'stage': 'collectors-creating', 'state': MAIL_MIGRATION_STATES.pending},
                ]

        create_mailboxes = None
        create_collectors = None
        # waiting_for_migrations = None

        # этот импорт тут, чтобы избежать циклических зависимостей
        from intranet.yandex_directory.src.yandex_directory.core.mail_migration import (
            CreateMailBoxesTask,
            CreateMailCollectorsTask,
            WaitingForMigrationsTask,
        )
        for subtask in migration_subtasks:
            task_name = subtask['task_name']
            if task_name == CreateMailBoxesTask.get_task_name():
                create_mailboxes = subtask
            elif task_name == CreateMailCollectorsTask.get_task_name():
                create_collectors = subtask
            # elif task_name == WaitingForMigrationsTask.get_task_name():
            #     waiting_for_migrations = subtask

        tasks = [
            create_mailboxes,
            create_collectors,
            # waiting_for_migrations,
        ]

        migration_progress = []
        stages = [
            'accounts-creating',
            'collectors-creating',
            # 'mail-collecting',
        ]
        for stage, task in zip(stages, tasks):
            migration_progress.append(
                {
                    'state': self._match_states(task, stage),
                    'stage': stage,
                }
            )

        # последний элемент списка - таск WaitingForMigrationsTask.
        # добавим туда процент перенесенных писем

        # пока закомментируем DIR-4706
        # migration_progress[-1]['progress'] = self.calculate_progress(waiting_for_migrations['id'])

        return migration_progress

    def get_migration_task(self):
        from intranet.yandex_directory.src.yandex_directory.core.mail_migration import MailMigrationTask

        if self.migration_task_id:
            return TaskModel(self.main_connection).get(id=self.migration_task_id)
        else:
            return TaskModel(self.main_connection).find(
                filter_data={
                    'params__contains': {'org_id': self.org_id},
                    'task_name': MailMigrationTask.get_task_name(),
                },
                order_by=['-created_at'],
                one=True,
            )

    def _match_states(self, task, stage):
        # переводит статус таска в статус миграции для выдачи на фронт

        if stage == 'accounts-creating' and task is None:
            # Может так случится, что таск про заведение аккаунтов
            # успешно выполнился, и был удалён по крону
            # в этом случае мы всё равно должны считать его успешным.
            # Подробности в DIR-7096.
            return MAIL_MIGRATION_STATES.success

        task_state = task['state']
        if task_state not in TERMINATE_STATES:
            # если у таска ещё нет подтасков кроме основного и статус не терминальный - статус pending
            if len(TaskModel(self.main_connection).get_dependencies(task['id'])) <= 1:
                return MAIL_MIGRATION_STATES.pending
            return MAIL_MIGRATION_STATES.in_progress
        elif task_state == TASK_STATES.success:
            return MAIL_MIGRATION_STATES.success
        else:
            return MAIL_MIGRATION_STATES.failed

    def calculate_progress(self, waiting_for_migrations_task_id):
        # берет все подтаски WaitingForMigrationTask таска WaitingForMigrationsTask
        # и считает, какой процент писем по домену в целом уже перенесён
        messages, collected, errors = 0, 0, 0

        waiting_for_migrations_subtasks = Task(self.main_connection)\
            .get_dependencies(task_id=waiting_for_migrations_task_id)

        for task in waiting_for_migrations_subtasks:
            metadata = task.get_metadata() or {}
            folders = metadata.get('folders', [])
            for folder in folders:
                messages += float(folder.get('messages', 0))
                collected += float(folder.get('collected', 0))
                errors += float(folder.get('errors', 0))
        if messages == 0:
            return 0
        else:
            return (collected + errors) * 100 / messages
