# -*- coding: utf-8 -*-

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log


__all__ = ['send', 'same_domain_send', 'dummy_send']

from intranet.yandex_directory.src.yandex_directory.common.db import get_meta_connection


def send(main_connection, campaign_slug, org_id, to_email, mail_args):
    """
    Отправляем письмо (асинхронно через свою очередь)
    :param main_connection:
    :param campaign_slug: компания в рассыляторе
    :param org_id: ид организации для которой идет отправка письма
    :param to_email: получатель
    :param mail_args: данные для вставки в шаблона письма
    :type mail_args: dict
    """
    from .tasks import SendEmailTask

    with log.name_and_fields('mailer', org_id=org_id, email=to_email):
        log.info('Creating task to send email')

        SendEmailTask(main_connection).delay(
            org_id=org_id,
            campaign_slug=campaign_slug,
            to_email=to_email,
            mail_args=mail_args,
        )


def same_domain_send(main_connection, campaign_slug, org_id, to_email, mail_args):
    """
    Отправка письма  (асинхронно через свою очередь) только на ящики с доменов организации (для testing)
    :param main_connection:
    :param campaign_slug: компания в рассыляторе
    :param org_id: ид организации для которой идет отправка письма
    :param to_email: получатель
    :param mail_args: данные для вставки в шаблона письма
    :type mail_args: dict
    """
    # решено было проверять для защиты от случайной утечки данных при тестировании
    if not _is_email_from_organization_domain(main_connection, org_id, to_email):
        with log.fields(to_email=to_email):
            log.warning('Disallow send mail')
            return
    send(
        main_connection,
        campaign_slug,
        org_id,
        to_email,
        mail_args,
    )


def dummy_send(main_connection, campaign_slug, org_id, to_email, mail_args):
    """
    Для использования в autotest.
    Никуда ни чего не отправляет.
    :param main_connection:
    :param campaign_slug: компания в рассыляторе
    :param org_id: ид организации для которой идет отправка письма
    :param from_name: имя отправителя
    :param from_email: адрес отправителя
    :param to_email: получатель
    :param subject: тема письма
    :param mail_args: данные для вставки в шаблона письма
    :type mail_args: dict
    """
    with log.fields(to_email=to_email):
        log.debug('Fake sending email')


def _is_email_from_organization_domain(main_connection, org_id, email):
    """
    Проверим что домен почты относится организации
    :param main_connection: connect к main базе
    :param email: ящик получателя
    :type email: str
    :param org_id: ид организации
    :type org_id: int
    :return: True-ящик с домена организации
    :rtype: bool
    """
    from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
        get_domains_from_db_or_domenator,
        DomainFilter,
    )

    # получим доменную часть из адреса
    domain_part = email.split('@', 1)[-1]

    filter_data = {
        'org_id': org_id,
        'name': domain_part,
    }
    with get_meta_connection() as meta_connection:
        return bool(get_domains_from_db_or_domenator(
            meta_connection=meta_connection,
            domain_filter=DomainFilter(**filter_data),
            main_connection=main_connection,
            one=True,
        ))



