# -*- coding: utf-8 -*-
"""
Клиент для Рассылятора
https://github.yandex-team.ru/sendr/sendr/tree/master/docs
https://sender.yandex-team.ru/
https://wiki.yandex-team.ru/users/lavrinenko/sender/
"""

import json

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory import app
from functools import wraps
from retrying import retry

from intranet.yandex_directory.src.yandex_directory.common.utils import url_join
from intranet.yandex_directory.src.yandex_directory.core.utils.retry import retry_http_errors

from intranet.yandex_directory.src.yandex_directory.core.mailer.sender.exceptions import SenderError
from intranet.yandex_directory.src.yandex_directory.common import http_client
from . import non_prod_envs


def skip_for_autotests(func):
    """ Ничего не делаем с рассылкой писем в момент запуска автотестов и в стрессовом окружении"""
    @wraps(func)
    def wrapper(*args, **kwargs):
        if not app.config['SENDER_BASE_URL'] and app.config['ENVIRONMENT'] == 'autotests':
            return True
        if app.config['ENVIRONMENT'] == 'stress':
            return True
        return func(*args, **kwargs)
    return wrapper


@skip_for_autotests
@retry(wait_exponential_multiplier=250,
       wait_exponential_max=2000,
       stop_max_attempt_number=3,
       retry_on_exception=retry_http_errors('sender'))
def _make_request(method, handle, get_params, post_params):
    base_url = app.config['SENDER_BASE_URL']
    url = url_join(base_url, handle, query_params=get_params)
    auth = (app.config['SENDER_USER'], '')
    # timeout в 10 секунд по совету разработчиков рассылятора
    response = http_client.request(method, url, data=post_params, auth=auth, timeout=10)

    fields = dict(
        method=method,
        url=url,
        data=post_params,
        response=response.content,
        response_code=response.status_code,
    )
    with log.fields(**fields):
        log_func = log.warning if response.status_code >= 400 else log.debug
        log_func('Sender request')

    response.raise_for_status()
    result_json = response.json()

    if result_json['result']['status'] == 'ERROR':
        raise SenderError(result_json['result']['message'])

    return result_json


def transactional_send(campaign_slug, to_email, mail_args):
    """
    Оправляем письмо в рамках транзакционной рассылки
    :param campaign_slug: Код расслылки
    :param to_email:  Адрес получателя
    :param mail_args: Дополнительные параметры письма
    """
    handle = '/api/0/{account_slug}/transactional/{campaign_slug}/send'.format(
        account_slug=app.config['SENDER_ACCOUNT_SLUG'],
        campaign_slug=campaign_slug,
    )

    headers, to_email = non_prod_envs.override(headers={}, original_email=to_email)

    get_params = {'to_email': to_email.encode('utf-8')}
    post_params = {
        'args': json.dumps(mail_args),
    }

    if headers:
        post_params['headers'] = json.dumps(headers)

    return _make_request('post', handle=handle, get_params=get_params, post_params=post_params)
