# -*- coding: utf-8 -*-
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log, default_log

from intranet.yandex_directory.src.yandex_directory.core.mailer.sender import transactional_send
from intranet.yandex_directory.src.yandex_directory.core.task_queue import Task
from intranet.yandex_directory.src.yandex_directory.core.models.organization import organization_type, OrganizationModel
from intranet.yandex_directory.src.yandex_directory.core.models import UserModel
from intranet.yandex_directory.src.yandex_directory import app


class SendEmailTask(Task):
    singleton = True
    default_priority = 50
    sensitive_params = ['password', 'invite_link']

    def do(self, campaign_slug, to_email, mail_args, org_id, **kwargs):
        """
        Отправляем письмо

        @param campaign_slug: компания в рассыляторе
        @param to_email: имя отправителя
        @param mail_args: данные для вставки в шаблона письма
        """
        from intranet.yandex_directory.src.yandex_directory.common.db import (
            get_meta_connection, get_main_connection, get_shard
        )
        with log.name_and_fields('mailer',
                                 campaign_slug=campaign_slug,
                                 to_email=to_email,
                                 org_id=org_id):
            log.info('We are about to send email for %s', to_email)
        from intranet.yandex_directory.src.settings import SENDER_TO_CLOUD_NOTIFY_MAPPING
        with get_meta_connection() as meta_connection:
            shard = get_shard(meta_connection, org_id)
            with get_main_connection(shard, for_write=True) as main_connection:
                organization = OrganizationModel(main_connection).filter(id=org_id).one()
                if (organization['organization_type'] in organization_type.cloud_types and
                    campaign_slug in SENDER_TO_CLOUD_NOTIFY_MAPPING):
                    template_name = SENDER_TO_CLOUD_NOTIFY_MAPPING[campaign_slug]
                    if template_name == '__disabled__':
                        log.info('Skipping sending mail because this template is disabled')
                        return
                    with log.name_and_fields('mailer', template_name=template_name):
                        log.info('Trying to send cloud email')
                    user = UserModel(main_connection).filter(org_id=org_id, email=to_email).one()
                    if not user or not user['cloud_uid']:
                        with log.name_and_fields('mailer', org_id=org_id, user_login=user['login'] if user else None):
                            log.info('User has no cloud_uid, retrying task')

                        metadata = self.get_metadata() or {}
                        count_tries = metadata.get('count_tries', 0)
                        if count_tries <= 10:
                            default_log.info('No cloud uid for user yet, defer task')
                            self.update_metadata(count_tries=count_tries + 1)
                            self.defer(countdown=60)
                        else:
                            raise RuntimeError('No cloud uid for user yet')

                    app.cloud_notifier.notify(
                        organization=organization,
                        template_name=template_name,
                        cloud_uid=user['cloud_uid'],
                        params=mail_args,
                    )
                    with log.name_and_fields('mailer',
                                             template_name=template_name,
                                             user_id=user['login'],
                                             user_cloud_uid=user['cloud_uid']):
                        log.info('Email via notify was sent')
                else:
                    with log.name_and_fields('mailer',
                                             campaign_slug=campaign_slug,
                                             to_email=to_email,
                                             org_id=org_id):
                        log.debug('Sending email')
                        transactional_send(
                            campaign_slug=campaign_slug,
                            to_email=to_email,
                            mail_args=mail_args
                        )
                        log.debug('Email sent')


class SendEmailToAllTask(Task):
    singleton = True
    default_priority = 50

    def do(self, **kwargs):
        from intranet.yandex_directory.src.yandex_directory.core.mailer.utils import send_email_to_all
        from intranet.yandex_directory.src.yandex_directory.common.db import (
            get_meta_connection, get_main_connection, get_shard
        )
        from intranet.yandex_directory.src.settings import SENDER_TO_CLOUD_NOTIFY_MAPPING

        org_id = kwargs['org_id']
        campaign_slug = kwargs['campaign_slug']

        with get_meta_connection() as meta_connection:
            shard = get_shard(meta_connection, org_id)
            with get_main_connection(shard, for_write=True) as main_connection:
                organization = OrganizationModel(main_connection).filter(id=org_id).one()
                if (organization['organization_type'] in organization_type.cloud_types and
                    campaign_slug in SENDER_TO_CLOUD_NOTIFY_MAPPING):
                    template_name = SENDER_TO_CLOUD_NOTIFY_MAPPING[campaign_slug]
                    if template_name == '__disabled__':
                        log.info('Skipping sending mail because this template is disabled')
                        return
                    users = UserModel(main_connection).find(
                        filter_data={'org_id': org_id},
                        fields=['id', 'cloud_uid'],
                    )
                    for user in users:
                        cloud_uid = user['cloud_uid']
                        if not cloud_uid:
                            # если у кого-то в организации нет cloud_uid, он не получит письмо
                            # вообще такого не должно быть, но лучше подложим соломки
                            log.error('No cloud uid for person with id %s', user['id'])
                            continue
                        app.cloud_notifier.notify(
                            organization=organization,
                            template_name=template_name,
                            cloud_uid=user['cloud_uid'],
                            params=kwargs,
                        )
                else:
                    send_email_to_all(
                        meta_connection,
                        main_connection,
                        **kwargs
                    )
