# -*- coding: utf-8 -*-

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory import app

from intranet.yandex_directory.src.yandex_directory.common.utils import get_user_data_from_blackbox_by_uid
from intranet.yandex_directory.src.yandex_directory.core import mailer
from intranet.yandex_directory.src.yandex_directory.core.models import (
    UserModel,
    UserMetaModel,
    OrganizationModel,
    OrganizationServiceModel,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    get_master_domain,
    build_email,
    lang_for_notification,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    UserNotFoundError, MasterDomainNotFound,
)


def _get_admin_emails(meta_connection, main_connection, org_id, domain=None):
    emails = []
    # внутренние админы
    inner_admins = UserModel(main_connection).get_organization_admins(org_id)
    for user in inner_admins:
        email = build_email(
            main_connection,
            user['nickname'],
            org_id,
            user_id=user['id'],
            org_domain=domain,
        )
        emails.append(
            (user['id'], email)
        )

    # внешние админы
    outer_admins = UserMetaModel(meta_connection).get_outer_admins(org_id=org_id)
    for user in outer_admins:
        info = get_user_data_from_blackbox_by_uid(user['id'])
        if not info:
            with log.fields(uid=user['id']):
                log.warning('Not sending email to outer admin. No outer admin in passport')
            continue
        email = info.get('default_email')
        if not email:
            log.warning('Not sending email to outer admin. "default_email" is empty.')
            continue
        emails.append(
            (user['id'], email)
        )
    return emails


def _send_email(
        meta_connection,
        main_connection,
        org_id,
        uid,
        email,
        domain,
        campaign_slug,
        **additional_mail_args
):
    """
    Отправка письма пользователю в организации
    """

    with log.name_and_fields('mailer', org_id=org_id, email=email):
        log.info('Preparing to send an email')

        org_info = OrganizationModel(main_connection).get(org_id, fields=['language', 'tld'])

        tld = org_info['tld']
        if tld not in app.config['PORTAL_TLDS']:
            tld = app.config['DEFAULT_TLD']

        lang = lang_for_notification(meta_connection, main_connection, uid, org_id)

        mail_args = {
            'lang': lang,
            'tld': tld,
            'domain': domain,
        }
        mail_args.update(additional_mail_args)
        with log.fields(lang=lang, campaign_slug=campaign_slug, tld=tld):
            try:
                mailer.send(main_connection, campaign_slug, org_id, email, mail_args)
                log.debug('Email for admin was sent')
            except Exception:
                log.trace().error('Error sending email')


def send_email_to_all_async(main_connection, org_id, domain, campaign_slug, **kwargs):
    from intranet.yandex_directory.src.yandex_directory.core.mailer.tasks import SendEmailToAllTask

    SendEmailToAllTask(main_connection).delay(
        org_id=org_id,
        domain=domain,
        campaign_slug=campaign_slug,
        **kwargs
    )


def send_email_to_all(meta_connection, main_connection, org_id, domain, campaign_slug, **kwargs):
    """
    Отправляем письма всем сотрудниками и внешним админам организации
    """
    with log.name_and_fields('mailer', org_id=org_id):
        emails = []

        # все сотрудники
        filter_data = {
            'org_id': org_id,
        }
        users = UserModel(main_connection).find(
            filter_data=filter_data,
            fields=['nickname', 'id'],
        )
        for user in users:
            email = build_email(main_connection, user['nickname'], org_id, user_id=user['id'])
            emails.append(
                (user['id'], email)
            )

        # внешние админы
        outer_admins = UserMetaModel(meta_connection).get_outer_admins(org_id)
        for user in outer_admins:
            info = get_user_data_from_blackbox_by_uid(user['id'])
            if not info:
                with log.fields(uid=user['id']):
                    log.warning('Not sending email to outer admin. No outer admin in passport')
                continue
            email = info.get('default_email')
            if not email:
                log.warning('Not sending email to outer admin. "default_email" is empty.')
                continue
            emails.append(
                (user['id'], email)
            )

        for uid, email in emails:
            with log.fields(uid=uid, email=email):
                try:
                    _send_email(
                        meta_connection,
                        main_connection,
                        org_id,
                        uid,
                        email,
                        domain,
                        campaign_slug,
                        **kwargs
                    )
                except UserNotFoundError:
                    log.warning('Unable to send email, user was not found')
                except Exception:
                    log.trace().error('Some error during send email')


def send_email_to_admins(meta_connection, main_connection, org_id, campaign_slug, domain=None, **kwargs):
    """
    Отправка письма всем администраторам организации
    """
    if domain is None:
        # Если домен не известен, то попробуем достать его из базы
        domain = get_master_domain(
            main_connection,
            org_id,
            # У некоторых организаций подтверждённого домена может и не быть, и это нормально
            domain_is_required=False,
        )

    with log.name_and_fields('mailer', org_id=org_id):
        emails = _get_admin_emails(
            meta_connection,
            main_connection,
            org_id,
            domain=domain,
        )
        for uid, email in emails:
            with log.fields(uid=uid, email=email):
                try:
                    _send_email(
                        meta_connection,
                        main_connection,
                        org_id,
                        uid,
                        email,
                        domain,
                        campaign_slug,
                        **kwargs
                    )
                except UserNotFoundError:
                    log.trace().error('Unable to send email, admin was not found')


def send_admin_after_create_organization_email(
        meta_connection,
        main_connection,
        org_id,
        uid,
        admin_nickname,
):
    """
    Отправка письма администратору после создания новой организации
    """
    from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
        get_domains_from_db_or_domenator,
        DomainFilter,
    )
    domain_info = get_domains_from_db_or_domenator(
        meta_connection=meta_connection,
        domain_filter=DomainFilter(org_id=org_id),
        main_connection=main_connection,
        one=True
    )
    if not domain_info:
        raise MasterDomainNotFound(org_id=org_id)

    domain = domain_info['name']

    email = build_email(main_connection, admin_nickname, org_id)
    _send_email(
        meta_connection,
        main_connection,
        org_id,
        uid,
        email,
        domain,
        app.config['SENDER_CAMPAIGN_SLUG']['AFTER_CREATE_ORGANIZATION_EMAIL'],
        admin=admin_nickname,
    )


# def send_welcome_email(
#         meta_connection,
#         main_connection,
#         org_id,
#         uid=None,
#         to_email=None,
# ):
#     """
#     Отправляем приветственное письмо.
#     Если uid не задан, то отправим письма всем сотрудникам организации.
#
#     :param meta_connection:
#     :param main_connection:
#     :param org_id: организация
#     :param uid: конкретный пользователь
#     :param to_email: переопределяет email, на который будет отправлено письмо. Используется для отладки.
#     :param subject: переопределяет subject письма.
#     :return:
#     """
#     with log.name_and_fields('mailer', org_id=org_id):
#         filter_data = {
#             'org_id': org_id,
#         }
#         if uid:
#             filter_data['id'] = uid
#
#         users = UserModel(main_connection).find(
#             filter_data=filter_data,
#             fields=['nickname'],
#         )
#         org_info = OrganizationModel(main_connection).get(org_id, fields=['language', 'tld'])
#
#         tld = org_info['tld']
#         if tld not in app.config['PORTAL_TLDS']:
#             tld = app.config['DEFAULT_TLD']
#
#         if to_email:
#             emails = [(org_info['language'], to_email)]
#         else:
#             emails = []
#             for user in users:
#                 login = build_email(
#                     main_connection,
#                     user['nickname'],
#                     org_id,
#                     user_id=user['id']
#                 )
#                 info = get_user_data_from_blackbox_by_uid(user['id'])
#                 if not info:
#                     with log.fields(login=login):
#                         log.warning('Cant get info from blackbox.')
#                     continue
#                 lang = lang_for_notification(meta_connection, main_connection, user['id'], org_id)
#                 emails.append(
#                     (lang, login)
#                 )
#
#         for lang, email in emails:
#             mail_args = {
#                 'lang': lang,
#                 'tld': tld,
#             }
#             campaign_slug = app.config['SENDER_CAMPAIGN_SLUG']['WELCOME_EMAIL']
#             with log.fields(email=email, campaign_slug=campaign_slug, lang=lang, tld=tld):
#                 try:
#                     mailer.send(main_connection, campaign_slug, org_id, email, mail_args)
#                     log.debug('Welcome email was sent.')
#                 except Exception:
#                     log.trace().error('Error send welcome email. Send error.')


def send_change_password_email(main_connection, to_email, password, login, org_id):
    """
    Отправляем письмо с временным паролем и просьбой установить новый пароль
    """
    org_info = OrganizationModel(main_connection).get(org_id, fields=['language', 'tld'])
    mail_args = {
        'lang': org_info['language'],
        'tld': org_info['tld'],
        'password': password,
        'login': login,
    }
    campaign_slug = app.config['SENDER_CAMPAIGN_SLUG']['CHANGE_PASSWORD_EMAIL']
    mailer.send(
        main_connection=main_connection,
        campaign_slug=campaign_slug,
        org_id=org_id,
        to_email=to_email,
        mail_args=mail_args,
    )


def send_invite_email(main_connection, to_email, invite_link, org_id, org_name, lang, tld, campaign_slug=None):
    """
    Отправляем письмо с приглашением вступить в организацию.
    """
    mail_args = {
        'lang': lang,
        'org_name': org_name,
        'invite_link': invite_link,
        'tld': tld,
    }
    campaign_slug = campaign_slug or app.config['SENDER_CAMPAIGN_SLUG']['INVITE_USER']
    with log.fields(email=to_email, campaign_slug=campaign_slug, lang=lang):
        mailer.send(main_connection, campaign_slug, org_id, to_email, mail_args)


def send_request_access_email_to_responsible_for_service(
        main_connection,
        org_id,
        service_slug,
        uid,
        resource_ids,
):
    organization_service = OrganizationServiceModel(main_connection).get_by_slug(
        org_id=org_id,
        service_slug=service_slug,
        fields=['responsible_id'],
    )
    email = UserModel(main_connection).get(
        user_id=organization_service['responsible_id'],
        org_id=org_id,
        fields=['email'],
    )['email']
    organization = OrganizationModel(main_connection).get(org_id, ['language', 'tld'])

    mail_args = {
        'lang': organization['language'],
        'tld': organization['tld'],
        'uid': uid,
        'resource_ids': resource_ids,
    }

    campaign_slug = app.config['SENDER_CAMPAIGN_SLUG']['REQUEST_ACCESS_TO_RESOURCE']

    with log.fields(email=email, campaign_slug=campaign_slug, lang=mail_args['lang']):
        mailer.send(main_connection, campaign_slug, org_id, email, mail_args)


def send_confirm_access_email_to_user(
        main_connection,
        org_id,
        email,
        lang,
        resource_id
):
    organization = OrganizationModel(main_connection).get(org_id, ['tld'])

    mail_args = {
        'lang': lang,
        'tld': organization['tld'],
        'resource_id': resource_id,
    }

    campaign_slug = app.config['SENDER_CAMPAIGN_SLUG']['CONFIRM_ACCESS_TO_RESOURCE']

    with log.fields(email=email, campaign_slug=campaign_slug, lang=mail_args['lang']):
        mailer.send(main_connection, campaign_slug, org_id, email, mail_args)


def send_deny_access_email_to_user(
        main_connection,
        org_id,
        email,
        lang,
        resource_id
):
    organization = OrganizationModel(main_connection).get(org_id, ['tld'])

    mail_args = {
        'lang': lang,
        'tld': organization['tld'],
        'resource_id': resource_id,
    }

    campaign_slug = app.config['SENDER_CAMPAIGN_SLUG']['CONFIRM_ACCESS_TO_RESOURCE']

    with log.fields(email=email, campaign_slug=campaign_slug, lang=mail_args['lang']):
        mailer.send(main_connection, campaign_slug, org_id, email, mail_args)


def send_resource_unbind_email(main_connection, service_slug, org_id, email, resource_id, lang=None):
    if service_slug == 'metrika':
        send_metrika_resource_unbind_email(main_connection, org_id, email, resource_id, lang)


def send_metrika_resource_unbind_email(main_connection, org_id, email, resource_id, lang=None):
    lang = lang or 'ru'

    organization = OrganizationModel(main_connection).get(org_id, ['tld'])

    mail_args = {
        'lang': lang,
        'tld': organization['tld'],
        'counter_id': resource_id,
    }

    campaign_slug = app.config['SENDER_CAMPAIGN_SLUG']['METRIKA_RESOURCE_UNBIND']

    with log.fields(email=email, campaign_slug=campaign_slug, lang=mail_args['lang']):
        mailer.send(main_connection, campaign_slug, org_id, email, mail_args)


def _send_tracker_disable_notification_to_admins(main_connection, org_id, campaign_slug):
    org_info = OrganizationModel(main_connection).get(org_id, fields=['language', 'tld', 'name'])
    mail_args = {
        'lang': org_info['language'],
        'tld': org_info['tld'],
        'org_name': org_info['name']
    }

    admins = OrganizationModel(main_connection).get_admins(org_id)
    for admin in admins:
        email = admin['email']
        with log.fields(email=email, campaign_slug=campaign_slug, lang=mail_args['lang']):
            mailer.send(main_connection, campaign_slug, org_id, email, mail_args)


def send_first_tracker_disable_warning_emails(main_connection, org_id):
    campaign_slug = app.config['SENDER_CAMPAIGN_SLUG']['TRACKER_DISABLE_NOTIFICATION_1']
    _send_tracker_disable_notification_to_admins(main_connection, org_id, campaign_slug)


def send_second_tracker_disable_warning_emails(main_connection, org_id):
    campaign_slug = app.config['SENDER_CAMPAIGN_SLUG']['TRACKER_DISABLE_NOTIFICATION_2']
    _send_tracker_disable_notification_to_admins(main_connection, org_id, campaign_slug)


def send_third_tracker_disable_warning_emails(main_connection, org_id):
    campaign_slug = app.config['SENDER_CAMPAIGN_SLUG']['TRACKER_DISABLE_NOTIFICATION_3']
    _send_tracker_disable_notification_to_admins(main_connection, org_id, campaign_slug)
