# -*- coding: utf-8 -*-

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.exceptions import UserNotFoundError
from intranet.yandex_directory.src.yandex_directory.common.utils import get_user_data_from_blackbox_by_uid, make_simple_strings
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.core.models import (
    OrganizationAccessRestoreModel,
    OrganizationModel,
)


from . import (
    send_email_to_admins,
    _get_admin_emails,
    _send_email,
)



def send_some_try_restore(meta_connection, main_connection, restore_id):
    """
    Отправляем о попытке восстановить доступ к организации
    """
    with log.fields(restore_id=restore_id):
        restore = OrganizationAccessRestoreModel(meta_connection).get(restore_id)
        if not restore:
            return

        campaign_slug = app.config['SENDER_CAMPAIGN_SLUG']['ACCESS_RESTORE_TRY_RESTORE']
        organization_name = make_simple_strings(
            OrganizationModel(main_connection).get(restore['org_id'], fields=['name'])
        )['name']
        send_email_to_admins(
            meta_connection,
            main_connection,
            restore['org_id'],
            campaign_slug,
            ip=restore['ip'],
            date=restore['created_at'].isoformat(),
            organization_name=organization_name,
            domain=restore['domain'],
        )

def send_organization_new_owner(meta_connection, main_connection, restore_id):
    """
    Отправляем письмо о смене владельца организации
    """
    with log.fields(restore_id=restore_id):
        restore = OrganizationAccessRestoreModel(meta_connection).get(restore_id)
        if not restore:
            return

        campaign_slug = app.config['SENDER_CAMPAIGN_SLUG']['ACCESS_RESTORE_SOME_RESTORE']
        org_id = restore['org_id']
        old_admin_uid = restore['old_admin_uid']
        domain = restore['domain']
        organization_name = make_simple_strings(
            OrganizationModel(main_connection).get(restore['org_id'], fields=['name'])
        )['name']
        with log.name_and_fields('mailer', org_id=org_id):
            emails = _get_admin_emails(
                meta_connection,
                main_connection,
                org_id,
                domain=domain,
            )

            info = get_user_data_from_blackbox_by_uid(old_admin_uid)
            if info and 'default_email' in info:
                email = info['default_email']
                emails.append(
                    (old_admin_uid, email)
                )

            for uid, email in set(emails):
                # новый владелец уже админ
                if uid == restore['new_admin_uid']:
                    continue

                with log.fields(uid=uid, email=email):
                    try:
                        _send_email(
                            meta_connection,
                            main_connection,
                            org_id,
                            uid,
                            email,
                            restore['domain'],
                            campaign_slug,
                            ip=restore['ip'],
                            date=restore['created_at'].isoformat(),
                            organization_name=organization_name,
                        )
                    except UserNotFoundError:
                        log.trace().error('Unable to send email, user was not found')


def send_take_organization(meta_connection, main_connection, restore_id):
    """
    Отправляем письмо о получении владения
    """
    with log.fields(restore_id=restore_id):
        restore = OrganizationAccessRestoreModel(meta_connection).get(restore_id)
        if not restore:
            return

        campaign_slug = app.config['SENDER_CAMPAIGN_SLUG']['ACCESS_RESTORE_SUCCESS_RESTORE']
        org_id = restore['org_id']
        new_admin_uid = restore['new_admin_uid']
        email = None
        with log.name_and_fields('mailer', org_id=org_id):

            info = get_user_data_from_blackbox_by_uid(new_admin_uid)
            if info and 'default_email' in info:
                email = info['default_email']

            if not email:
                return

            organization_name = make_simple_strings(
                OrganizationModel(main_connection).get(restore['org_id'], fields=['name'])
            )['name']

            with log.fields(uid=new_admin_uid, email=email):
                try:
                    _send_email(
                        meta_connection,
                        main_connection,
                        org_id,
                        new_admin_uid,
                        email,
                        restore['domain'],
                        campaign_slug,
                        ip=restore['ip'],
                        date=restore['created_at'].isoformat(),
                        organization_name=organization_name,
                    )
                except UserNotFoundError:
                    log.trace().error('Unable to send email, user was not found')
