# -*- coding: utf-8 -*-
from intranet.yandex_directory.src.yandex_directory.core.models.group import GROUP_TYPE_GENERIC
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory import bigml
from intranet.yandex_directory.src.yandex_directory.core.models import (
    MaillistCheckModel,
    UserModel,
    GroupModel,
    DepartmentModel,
    OrganizationModel,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    walk_department,
    walk_group,
)
from intranet.yandex_directory.src.yandex_directory.core.task_queue import (
    Task,
)
from intranet.yandex_directory.src.yandex_directory.bigml import BigmlError
from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationRevisionCounterModel
from intranet.yandex_directory.src.yandex_directory.common.db import get_meta_connection
from intranet.yandex_directory.src.yandex_directory.core.features import is_feature_enabled, DISABLE_MAILLISTS_CHECK


class MaillistsCheckTask(Task):
    default_priority = 25
    """
    Сверяем рассылки в директории и в BigML и, если есть проблемы, то запускаем синхронизацию рассылок
    """
    def do(self, org_id, last_revision=None):
        # last_revision оставлен для обратной совместимости, можно будет потом выпилить
        with log.name_and_fields('maillist_monitoring', org_id=org_id):
            if not OrganizationModel(self.main_connection).exists(org_id):
                log.info('Organization was removed.')
                return

            with get_meta_connection() as meta_connection:
                if is_feature_enabled(meta_connection, org_id, DISABLE_MAILLISTS_CHECK):
                    log.info('Maillists check disabled')
                    return

            last_revision = OrganizationRevisionCounterModel(self.main_connection).get(org_id)['revision']

            try:
                try:
                    problems = check_organization_mail_lists(self.main_connection, org_id)
                except BigmlError as e:
                    if e.error_code == 404:
                        problems = '404 error code'
                    else:
                        raise

                updated_data = {
                    'revision': last_revision,
                    'ml_is_ok': not bool(problems),
                    'problems': problems or None,
                }
                MaillistCheckModel(self.main_connection).insert_or_update(org_id, updated_data)
            except Exception:
                log.trace().error('maillist_monitoring failed')
                raise

        if problems:
            with log.name_and_fields('bigml-sync', org_id=org_id):
                log.info('Syncing maillists for organization')
                bigml.sync_organization(org_id)


def check_organization_mail_lists(main_connection, org_id):
    department_problems = check_departments(main_connection, org_id)
    group_problems = check_groups(main_connection, org_id)
    return department_problems + group_problems


def check_departments(main_connection, org_id):
    log.info('Check departments maillists')
    all_departments = DepartmentModel(main_connection).filter(org_id=org_id).fields('uid', 'label').all()

    problem = ''
    for dep in all_departments:
        label = dep['label']
        uid = dep['uid']
        department_id = dep['id']
        with log.name_and_fields('maillist_monitoring', department_label=label, department_uid=uid):
            if label:
                if not uid:
                    problem += 'Department {0}@ has not uid\n'.format(label)
                else:
                    all_users = set(
                        UserModel(main_connection)
                        .filter(
                            org_id=org_id,
                            recursive_department_id=department_id,
                        )
                        .distinct()
                        .scalar('id')
                    )
                    child_items = set()

                    def save_uid(obj):
                        # Сохраним uid вложенной сущности
                        child_items.add(obj['uid'])

                    walk_department(
                        main_connection,
                        org_id,
                        department_id,
                        on_department=save_uid,
                        fields=['uid'],
                    )

                    directory_subscribers = all_users | child_items
                    directory_subscribers.discard(None)
                    ml_subscribers = set(bigml.get_maillist_subscribers(uid))
                    ml_subscribers.discard(uid)

                    only_in_directory = directory_subscribers - ml_subscribers
                    only_in_ml = ml_subscribers - directory_subscribers

                    if only_in_directory or only_in_ml:
                        problem += 'Department ({0}@, {1}) has problems\n'.format(label, uid)
                        if only_in_directory:
                            problem += 'This users are not subscribed: {0}\n'.format(
                                comma_separated(sorted(only_in_directory))
                            )
                        if only_in_ml:
                            problem += 'This users should not be subscribed: {0}\n'.format(
                                comma_separated(sorted(only_in_ml))
                            )
    return problem


def check_groups(main_connection, org_id):
    log.info('Check groups maillists')
    all_groups = GroupModel(main_connection).filter(
        org_id=org_id,
        type=GROUP_TYPE_GENERIC,
    ).fields('uid', 'label', 'members.*').all()
    problem = ''
    for group in all_groups:
        label = group['label']
        uid = group['uid']
        with log.name_and_fields('maillist_monitoring', group_label=label, group_uid=uid):
            if label:
                if not uid:
                    problem += 'Group {0}@ has not uid\n'.format(label)
                else:
                    all_users = set(
                        UserModel(main_connection)
                        .filter(
                            org_id=org_id,
                            recursive_group_id=group['id'],
                        )
                        .distinct()
                        .scalar('id')
                    )
                    child_items = set()

                    def save_uid(obj):
                        # Сохраним uid вложенной сущности
                        child_items.add(obj['uid'])

                    walk_group(
                        main_connection,
                        org_id,
                        group['id'],
                        on_department=save_uid,
                        on_group=save_uid,
                        department_fields=['uid'],
                    )

                    directory_subscribers = all_users | child_items
                    directory_subscribers.discard(None)
                    ml_subscribers = set(bigml.get_maillist_subscribers(uid))
                    ml_subscribers.discard(uid)

                    only_in_directory = directory_subscribers - ml_subscribers
                    only_in_ml = ml_subscribers - directory_subscribers

                    if only_in_directory or only_in_ml:
                        problem += 'Group ({0}@, {1}) has problems\n'.format(label, uid)
                        if only_in_directory:
                            problem += 'This users are not subscribed: {0}\n'.format(
                                comma_separated(sorted(only_in_directory))
                            )
                        if only_in_ml:
                            problem += 'This users should not be subscribed: {0}\n'.format(
                                comma_separated(sorted(only_in_ml))
                            )
    return problem


def comma_separated(lst):
    return ', '.join(
        map(str, lst)
    )


# Утилиты для тестирования из manage shell консоли

bad_org_ids = []

def recheck():
    """Перепроверить организации в текущем шарде.
       Список организаций с разъехавшимися рассылками окажется в bad_org_ids.
       Сменить шард можно командой use_shard(5)
    """
    rows = main_connection.execute('select org_id from maillist_checks where ml_is_ok = False order by org_id').fetchall()
    good = 0
    bad = 0
    import tqdm

    for org_id, in tqdm.tqdm(rows):
        if not check_organization_mail_lists(main_connection, org_id):
            good += 1
        else:
            bad += 1
            bad_org_ids.append(org_id)
    return good, bad

# А вот так можно проверить рассылки одной организации
# check_organization_mail_lists(main_connection, 76122)

# Если нужно посмотреть структуру отделов, то можно сделать
# from intranet.yandex_directory.src.yandex_directory.core.utils import print_departments
# print_departments(main_connection, 76122)
