# -*- coding: utf-8 -*-
from dateutil.relativedelta import relativedelta

from intranet.yandex_directory.src.yandex_directory.common.models.base import BaseModel
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    utcnow,
    to_punycode,
    generate_id,
)
from intranet.yandex_directory.src.yandex_directory.core.models.domain import (
    to_lowercase,
)


TTL_DAYS = 3


class RestoreTypes(object):
    in_progress = 'in_progress'
    success = 'success'
    failed = 'failed'
    expired = 'expired'
    invalid_answers = 'invalid_answers'

    valid_types = [
        in_progress,
        failed,
        success,
    ]


class OrganizationAccessRestoreModel(BaseModel):
    db_alias = 'meta'
    table = 'organization_access_restore'
    primary_key = 'id'
    json_fields = ['control_answers']

    all_fields = [
        'id',
        'org_id',
        'domain',
        'new_admin_uid',
        'old_admin_uid',
        'created_at',
        'state',
        'ip',
        'control_answers',
        'restore_task_id',
        'expires_at',
    ]

    def create(self,
               org_id,
               domain,
               new_admin_uid,
               old_admin_uid,
               ip,
               control_answers,
               ):

        data = {
            'id': generate_id(),
            'state': 'in_progress',
            'org_id': org_id,
            'domain': to_lowercase(to_punycode(domain)),
            'new_admin_uid': new_admin_uid,
            'old_admin_uid': old_admin_uid,
            'ip': ip,
            'control_answers': control_answers,
            'expires_at': utcnow() + relativedelta(days=TTL_DAYS),
        }

        return self.insert_into_db(**data)

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('domain', encode=to_punycode) \
            ('org_id', can_be_list=True) \
            ('new_admin_uid') \
            ('old_admin_uid') \
            ('state', can_be_list=True) \
            ('id') \
            ('created_at') \
            ('ip') \
            ('expires_at') \
            ('restore_task_id')

        return distinct, filter_parts, joins, used_filters

    def get_all_valid_restores_for_uid(self, new_admin_uid):
        # валидными считаем статусы failed, success и in_progress
        return self.filter(
            new_admin_uid=new_admin_uid,
            state=RestoreTypes.valid_types,
        )

    def get_active_restore_for_uid_domain(self, new_admin_uid, domain):
        return self.filter(
            new_admin_uid=new_admin_uid,
            domain=domain,
            state=RestoreTypes.in_progress
        ).one()
