# -*- coding: utf-8 -*-
import datetime

from intranet.yandex_directory.src.yandex_directory.common.models.base import BaseModel
from intranet.yandex_directory.src.yandex_directory.core.models.organization import OrganizationModel
from intranet.yandex_directory.src.yandex_directory.common.db import Json
from intranet.yandex_directory.src.yandex_directory.core.db import queries


class ActionModel(BaseModel):
    db_alias = 'main'
    table = 'actions'
    order_by = 'id'

    all_fields = [
        'id',
        'org_id',
        'revision',
        'timestamp',
        'name',
        'author_id',
        'object_type',
        'object',
        'old_object',
    ]
    json_fields = ['object', 'old_object']

    def create(self,
               org_id,
               name,
               author_id,
               object_value,
               object_type,
               old_object=None):
        if not old_object:
            old_object = {}

        # получаем номер следующей ревизии организации
        revision = OrganizationModel(self._connection).increment_revision(org_id=org_id)

        return self.insert_into_db(
            org_id=org_id,
            name=name,
            author_id=author_id,
            object=object_value,
            object_type=object_type,
            old_object=old_object,
            revision=revision,
        )

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts, joins, used_filters = [], [], []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('id') \
            ('revision', can_be_list=True) \
            ('revision__gt') \
            ('revision__lt') \
            ('revision__between') \
            ('object_type') \
            ('name', can_be_list=True) \
            ('org_id', can_be_list=True)

        if 'object_id' in filter_data:
            value = int(filter_data.get('object_id'))

            filter_parts.append(
                self.mogrify(
                    'actions.object @> %(value)s',
                    {
                        'value': Json({'id': value})
                    }
                )
            )
            used_filters.append('object_id')

        if 'timestamp__gt' in filter_data:
            dt = filter_data.get('timestamp__gt')
            if not isinstance(dt, datetime.datetime):
                raise ValueError('timestamp__gt must be datetime.datetime not {}'.format(type(dt)))

            filter_parts.append(
                self.mogrify(
                    'actions.timestamp > %(timestamp)s',
                    {
                        'timestamp': dt.isoformat()
                    }
                )
            )
            used_filters.append('timestamp__gt')

        if 'nickname' in filter_data:
            nickname = filter_data.get('nickname')
            filter_parts.append(
                self.mogrify(
                    'actions.object @> %(nickname)s',
                    {
                        'nickname': Json({'nickname': nickname})
                    }
                )
            )
            used_filters.append('nickname')

        return distinct, filter_parts, [], used_filters

    def get_by_revision(self, revision, org_id):
        return dict(
            self._connection.execute(
                queries.GET_ACTION_BY_REVISION['query'],
                {
                    'org_id': org_id,
                    'revision': revision,
                }
            ).fetchone()
        )

    def get_max_revision(self, org_id):
        query = 'SELECT MAX(revision) from actions where org_id=%(org_id)s'
        revision = dict(
            self._connection.execute(
                query,
                {
                    'org_id': org_id
                }
            ).fetchone()
        ).get('max')
        if not revision:
            return 0
        return int(revision)


class SupportActions:
    change_organization_type = 'change_organization_type'
    change_subscription_plan = 'change_subscription_plan'
    change_service_limits = 'change_service_limits'
    change_organization_admin = 'change_organization_admin'  # передача управления
    verify_domain = 'verify_domain'
    manage_features = 'manage_features'
    manage_whitelist = 'manage_whitelist'
    change_organization_limits = 'change_organization_limits'
    block_organization = 'block_organization'
    unblock_organization = 'unblock_organization'


class SupportActionMetaModel(BaseModel):
    db_alias = 'meta'
    table = 'support_actions'
    order_by = 'id'

    all_fields = [
        'id',
        'org_id',
        'timestamp',
        'name',
        'author_id',
        'object_type',
        'object',
        'old_object',
        'comment',
    ]
    json_fields = ['object', 'old_object']

    def create(self,
               org_id,
               name,
               author_id,
               object_value,
               object_type,
               comment,
               old_object=None):
        if not old_object:
            old_object = {}

        return self.insert_into_db(
            org_id=org_id,
            name=name,
            author_id=author_id,
            object=object_value,
            object_type=object_type,
            old_object=old_object,
            comment=comment,
        )

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts, joins, used_filters = [], [], []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('id') \
            ('object_type') \
            ('name', can_be_list=True) \
            ('org_id', can_be_list=True) \
            ('author_id', can_be_list=True)

        if 'object_id' in filter_data:
            value = int(filter_data.get('object_id'))

            filter_parts.append(
                self.mogrify(
                    'actions.object @> %(value)s',
                    {
                        'value': Json({'id': value})
                    }
                )
            )
            used_filters.append('object_id')

        if 'timestamp__gt' in filter_data:
            dt = filter_data.get('timestamp__gt')
            if not isinstance(dt, datetime.datetime):
                raise ValueError('timestamp__gt must be datetime.datetime not {}'.format(type(dt)))

            filter_parts.append(
                self.mogrify(
                    'actions.timestamp > %(timestamp)s',
                    {
                        'timestamp': dt.isoformat()
                    }
                )
            )
            used_filters.append('timestamp__gt')

        return distinct, filter_parts, [], used_filters

