# -*- coding: utf-8 -*-
import json

from copy import deepcopy
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory.common.models.base import (
    BaseModel,
    set_to_none_if_no_id,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    ConstraintValidationError,
    ServiceNotFound,
    ServiceNotLicensed,
    AuthorizationError,
)

from intranet.yandex_directory.src.yandex_directory.core.db import queries
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    build_email,
    only_ids,
    flatten_paths,
)
from intranet.yandex_directory.src.yandex_directory.common.models.types import (
    ROOT_DEPARTMENT_ID,
    TYPE_DEPARTMENT,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    prepare_for_tsquery,
    to_lowercase,
    NotGiven,
    Ignore,
    make_simple_strings,
)
from intranet.yandex_directory.src.yandex_directory.core.models.user import UserModel
from intranet.yandex_directory.src.yandex_directory.core.utils.organization import (
    get_organization_language,
    get_organization_maillist_type,
)
from intranet.yandex_directory.src.yandex_directory.core.models.service import (
    OrganizationServiceModel,
    MAILLIST_SERVICE_SLUG,
)
from intranet.yandex_directory.src.yandex_directory.core.models.resource import ResourceRelationModel
from intranet.yandex_directory.src.yandex_directory import app
from functools import reduce

_head_group_name_templates = {
    'default': 'Руководитель отдела "{0}"',
    'en': 'Head of department "{0}"',
}


def get_name_for_heads_group(department_name, language):
    """Эта функция принимает локализованную строку с
    названием отдела, и возвращает локализованную
    строку с названием группы для руководителя этого отдела.
    """

    # Временный костыль, вбитый до тех пор, пока мы не откажемся
    # от переводов названий на разные языки.
    # Сделано в рамках работы над тикетом https://st.yandex-team.ru/DIR-3000

    # Для русского языка вернём все ключи, которые есть в department_name,
    # чтобы ничего не сломать.
    if language == 'ru':
        default_template = _head_group_name_templates['default']
        return dict(
            (
                key,
                _head_group_name_templates.get(key, default_template).format(value)
            )
            for key, value in list(department_name.items())
        )
    else:
        # а для остальных языков, возвращаем два ключа - ru, en
        # в обоих - англоязычное название команды

        # В качестве названия отдела сначала пробуем взять английское
        # название, и если его нет, то фаллбэчимся на русское
        value = department_name.get('en') or department_name['ru']
        en_name = _head_group_name_templates.get('en').format(value)
        return {
            'ru': en_name,
            'en': en_name,
        }


class DepartmentNotEmptyError(Exception):
    """
    При попытке удаления департамента, обнаружилось, что в нём есть активные люди или отделы
    """
    pass


class DepartmentNotFoundError(Exception):
    pass


class RootDepartmentRemovingError(Exception):
    pass


class DepartmentModel(BaseModel):
    db_alias = 'main'
    table = 'departments'
    json_fields = ('name', 'description', 'aliases')

    all_fields = [
        'org_id',
        'id',
        'parent_id',
        'name',
        'path',
        'description',
        'heads_group_id',
        'label',
        'email',
        'removed',
        'aliases',
        'members_count',
        'external_id',
        'created',
        'maillist_type',
        'uid',  # uid аккаунта в паспорте
        'name_plain',
        'description_plain',
        # не из базы
        'parents',
        'parent',
        'head',
        'is_outstaff',
        'tracker_license',
    ]
    select_related_fields = {
        'parent': 'DepartmentModel',
        'head': 'UserModel',
        'tracker_license': 'ResourceRelationModel',
    }
    prefetch_related_fields = {
        'parents': 'DepartmentModel',
        'email': 'DomainModel',
        'is_outstaff': None,
    }
    field_dependencies = {
        'parents': ['org_id', 'path'],
        'is_outstaff': ['path', 'org_id'],
        'email': ['org_id'],
        'tracker_license': ['org_id'],
    }

    def create(self,
               name,
               org_id,
               label=None,
               email=None,
               aliases=None,
               parent_id=None,
               head_id=None,
               id=None,
               description=None,
               external_id=None,
               maillist_type=None,
               uid=None, ):
        """Этот метод создает новый отдел, а вместе с ним и группу,
        в которую нужно включать руководителя отдела, если тот
        когда-либо будет назначен.
        """
        from intranet.yandex_directory.src.yandex_directory.core.models.group import (
            GroupModel,
        )
        connection = self._connection

        organization_language = get_organization_language(
            connection,
            org_id,
        )

        # добавляем специальную группу для руководителя
        heads_group = GroupModel(connection=connection).create(
            name=get_name_for_heads_group(name, organization_language),
            org_id=org_id,
            type='department_head',
            generate_action=False,
        )

        # если был задан UID руководителя, то добавляем его
        # в соответствующую группу
        if head_id is not None:
            GroupModel(connection=connection).add_member(
                org_id=org_id,
                group_id=heads_group['id'],
                member={'type': 'user', 'id': head_id}
            )
        if aliases is None:
            aliases = []

        label = to_lowercase(label)
        email = to_lowercase(email)
        description = description or {'ru': ''}
        if not maillist_type:
            maillist_type = get_organization_maillist_type(connection, org_id)
        query_kwargs = dict(
            org_id=org_id,
            parent_id=parent_id,
            label=label,
            email=email or build_email(
                connection,
                label,
                org_id=org_id,
            ),
            aliases=aliases,
            name=name,
            description=description,
            heads_group_id=heads_group['id'],
            external_id=external_id,
            maillist_type=maillist_type or 'inbox',
            uid=uid,
            name_plain=make_simple_strings(name),
            description_plain=make_simple_strings(description) or None,
        )

        if id:
            query_kwargs['id'] = id  # todo: test id
            query = queries.CREATE_DEPARTMENT_WITH_ID['query']

            # Обновим макимально известный id отдела, чтобы те отделы,
            # которые будут создаваться без указания id, гарантированно
            # имели больший id чем этот.
            self._connection.execute(
                """
                UPDATE organizations
                SET max_department_id = GREATEST(max_department_id, %(id)s)
                WHERE id = %(org_id)s
                """,
                {'org_id': org_id, 'id': id}
            )
        else:
            query = queries.CREATE_DEPARTMENT['query']
        department = self.insert(query, query_kwargs)

        # рассчитываем путь к корню организации
        # чтобы впоследствии делать быстрые выборки из базы
        path = self.rebuild_path(
            department_id=department['id'],
            org_id=org_id
        )
        department['path'] = path

        return department

    def update_one(self, id, org_id, data):
        from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel
        from intranet.yandex_directory.src.yandex_directory.core.utils.departments import is_child_department

        connection = self._connection
        groups = GroupModel(connection=connection)
        head_id = data.pop('head_id', NotGiven)

        if 'name' in data or head_id is not NotGiven:
            heads_group_id = self.get(org_id=org_id, department_id=id)['heads_group_id']

            # если изменилось название отдела, то меняем так же
            # название группы, в которую входит руководитель
            if 'name' in data:
                organization_language = get_organization_language(
                    connection,
                    org_id,
                )
                group_name = get_name_for_heads_group(
                    data['name'],
                    organization_language,
                )

                if heads_group_id:
                    groups.update_one(
                        org_id=org_id,
                        group_id=heads_group_id,
                        data={
                            'name': group_name
                        }
                    )
                data['name_plain'] = make_simple_strings(data['name'])

            if head_id is not NotGiven:
                users = groups.get_all_users(
                    org_id=org_id,
                    group_id=heads_group_id
                )

                if len(users) == 1:
                    current_head = users[0]
                else:
                    current_head = None

                # если существующий руководитель не тот, что приехал
                # в update, то смещаем его
                if current_head and head_id != current_head['id']:
                    groups.remove_member(
                        org_id=org_id,
                        group_id=heads_group_id,
                        member={'type': 'user', 'id': current_head['id']}
                    )

                # если руководитель задан, и не такой, как прежде
                # то делаем его новым руководителем
                if head_id is not None and (not current_head or head_id != current_head):
                    groups.add_member(
                        org_id=org_id,
                        group_id=heads_group_id,
                        member={'type': 'user', 'id': head_id},
                    )

        if 'parent_id' in data:
            parent_id = int(data['parent_id'])
            id = int(id)

            if parent_id == id:
                raise ConstraintValidationError(
                    'cant_move_department_to_itself',
                    'Department could not be used as parent of itself',
                )

            if is_child_department(self._connection, org_id, id, parent_id):
                raise ConstraintValidationError(
                    'cant_move_department_to_descendant',
                    ('Department with id {child_id} is a descendant of '
                     'department with id {parent_id} and '
                     'cannot be used as it\'s parent'),
                    child_id=parent_id,
                    parent_id=id,
                )
        if 'description' in data:
            data['description_plain'] = make_simple_strings(data['description'])

        if data:
            self.update(
                update_data=data,
                filter_data={
                    'id': id,
                    'org_id': org_id,
                }
            )

        if 'parent_id' in data:
            self.rebuild_path_for_self_and_descendants(
                department_id=id,
                org_id=org_id
            )

    def rebuild_path_for_self_and_descendants(self, department_id, org_id):
        descendant_ids = [
            i['id'] for i in
            self.find({
                'path__ltree_match': '*.%s.*' % department_id,
                'org_id': org_id
            })
        ]

        self_path = self.rebuild_path(
            department_id=department_id,
            org_id=org_id
        )
        for department_id in descendant_ids:
            self.rebuild_path(
                department_id=department_id,
                org_id=org_id
            )
        return self_path

    def rebuild_path(self, department_id, org_id):
        """Этот метод обновлеят значение поля `path`, которое потом
        ипользуется для авторизационных запросов. Например, "найти всех
        пользователей, департамент которых является потомком
        департамента с id=1".
        """
        path = self.build_path(department_id, org_id)
        self.update(
            update_data={
                'path': path,
            },
            filter_data={
                'id': department_id,
                'org_id': org_id,
            }
        )
        return path

    def _build_model_field(self, field):
        if field == 'tracker_license':
            return None
        return super()._build_model_field(field)

    def build_path(self, department_id, org_id):
        query = """
        WITH RECURSIVE get_tree(parent_id) AS (
            SELECT departments.parent_id FROM departments
            WHERE departments.id=%(department_id)s AND departments.org_id=%(org_id)s
            UNION ALL
            SELECT departments.parent_id FROM departments, get_tree
            WHERE departments.id = get_tree.parent_id AND departments.org_id=%(org_id)s
        )
        select * from get_tree;
        """
        parts = reversed(
            self._connection.execute(
                query,
                {
                    'department_id': department_id,
                    'org_id': org_id
                }
            ).fetchall()
        )
        return '.'.join(
            map(
                str,
                [i['parent_id'] for i in parts if i['parent_id']] + [department_id]
            )
        )

    def get_select_related_data(self, select_related):
        if not select_related:
            return [self.default_all_projection], [], []

        select_related = select_related or []
        projections = set()
        joins = []
        processors = []

        if 'parent' in select_related:
            projections.update([
                'departments.parent_id',
                'departments.org_id',
                'parent.id AS "parent.id"',
                'parent.name AS "parent.name"',
                'parent.parent_id AS "parent.parent_id"',
                'parent.removed AS "parent.removed"',
                'parent.external_id AS "parent.external_id"',
            ])
            joins.append("""
            LEFT OUTER JOIN departments as parent ON (
                departments.parent_id = parent.id AND departments.org_id = parent.org_id
            )
            """)

        if 'head' in select_related:
            projections.update([
                'departments.heads_group_id',
                'departments.org_id',
                'head.id AS "head.id"',
                'head.org_id AS "head.org_id"',
                'head.nickname AS "head.nickname"',
                'head.aliases AS "head.aliases"',
                'head.name AS "head.name"',
                'head.email AS "head.email"',
                'head.gender AS "head.gender"',
                'head.about AS "head.about"',
                'head.birthday AS "head.birthday"',
                'head.contacts AS "head.contacts"',
                'head.position AS "head.position"',
                'head.department_id AS "head.department_id"',
                'head.is_dismissed AS "head.is_dismissed"',
                'head.external_id AS "head.external_id"',
            ])
            joins.append("""
            LEFT OUTER JOIN groups as head_groups ON (
                departments.heads_group_id = head_groups.id
            AND departments.org_id = head_groups.org_id
            )
            LEFT OUTER JOIN user_group_membership as hg_membership ON (
                head_groups.id = hg_membership.group_id
            AND head_groups.org_id = hg_membership.org_id
            )
            LEFT OUTER JOIN users as head ON (
                hg_membership.user_id = head.id
            AND hg_membership.org_id = head.org_id
            )
            """)
            processors.append(
                set_to_none_if_no_id('head')
            )

        if 'tracker_license' in select_related:
            projections.update([
                'case when resource_relations.department_id is not null then True else False end as tracker_license',
            ])
            joins.append("""
            LEFT OUTER JOIN resource_relations ON (
                resource_relations.org_id = departments.org_id
                AND resource_relations.department_id = departments.id
            )
            LEFT OUTER JOIN organization_services ON (
                organization_services.resource_id = resource_relations.resource_id
            )
            LEFT OUTER JOIN services ON (
                services.id = organization_services.service_id
                AND services.slug = 'tracker'
            )
            """)

        return projections, joins, processors

    def get_suggest_filters(self, suggest):
        text = prepare_for_tsquery(suggest)
        return self.mogrify(
            "ydir.make_user_name_tsvector(coalesce(departments.name_plain, '')) || ' ' || ydir.make_user_name_tsvector(coalesce(departments.label, '')) @@ %(text)s::tsquery",
            {'text': text}
        )

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_data_removed = deepcopy(filter_data)
        if 'removed' not in filter_data_removed:
            filter_data_removed['removed'] = False

        joins = []
        filter_parts = []
        used_filters = []

        if filter_data.get('removed', False) is Ignore:  # убираем из фильтра
            del filter_data_removed['removed']

        if 'removed' in filter_data_removed:
            filter_parts.append(
                self.mogrify(
                    'departments.removed = %(removed)s',
                    {
                        'removed': filter_data_removed.get('removed')
                    }
                )
            )
            used_filters.append('removed')

        # removed - не учитываем при фильтрации внутри filter_data
        # (только внутри filter_data_removed),
        # но добавляем в used_filters, чтоб проходила проверка
        # get_filters_and_check
        if 'removed' in filter_data:
            used_filters.append('removed')

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('id', can_be_list=True) \
            ('org_id', can_be_list=True) \
            ('parent_id', can_be_list=True) \
            ('external_id') \
            ('label', can_be_list=True) \
            ('uid', can_be_list=True)

        if 'suggest' in filter_data:
            mogrify_condition = self.get_suggest_filters(
                filter_data.get('suggest')
            )
            filter_parts.append(mogrify_condition)
            used_filters.append('suggest')

        if 'path__ltree_match' in filter_data:
            filter_parts.append(
                self.mogrify(
                    'path ~ %(path__ltree_match)s',
                    {
                        'path__ltree_match': filter_data.get('path__ltree_match')
                    }
                )
            )
            used_filters.append('path__ltree_match')

        if 'alias' in filter_data:
            filter_parts.append(
                self.mogrify(
                    'departments.aliases::jsonb @> %(alias)s',
                    {
                        'alias': json.dumps(filter_data.get('alias'))
                    }
                )
            )
            used_filters.append('alias')

        if 'uid__isnull' in filter_data:
            uid__isnull = 'NULL' if bool(filter_data.get('uid__isnull')) else 'NOT NULL'
            filter_parts.append(
                self.mogrify(
                    'departments.uid IS {}'.format(uid__isnull)
                )
            )
            used_filters.append('uid__isnull')

        return distinct, filter_parts, joins, used_filters

    def prefetch_related(self, items, prefetch_related):
        if not prefetch_related or not items:
            return

        org_id = set(item['org_id'] for item in items)
        if len(org_id) != 1:
            raise RuntimeError('All departments should belong to one organization')
        org_id = org_id.pop()

        if 'parents' in prefetch_related:
            item_parent_ids = [
                list(map(int, i['path'].split('.')[:-1]))  # за исключением самого себя
                for i in items
            ]
            all_parent_ids = reduce(lambda x, y: x + y, item_parent_ids)
            parent_department_by_id = {}

            if all_parent_ids:  # todo: test me
                department_model = DepartmentModel(self._connection)
                fields = ['*', 'email']
                for department in department_model.find(
                        filter_data={
                            'id': all_parent_ids,
                            'org_id': org_id,
                        },
                        fields=fields,
                ):
                    parent_department_by_id[department['id']] = department

            for i, department in enumerate(items):
                department['parents'] = [
                    parent_department_by_id[id]
                    for id in item_parent_ids[i]
                ]

        if 'email' in prefetch_related:
            department_by_id = {}
            for department in items:
                department_by_id[department['id']] = department

            department_model = DepartmentModel(self._connection)
            filter_data = {
                'id': list(department_by_id.keys()),
                'org_id': org_id,
            }
            fields = ['label']
            for department in department_model.find(
                    filter_data=filter_data,
                    fields=fields,
            ):
                department_by_id[department['id']]['email'] = build_email(
                    self._connection,
                    department['label'],
                    org_id
                )

        if 'is_outstaff' in prefetch_related:
            for department in items:
                path = department['path']
                department['is_outstaff'] = is_outstaff(path)

    def get(self, department_id, org_id, removed=None, fields=None):

        filter_data = {
            'id': department_id,
            'org_id': org_id,
        }

        if removed is not None:
            filter_data['removed'] = removed

        response = self.find(
            filter_data=filter_data,
            fields=fields,
            limit=1
        )

        if response:
            return response[0]

    def get_by_external_id(self, org_id, id, fields=None):
        return self.find(
            filter_data={
                'org_id': org_id,
                'external_id': id,
            },
            fields=fields,
            one=True,
        )

    def remove(self, department_id, org_id, author_uid, force=False, real_remove=False):
        """
        Удаление отдела.
        1. Проверяем, что в департаменте не осталось людей
        2. Проверяем, что удаляемый департамент не является чьим-то родителем
        3. Удаляем связи с группами и ресурсами
        4. Удаляем департамент (если real_remove=True, то удаляем из базы, если нет - ставим флаг removed)
        5. События про удаление департамента (Удаляем рассылку из паспорта)

        force: bool принудительно увольняем людей
        real_remove: bool удаляем департамент по-настоящему, а не проставлением галочки removed=True
        """
        if department_id == ROOT_DEPARTMENT_ID:
            raise RootDepartmentRemovingError()

        department_model = DepartmentModel(self._connection)
        department = department_model.get(department_id=department_id, org_id=org_id)
        if not department:
            raise DepartmentNotFoundError()

        self._ensure_all_child_departments_are_removed(
            department_id=department_id,
            org_id=org_id,
            author_uid=author_uid,
            force_remove=force,
            real_remove=real_remove,
        )

        if real_remove:
            self._ensure_all_users_are_removed(department_id, org_id, author_uid)
        else:
            self._ensure_all_users_are_dismissed(department_id, org_id, author_uid, force_dismiss=force)

        self._remove_from_all_groups(department_id, org_id)

        # если включен сервис рассылок, удаляем рассылку из паспорта и у себя
        if OrganizationServiceModel(self._connection).is_service_enabled(org_id, MAILLIST_SERVICE_SLUG):
            uid = self.get(department_id, org_id, fields=['uid'])['uid']
            if uid:
                app.passport.maillist_delete(uid)
                real_remove = True

        if real_remove:
            self.delete(filter_data={'org_id': org_id, 'id': department_id}, force=True)
        else:
            self.filter(org_id=org_id, id=department_id) \
                .update(removed=True, heads_group_id=None)

        from intranet.yandex_directory.src.yandex_directory.core.actions import action_department_delete
        action_department_delete(
            self._connection,
            org_id=org_id,
            author_id=author_uid,
            object_value=department,
            old_object=department,
        )

    def _ensure_all_child_departments_are_removed(self, department_id, org_id, author_uid, force_remove, real_remove):
        """
        Проверяем что нет департаментов, входящих в данный отдел.
        При force_remove=True - удаляем все департаменты
        """
        department_model = DepartmentModel(self._connection)
        active_child_departments = department_model.find(
            filter_data=dict(
                org_id=org_id,
                parent_id=department_id,
            )
        )

        if not force_remove and active_child_departments:
            raise DepartmentNotEmptyError()

        for child_department in active_child_departments:
            self.remove(
                department_id=child_department['id'],
                org_id=org_id,
                author_uid=author_uid,
                force=True,
                real_remove=real_remove,
            )

    def _ensure_all_users_are_dismissed(self, department_id, org_id, author_uid, force_dismiss=False):
        """
        Проверяем что нет людей в данном отделе.
        При force_dismiss=True принудительно увольняем, если они есть.
        """
        filter_data = {
            'department_id': department_id,
            'org_id': org_id,
        }
        connection = self._connection

        # fields=['groups'] здесь нужен для дальнейшей передачи
        # каждого из увольняемых пользователей в dismiss метод,
        # который ожидает old_user с группами увольняемого пользователя
        users_in_department = UserModel(connection).find(
            filter_data=filter_data,
            fields=[
                '*',
                'groups.*',
            ],
        )

        if not force_dismiss and users_in_department:
            raise DepartmentNotEmptyError()

        user_model = UserModel(self._connection)
        for user in users_in_department:
            user_model.dismiss(org_id=org_id, user_id=user['id'], author_id=author_uid, old_user=user)

    def _ensure_all_users_are_removed(self, department_id, org_id, author_uid):
        """
        Проверяем что нет людей в данном отделе.
        """
        connection = self._connection
        filter_data = {
            'department_id': department_id,
            'org_id': org_id,
            'is_dismissed': Ignore,
        }

        if UserModel(connection).count(filter_data=filter_data):
            raise DepartmentNotEmptyError()

    def _remove_from_all_groups(self, department_id, org_id):
        """
        Удаляем департамент из всех групп
        """
        from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel

        member = dict(id=department_id, type=TYPE_DEPARTMENT)
        group_model = GroupModel(self._connection)
        groups = group_model.get_member_groups(org_id, member)

        for group_id in only_ids(groups):
            group_model.remove_member(
                org_id=org_id,
                group_id=group_id,
                member=member,
            )

    def update_members_count(self, department_ids, org_id):
        """
        Пересчитать количество сотрудников в отделе + подотделах
        для каждого из перечисленных отделов и всех их родителей.

        Этот метод может работать с несколькими отделами сразу.
        Это нужно в том случае, чтобы консистентно обновить счетчики,
        когда сотрудник или подотдел переносится из одной ветки в
        другую.
        """
        # Для удобства, позволим передавать в качестве первого
        # аргумента и отдельное число
        if isinstance(department_ids, int):
            department_ids = (department_ids,)

        # Получим сами отделы
        deps = [
            self.get(dep_id, org_id)
            for dep_id in department_ids
        ]
        deps = [_f for _f in deps if _f]

        # Проверим, все ли отделы были найдены
        # если нет, то надо выдать Warning.
        if len(deps) < len(department_ids):
            missing_deps = set(department_ids) - set(only_ids(deps))
            with log.fields(department_ids=list(missing_deps)):
                log.warning('Some departments wasn\'t found in database')

        paths = [
            list(map(int, dep['path'].split('.')))
            for dep in deps
        ]
        ids = flatten_paths(paths)
        # развернём список id отделов так, чтобы
        # сначала шли наиболее глубоко вложенные
        ids.reverse()

        query = """
            UPDATE departments
               SET members_count = (
                 SELECT SUM(r.count)
                   FROM (
                    SELECT COUNT(*) as count
                      FROM users
                     WHERE org_id=%(org_id)s AND department_id = %(dep_id)s AND NOT is_dismissed
                 UNION ALL
                    SELECT members_count AS count
                      FROM departments
                     WHERE org_id=%(org_id)s AND parent_id = %(dep_id)s
                 ) as r
               )
            WHERE org_id=%(org_id)s AND id=%(dep_id)s
            RETURNING members_count
        """

        for _id in ids:
            self._connection.execute(
                query, {
                    'dep_id': _id,
                    'org_id': org_id,
                }
            )

    def delete(self, filter_data=None, force=False, force_remove_all=False):
        # если не переданы данные с фильтром и org_id, то ничего
        # не удаляем
        if not filter_data or not filter_data.get('org_id'):
            return

        deleted_data = deepcopy(filter_data)
        if force:
            deleted_data['removed'] = Ignore

        super(DepartmentModel, self).delete(deleted_data, force_remove_all=force_remove_all)

    def get_or_create_outstaff(self, org_id, language='ru', outstaff_dep_label=None):
        # Создание отдела "Внешние сотрудники".
        # Пока не будем заводить рассылку для такого отдела
        from intranet.yandex_directory.src.yandex_directory.core.actions import action_department_add

        outstaff_department = self.filter(org_id=org_id, parent_id__isnull=True, id__notequal=ROOT_DEPARTMENT_ID).one()
        if outstaff_department:
            return outstaff_department

        english_oustaff_dep_name = 'Outstaff'

        if language == 'ru':
            outstaff_dep_name = {
                'ru': 'Внешние сотрудники',
                'en': english_oustaff_dep_name
            }
        else:
            outstaff_dep_name = {
                'ru': english_oustaff_dep_name,
                'en': english_oustaff_dep_name,
            }

        # создаем outstaff отдел
        outstaff_department = self.create(
            org_id=org_id,
            name=outstaff_dep_name,
            label=outstaff_dep_label,
            uid=None,
        )

        action_department_add(
            self._connection,
            org_id=org_id,
            author_id=None,
            object_value=outstaff_department,
        )

        return outstaff_department


def is_ancestor(department1, department2):
    """
    Проверяет, является ли первый департамент предком второго.

    И если первый департамент стоит по иерархии выше, и является
    предком, то возвращается True, в противном случае - False.

    Уровень вложенности может быть любой.
    """
    # последнюю составляющую откусываем, так как это id самого отдела 2
    splitted = department2['path'].split('.')[:-1]
    path = list(map(int, splitted))
    return department1['id'] in path


_ROOT_DEPARTMENT_ID_AS_STR = str(ROOT_DEPARTMENT_ID)

def is_outstaff(path):
    """Вспомогательная функция, чтобы по пути типа '1.5.7.10' определить
       является ли отдел внештатным.
    """
    first_parent_id = path.split('.', 1)[0]

    if first_parent_id == _ROOT_DEPARTMENT_ID_AS_STR:
        # Штатными считаем только отделы, относящиеся к основному корневому
        return False
    else:
        # Все остальные деревья отделов считаем внештатными
        # https://st.yandex-team.ru/DIR-6374
        return True

