# -*- coding: utf-8 -*-
from collections import defaultdict

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection,
    get_shard_numbers, get_meta_connection,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    DomainNotFound,
    MasterDomainNotFound,
    DomainNotValidatedError,
    DeleteMasterDomainError,
    BlockedDomainError,
)
from intranet.yandex_directory.src.yandex_directory.common.models.base import (
    BaseModel,
    BaseAnalyticsModel,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    delete_domain_in_passport,
    from_punycode,
    to_punycode,
    to_lowercase,
    utcnow,
    get_domain_info_from_blackbox,
    get_domain_id_from_blackbox,
    NotGiven,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    get_organization_admin_uid,
    get_master_domain,
)
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log


class domain_action:
    """
    Типы действий для доменов подтверждаемых через webmaster
    """
    add = 'add'  # домен добавлен
    delete = 'delete'  # домен удален
    verify = 'verify'  # запущено подтверждение прав


def _decode_name_from_punycode(domain):
    """Вспомогательная функция для того, чтобы перекодировать имя домена,
    только что вытащенного из БД, из punycode в unicode.

    Внимание, эта функция меняет входной словарь.
    """
    domain['name'] = from_punycode(domain['name'])
    return domain


def _encode_name_to_punycode(domain):
    """Вспомогательная функция для того, чтобы перекодировать имя домена,
    в punycode перед сохранением в БД.

    Внимание, эта функция меняет входной словарь.
    """
    if 'name' in domain:
        domain['name'] = to_punycode(domain['name'])
    return domain


class DomainModel(BaseModel):
    db_alias = 'main'
    table = 'domains'
    order_by = 'name'
    primary_key = 'name'

    all_fields = [
        'org_id',
        'name',
        'master',
        'owned',
        'display',
        'validated',
        'mx',
        'delegated',
        'via_webmaster',
        'created_at',
        'validated_at',
        'blocked_at',
    ]

    def create(self,
               name,
               org_id,
               owned=False,
               master=False,
               via_webmaster=NotGiven,
               mx=False):
        """ Добавить новый домен.

        Домен всегда создается с master == False.

        :param name: Название домена.
        :param org_id: Какой организации принадлежит домен.
        :param owned: Признак того, подтвержден ли домен.
        :param master: Является ли домен основным.
        :param via_webmaster: True, если домен необходимо подтвердить через Вебмастер и False, если через ПДД.
        :param mx: mx смотрят на нас.
        :return: dict - Словарь с созданными параметрами.
        """

        new_domain = self.insert_into_db(
            org_id=org_id,
            # DIR-1459
            # кодируем домен в punicode при сохранении данных
            name=to_lowercase(to_punycode(name)),
            master=master,
            display=master,
            owned=owned,
            via_webmaster=(
                True
                if via_webmaster is NotGiven
                else via_webmaster
            ),
            mx=mx
        )
        return _decode_name_from_punycode(new_domain)

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('name', encode=to_punycode, can_be_list=True) \
            ('org_id', can_be_list=True) \
            ('master') \
            ('owned') \
            ('display') \
            ('via_webmaster') \
            ('delegated') \
            ('mx')

        if 'registrar_id' in filter_data:
            value = filter_data.get('registrar_id')
            operator = '='
            joins.append("""
                LEFT OUTER JOIN organizations ON (domains.org_id = organizations.id)
            """)

            filter_parts.append(
                self.mogrify(
                    'organizations.registrar_id {operator} %(value)s'.format(
                        operator=operator
                    ),
                    {
                        'value': value
                    }
                )
            )
            used_filters.append('registrar_id')

        return distinct, filter_parts, joins, used_filters

    @staticmethod
    def _check_domain_is_blocked(domains):
        for domain in domains:
            if bool(domain['blocked_at']):
                raise RuntimeError('Can\'t modify blocked domain {}'.format(domain['name']))

    def update(self, update_data, filter_data=None, force=False):
        domains = self.find(filter_data=filter_data)
        if not force:
            self._check_domain_is_blocked(domains)
        super(DomainModel, self).update(update_data=update_data, filter_data=filter_data)

    def delete(self, filter_data=None, force_remove_all=False, delete_blocked=False):
        filter_data = _encode_name_to_punycode(filter_data)
        domains = self.find(filter_data=filter_data)
        if not force_remove_all:
            for domain in domains:
                domain_name = domain['name']
                if domain['master']:
                    raise RuntimeError('Can\'t delete master domain {}'.format(domain_name))

        if not delete_blocked:
            self._check_domain_is_blocked(domains)
        super(DomainModel, self).delete(filter_data=filter_data, force_remove_all=force_remove_all)

    def delete_domain(self, domain_name, org_id, author_id, delete_blocked=False):
        from intranet.yandex_directory.src.yandex_directory.core.actions import action_domain_delete

        lower_domain_name = domain_name.lower()
        domain = DomainModel(self._connection).get(lower_domain_name, org_id)

        if not domain:
            raise DomainNotFound(domain=domain)
        if domain['master']:
            raise DeleteMasterDomainError(domain=lower_domain_name)

        if bool(domain['blocked_at']) and not delete_blocked:
            raise BlockedDomainError(domain=lower_domain_name)

        if domain['via_webmaster']:
            WebmasterDomainLogModel(self._connection).create(
                org_id=org_id,
                uid=author_id,
                name=lower_domain_name,
                action=domain_action.delete
            )

        admin_uid = get_organization_admin_uid(self._connection, org_id)
        try:
            # мастера в организации может не быть, если все домены неподтверждены
            master_domain = self.get_master(org_id)['name']
            master_domain_id = get_domain_id_from_blackbox(master_domain)
        except MasterDomainNotFound:
            master_domain_id = None
        delete_domain_in_passport(domain_name,
                                  admin_uid,
                                  is_alias=bool(master_domain_id),
                                  master_domain_id=master_domain_id)

        DomainModel(self._connection).delete(
            {
                'name': domain_name,
                'org_id': org_id
            },
            delete_blocked=delete_blocked
        )

        action_domain_delete(
            self._connection,
            org_id=org_id,
            author_id=author_id,
            object_value=domain,
            old_object=domain,
        )

    def _custom_find(self, *args, **kwargs):
        """
        Вызываем стандарный поиск из BaseModel.find(). Этот метод ищет по одному шарду.
        Используем _custom_find в момент вызовов self.find & self.get
        """

        # DIR-1459: Тут просто перекодируем имена доменов из punycode в unicode,
        #           так как в БД они хранятся закодированными

        results = super(DomainModel, self).find(*args, **kwargs)
        if not results:
            return results

        if isinstance(results, list):
            return list(map(_decode_name_from_punycode, results))
        else:
            return _decode_name_from_punycode(results)

    def find(self, *args, **kwargs):
        """
        Если не задан _connection или явно задан all_shards=True, то ищем домен по всем шардам
        """
        if kwargs.pop('all_shards', None) or self._connection is None:
            # если connection не указан, то производим поиск по всем шардам
            shards = get_shard_numbers()
            results = []
            one = kwargs.get('one')
            for shard in shards:
                # NODE кажется, что это работает неправильно, т.к. возращает данные
                # из 1 шарда
                with get_main_connection(shard) as self._connection:
                    shard_result = self._custom_find(*args, **kwargs)
                    if shard_result:
                        if one:
                            return shard_result
                        results.extend(shard_result)
            return results

        return self._custom_find(*args, **kwargs)

    def find_all(self, *args, **kwargs):
        """
        Аналог функции find, но возращает данные из всех шардов.
        Работает неоптимально, возможно, это можно будет распаралелить.
        :param args:
        :param kwargs:
        :return:
        """
        kwargs.pop('one', None)
        results = []
        if kwargs.pop('all_shards', None) or self._connection is None:
            # если connection не указан, то производим поиск по всем шардам
            shards = get_shard_numbers()
            for shard in shards:
                with get_main_connection(shard) as self._connection:
                    found = self._custom_find(*args, **kwargs)
                    results.extend(found)

            return results
        return self._custom_find(*args, **kwargs)

    def get(self,
            domain_name=None,
            org_id=None,
            fields=None,
            all_shards=False,
            filter_data=None,
            ):

        # Если не задан _connection или явно задан all_shards=True, то ищем домен по всем шардам

        if filter_data is None:
            filter_data = {}

        filter_data['name'] = domain_name

        if org_id:
            filter_data['org_id'] = org_id

        return self.find(
            filter_data=filter_data,
            all_shards=all_shards,
            fields=fields,
            one=True,
        )

    def update_one(self, name=None, org_id=None, data=None, force=False):
        """
        Считаем, что master & display уже изменились в паспорте.
        Если данные пришли в виде data = dict(master=True,
        display=True), а для организации уже есть домен domain1
        с master=True или display=True, то для
        него master & display установится в False, а для нового домена
        проставятся в True.

        :param: name - имя домена, для которого меняем параметры
        """
        filter_data = {
            # перед запросом в базу, нужно закодировать
            # имя в punycode, так как в БД у нас хранятся
            # нормализованные имена (DIR-1459).
            'name': to_punycode(name),
            'org_id': org_id,
        }
        # проверим, что домен существует в нашей базе
        domain = self.find(filter_data=filter_data, one=True)
        if not domain:
            raise DomainNotFound(domain=name)

        # дальше работаем с текущим master-id организaции и меняем данные для него прям тут
        try:
            current_master_domain = self.get_master(org_id)
        except MasterDomainNotFound:
            current_master_domain = None
        # https://st.yandex-team.ru/DIR-1992
        # если домен хотим сделать отображаемым, то у текущего current_master_domain
        # в базе меняем display & master на False,
        # а для нового master_domain проставляем master & display = True
        if (data.get('master') or data.get('display')) and current_master_domain:
            if not force and (bool(current_master_domain['blocked_at']) or bool(domain['blocked_at'])):
                raise BlockedDomainError(domain=current_master_domain['name'])
            # проверим, что домен который пытаются сделать мастером,
            # подтвержден
            if not domain['owned']:
                raise DomainNotValidatedError(
                    'not_owned_domain',
                    'Can\'t make master not owned domain'
                )

            # тут выбираем uid админа домена и название, потому что паспорт ищет домены по admin_uid и
            # меняет master_id только таким образом
            admin_uid = get_organization_admin_uid(self._connection, org_id)
            current_master_name = current_master_domain['name']
            new_master_name = name

            current_master = get_domain_info_from_blackbox(current_master_name) or {}
            new_master = get_domain_info_from_blackbox(new_master_name) or {}

            master_domid = current_master.get('domain_id')
            new_master_domid = new_master.get('domain_id')

            with log.fields(master_domid=master_domid, new_master_domid=new_master_domid):
                if not master_domid:
                    # такое могло быть если он удалился при автоотрыве домена и перешел к другому админу
                    # а у нас в базе данные по домену не поменялись
                    # мы тут оптимистично думаем, что возможен только такой варинт рассинхрона
                    log.error('No current master domain in Passport')

                if not new_master_domid:
                    log.error('No new master domain in Passport')
                    raise RuntimeError('No new master domain in Passport')

                #  старый мастер ещё не алиас нового в паспорте
                if current_master.get('master_domain') != to_punycode(new_master_name) and master_domid and new_master_domid:
                    log.info('Changing master domain in Passport')
                    app.passport.set_master_domain(
                        master_domid,
                        new_master_domid,
                    )

            # Для текущего мастер-домена меняем в базе параметры display & master
            self.update(
                update_data={
                    'master': False,
                    'display': False,
                },
                filter_data={
                    'name': current_master_name,
                    'org_id': org_id,
                },
                force=force,
            )

            # https://st.yandex-team.ru/DIR-1992
            # Готовим данные для нового мастер-домена
            if data.get('master') or data.get('display'):
                data['master'] = True
                data['display'] = True

        if data:
            self.update(
                update_data=_encode_name_to_punycode(data),
                filter_data=filter_data,
                force=force,
            )

    def change_master_domain(self, org_id, domain_name, force=False):
        # меняет мастер домен в организации org_id на domain_name
        # эту функцию не меняем, стираем после миграции на доменатор
        from intranet.yandex_directory.src.yandex_directory.core.actions import action_domain_master_modify
        from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationModel
        from intranet.yandex_directory.src.yandex_directory.core.tasks.tasks import UpdateEmailFieldsTask

        old_master_domain = self.get_master(org_id)

        self.update_one(
            name=domain_name,
            org_id=org_id,
            data={'master': True},
            force=force
        )

        # асинхронно обновляем все поля email у групп, департаментов и пользователей
        # https://st.yandex-team.ru/DIR-2438
        UpdateEmailFieldsTask(self._connection).delay(
            master_domain=domain_name,
            org_id=org_id,
        )

        new_master_domain = self.get_master(org_id)
        org = OrganizationModel(self._connection).get(id=org_id)
        action_domain_master_modify(
            self._connection,
            org_id=org_id,
            author_id=org['admin_uid'],
            object_value=new_master_domain,
            old_object=old_master_domain,
        )

    def get_master(self, org_id):
        from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
            get_domains_from_db_or_domenator, DomainFilter
        )
        with get_meta_connection() as meta_connection:
            domain = get_domains_from_db_or_domenator(
                meta_connection,
                DomainFilter(org_id=org_id, master=True),
                main_connection=self._connection,
                one=True,
            )

        if not domain:
            raise MasterDomainNotFound(org_id=org_id)
        return domain

    def get_aliases(self, org_id, owned=True):
        """
        Отдаем все домены организации за исключением того, который указан в
        domain_name.
        В терминах Директории: мы отдаем все алиасы для домена.
        :param org_id:
        :param owned:
        :return: [
            {
                domain1_name: 'domain1.ru',
                domain1_display: False,
                domain1_master: False,
                domain1_owned: False,
            },
            {
                domain2_name: 'domain2.ru',
                domain2_display: False,
                domain2_master: True,
                domain2_owned: True,
            },
            ...
        ]
        """
        filter_data = {'org_id': org_id, 'master': False}
        if owned is not None:
            filter_data['owned'] = owned

        return self.find(filter_data=filter_data)

    def count_organizations_with_non_tech_domains(self, owned=None):
        """
        Считаем количество организаций, у которых есть хотя бы один не технический домен
        """
        domain_part = '%%%s' % app.config['DOMAIN_PART']

        if owned is None:
            query = """
                SELECT COUNT(DISTINCT org_id) FROM domains
                WHERE name NOT LIKE %(name)s
            """
            query_kwargs = {
                'name': domain_part,
            }
        else:
            query = """
                SELECT COUNT(DISTINCT org_id) FROM domains
                WHERE name NOT LIKE %(name)s AND owned=%(owned)s
            """
            query_kwargs = {
                'name': domain_part,
                'owned': owned,
            }

        return self._connection.execute(query, query_kwargs).fetchone()[0]

    def get_validation_stat(self, days):
        """
        Возвращает статистику по подтвердению домена
        validation_delay - список с временами потдверждения доменов
        added_count - количество доменов добавленных за период времени
        added_owned - сколько из них подтверденных
        :param days: за сколько  последних дней нужны данные
        :rtype: dict
        """
        assert isinstance(days, int), 'invalid days parameter type'

        stat = defaultdict(dict)

        query = """
            SELECT
                CASE WHEN via_webmaster THEN 'webmaster' ELSE 'pdd' END as via,
                CEIL(EXTRACT(EPOCH FROM (validated_at - created_at))) as validation_delay
            FROM domains
            WHERE created_at > NOW() - INTERVAL '{days} days' AND owned AND validated_at IS NOT NULL
        """.format(days=days)

        validation_delay = self._connection.execute(query).fetchall()
        if validation_delay:
            for via, validation_delay in validation_delay:
                stat[via].setdefault('validation_delay', []).append(validation_delay)

        query = """
            SELECT
                CASE WHEN via_webmaster THEN 'webmaster' ELSE 'pdd' END as via,
                COUNT(*) as added_count,
                SUM(
                    case when owned then 1 else 0 end
                ) as added_owned
            FROM domains
            WHERE created_at > NOW() - INTERVAL '{days} days' AND NOT (validated_at IS NULL AND owned)
            GROUP BY  via_webmaster
        """.format(days=days)
        domain_stat = self._connection.execute(query).fetchall()
        if domain_stat:
            for via, added_count, added_owned in domain_stat:
                stat[via]['added_count'] = added_count
                stat[via]['added_owned'] = added_owned
        return stat

    def count_long_blocked_domains(self, longer_than_minutes=10):
        query = """
            SELECT COUNT(name) FROM domains
                WHERE blocked_at is not NULL
                AND(now() - blocked_at) > INTERVAL '%(longer_than_minutes)s minutes';
        """
        return self._connection.execute(
            query,
            self.prepare_dict_for_db({
                'longer_than_minutes': longer_than_minutes,
            })
        ).fetchone()[0]

    def block(self, org_id):
        log.debug('Blocking domains')
        self.update(
            update_data={'blocked_at': utcnow()},
            filter_data={'org_id': org_id},
        )

    def unblock(self, org_id):
        log.debug('Unblocking domains')
        self.update(
            update_data={'blocked_at': None},
            filter_data={'org_id': org_id},
            force=True
        )



class DomainsAnalyticsInfoModel(BaseAnalyticsModel):
    db_alias = 'main'
    table = 'domains_analytics_info'
    primary_key = 'org_id'
    order_by = 'org_id'

    simple_fields = {
        'org_id',
        'name',
        'for_date',
        'master',
    }

    all_fields = simple_fields

    def save(self, org_id=None):
        # сохраняем информацию о доменах за сегодня
        if org_id:
            org_id_filter = 'AND domains.org_id=%(org_id)s'
        else:
            org_id_filter = ''

        query = '''
                   INSERT INTO domains_analytics_info(org_id, name, master)
                       SELECT
                           domains.org_id,
                           domains.name,
                           domains.master
                       FROM domains
                       WHERE
                           domains.owned = TRUE
                           {org_id_filter}
                   ON CONFLICT DO NOTHING;
        '''

        query = query.format(
            org_id_filter=org_id_filter,
        )
        query = self.mogrify(
            query,
            {
                'org_id': org_id,
            }
        )

        with log.name_and_fields('analytics', org_id=org_id, for_date=utcnow().isoformat()):
            log.info('Saving domains analytics data...')
            self._connection.execute(query)
            log.info('Domains analytics data has been saved')
            return True


class WebmasterDomainLogModel(BaseModel):
    db_alias = 'main'
    table = 'webmaster_domains_log'
    all_fields = [
        'id',
        'org_id',
        'uid',
        'name',
        'action',
        'verification_type',
        'timestamp',
    ]

    def create(self, org_id, uid, name, action, verification_type=None):
        new_domain = self.insert_into_db(
            org_id=org_id,
            uid=uid,
            name=to_lowercase(to_punycode(name)),
            action=action,
            verification_type=verification_type,
        )
        return _decode_name_from_punycode(new_domain)

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('name', encode=to_punycode) \
            ('action') \
            ('org_id') \
            ('verification_type')
        return distinct, filter_parts, joins, used_filters

    def get_validation_start_count(self, days):
        """
        Возвращает y скольки доменов было запущено подтверждение владения
        :param days: за сколько  последних дней нужны данные
        :rtype: int
        """

        assert isinstance(days, int), 'invalid days parameter type'

        query = """
                  SELECT  COUNT(*) FROM (
                    SELECT DISTINCT d.org_id, d.name FROM domains AS d
                    INNER JOIN webmaster_domains_log  AS wdl ON d.org_id=wdl.org_id AND  d.name=wdl.name
                    WHERE d.via_webmaster
                      AND d.created_at > NOW() - INTERVAL '{days} days'
                      AND wdl.action = 'verify'
                ) as t;
                """.format(days=days)

        validation_start = self._connection.execute(query).fetchone()[0]
        return validation_start

    def get_last_verification_type(self, org_id, domain_name, admin_uid):
        query = """
                  SELECT verification_type FROM webmaster_domains_log
                  WHERE org_id=%(org_id)s AND name=%(name)s AND uid=%(uid)s
                  ORDER BY timestamp DESC LIMIT 1;
                """

        row = self._connection.execute(
            query,
            {
                'org_id': org_id,
                'name': domain_name,
                'uid': admin_uid
            },
        ).fetchone()
        if row:
            return row[0]
