# -*- coding: utf-8 -*-
import datetime

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.models.base import BaseModel
from intranet.yandex_directory.src.yandex_directory.common.utils import utcnow


class EventModel(BaseModel):
    db_alias = 'main'
    table = 'events'

    json_fields = [
        'content',
        'object',
    ]
    all_fields = [
        'id',
        'org_id',
        'revision',
        'name',
        'content',
        'object_type',
        'object',
        'timestamp',
        'notify_at',
        # не из таблицы
        'environment'
    ]
    select_related_fields = {
        'environment': None,
    }

    def create(self,
               org_id,
               revision,
               name,
               object_value,
               object_type,
               content,
               notify_at=None
               ):
        if not content:
            content = {}

        data = {
            'revision': revision,
            'org_id': org_id,
            'name': name,
            'object_type': object_type,
            'object': object_value,
            'content': content,
            'notify_at': notify_at or utcnow()
        }

        return self.insert_into_db(**data)

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('id', can_be_list=True) \
            ('id__gt') \
            ('id__between') \
            ('name', can_be_list=True) \
            ('org_id') \
            ('revision', can_be_list=True) \
            ('revision__gt') \
            ('revision__lt') \
            ('revision__between')

        #  Фильтр по сервису, испрользуется в тестах для выборки событий
        #  resource_grant_changed
        if 'object.service' in filter_data:
            filter_parts.append(
                self.mogrify(
                    'object->>\'service\' = %(service)s',
                    {
                        'service': filter_data.get('object.service')
                    }
                )
            )
            used_filters.append('object.service')

        if 'environment' in filter_data:
            joins.append("INNER JOIN organizations ON organizations.id = events.org_id")
            filter_parts.append(
                self.mogrify(
                    'organizations.environment = %(environment)s',
                    {
                        'environment': filter_data.get('environment')
                    }
                )
            )
            used_filters.append('environment')

        if 'timestamp__lt' in filter_data:
            dt = filter_data.get('timestamp__lt')
            if not isinstance(dt, datetime.datetime):
                raise ValueError('timestamp__lt must be datetime.datetime not {}'.format(type(dt)))

            filter_parts.append(
                self.mogrify(
                    'events.timestamp < %(timestamp)s',
                    {
                        'timestamp': dt.isoformat()
                    }
                )
            )
            used_filters.append('timestamp__lt')

        return distinct, filter_parts, joins, used_filters

    def get_select_related_data(self, select_related):
        if not select_related:
            return [self.default_all_projection], [], []

        select_related = select_related or []
        projections = []
        joins = []
        processors = []

        if 'environment' in select_related:
            joins.append("INNER JOIN organizations ON organizations.id = events.org_id")
            projections += [
                'organizations.environment AS "environment"'
            ]

        return projections, joins, processors

    def truncate(self, days):
        """
        Усекаем таблицу events
        Удаляем записи старше `days` дней
        """
        self.delete({
            'timestamp__lt': utcnow() - datetime.timedelta(days=days)
        })


class LastProcessedRevisionModel(BaseModel):
    primary_key = ('org_id', 'environment')
    order_by = 'org_id'
    db_alias = 'main'
    table = 'last_processed_revision'

    all_fields = [
        'org_id',
        'environment',
        'revision',
        'wait_till',
    ]

    def get_last_processed_revision(self, org_id):
        """
        Получаем id последней обработанной ревизии для заданной организации.
        """
        found = self.filter(org_id=org_id).fields('revision').one()
        if found:
            return found['revision']

    def get_not_processed_count(self):
        """Возвращает число ревизий за последний час, которые есть в базе,
           но ещё не обработаны.
           Это нужно для построения графика по необработанным ревизиям.
        """
        env = app.config['ENVIRONMENT']
        query = """
SELECT COUNT(*) FROM actions
INNER JOIN last_processed_revision
ON actions.org_id = last_processed_revision.org_id
WHERE actions.revision > last_processed_revision.revision
  AND last_processed_revision.environment = %(env)s
  AND actions.timestamp > NOW() - INTERVAL '1 hour'
"""
        result = self._connection.execute(query, dict(env=env))
        counter = result.fetchone()[0]
        return counter

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts, joins, used_filters = [], [], []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('environment') \
            ('org_id', can_be_list=True)

        return distinct, filter_parts, joins, used_filters

    def delete(self, filter_data=None, force_remove_all=False):
        raise RuntimeError('Delete not allowed')

    def create(self, org_id):
        """
        Создаёт запись в таблице для организации с ревизией 0
        """
        return self.insert_into_db(
            org_id=org_id,
            environment=app.config['ENVIRONMENT'],
            revision=0,
        )


class CallbackEventsModel(BaseModel):
    """
    Список событий о которых нужно оповестить подписчиков.
    """
    db_alias = 'main'
    table = 'callback_events'

    json_fields = [
        'settings',
    ]

    all_fields = [
        'id',
        'callback',
        'event_id',
        'settings',
        'environment',
        'created_at',
        'done',
        'last_response_at',
        'response_code',
        'response_content',
        # not from db
        'event',
    ]

    select_related_fields = {
        'event': 'EventModel',
    }

    def create(self, callback, event_id, settings, environment):
        return self.insert_into_db(
            callback=callback,
            event_id=event_id,
            settings=settings,
            environment=environment,
        )

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []
        filter_parts, joins, used_filters = [], [], []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('id', can_be_list=True) \
            ('event_id') \
            ('event_id__lte') \
            ('event_id__gte') \
            ('event_id__between') \
            ('callback') \
            ('settings') \
            ('environment') \
            ('done') \
            ('created_at') \
            ('created_at__gt') \
            ('created_at__lt') \
            ('last_response_at') \
            ('last_response_at__gt') \
            ('last_response_at__lt') \
            ('response_code') \
            ('response_content')

        if 'org_id' in filter_data:
            filter_parts.append(
                self.mogrify(
                    """
                    events.org_id =  %(org_id)s
                    """,
                    {
                        'org_id': filter_data.get('org_id'),
                    }
                )
            )

            used_filters.append('org_id')

        return distinct, filter_parts, joins, used_filters

    def get_select_related_data(self, select_related):
        select_related = select_related or []
        projections = set()
        joins = []
        processors = []

        projections.update([
            'callback_events.id',
            'callback_events.callback',
            'callback_events.event_id',
            'callback_events.environment',
            'events.org_id as "event.org_id"',
            'events.id as "event.id"',
            'events.notify_at as "event.notify_at"',
            'events.object as "event.object"',
            'events.revision as "event.revision"',
            'events.org_id as "event.org_id"',
            'events.name as "event.name"',
            'events.content as "event.content"',
        ])
        joins.append("INNER JOIN events ON events.id = callback_events.event_id")
        return projections, joins, processors

    def get_delays(self):
        return self._connection.execute("""
            select
                count(*) as cnt,
                max(now() - created_at) as max,
                callback
            from
                callback_events
            where
                done = False and
                environment = %(env)s
            group by
                callback
        """, env=app.config['ENVIRONMENT']).fetchall()


class ProcessedEventsModel(BaseModel):
    """
    Запоминаем обработанные событий для статистики о детекции зарастания дыр
    """
    db_alias = 'main'
    table = 'processed_events'

    all_fields = [
        'id',
        'event_id',
        'environment',
        'created_at',
    ]

    def create(self, event_id, environment):
        return self.insert_into_db(
            event_id=event_id,
            environment=environment,
        )

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []
        filter_parts, joins, used_filters = [], [], []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('id', can_be_list=True) \
            ('environment') \
            ('created_at') \
            ('event_id') \
            ('created_at__lt') \
            ('created_at__lte') \
            ('created_at__gt') \
            ('created_at__gte')

        return distinct, filter_parts, joins, used_filters

    def get_events_hole_for_last_hour(self):
        environment = app.config['ENVIRONMENT']
        query = """
        SELECT
            min(event_id) AS event_min,
            max(event_id) AS event_max,
            count(*) AS event_count
        FROM processed_events
        WHERE created_at >= %(created_at)s AND environment = %(environment)s
        """
        hour_ago = utcnow() - datetime.timedelta(hours=1)
        prepared_params = self.prepare_dict_for_db({
            'created_at': hour_ago,
            'environment': environment,
        })

        data = dict(self._connection.execute(query, prepared_params).fetchone())
        if data['event_count'] == 0:
            return 0

        events_count = EventModel(self._connection).filter(
            environment=environment,
            id__between=(
                data['event_min'],
                data['event_max']
            ),
        ).count()
        return events_count - data['event_count']
