# -*- coding: utf-8 -*-
import json
from collections import defaultdict
from copy import deepcopy
from itertools import chain, starmap
from operator import itemgetter

from sqlalchemy.exc import IntegrityError

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    ConstraintValidationError,
    ServiceNotFound,
    ServiceNotLicensed,
    AuthorizationError,
)
from intranet.yandex_directory.src.yandex_directory.common.models.base import (
    BaseModel,
    PseudoModel,
    ALL_FIELDS,
    ALL_SIMPLE_FIELDS,
    _model_registry,
    set_to_none_if_no_id,
    Values,
)
from intranet.yandex_directory.src.yandex_directory.common.models.types import TYPE_USER
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    prepare_for_tsquery,
    to_lowercase,
    Ignore,
    join_dicts,
    make_simple_strings,
)
from intranet.yandex_directory.src.yandex_directory.core.db import queries
from intranet.yandex_directory.src.yandex_directory.core.models.department import (
    DepartmentModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models.resource import (
    ResourceModel,
    ResourceRelationModel,
    relation_name,
)
from intranet.yandex_directory.src.yandex_directory.core.models.service import (
    OrganizationServiceModel,
    MAILLIST_SERVICE_SLUG,
)
from intranet.yandex_directory.src.yandex_directory.core.models.user import (
    UserModel,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    only_ids,
    build_email,
    check_objects_exists,
    unfreeze_or_copy,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.organization import get_organization_maillist_type
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

GROUP_TYPE_GENERIC = 'generic'
GROUP_TYPE_ORGANIZATION_ADMIN = 'organization_admin'
GROUP_TYPE_ROBOTS = 'robots'
GROUP_TYPE_DEPARTMENT_HEAD = 'department_head'
GROUP_TYPE_ORGANIZATION_DEPUTY_ADMIN = 'organization_deputy_admin'
GROUP_TYPE_OUTSTAFF = 'outstaff'
TECHNICAL_TYPES = (
    GROUP_TYPE_ORGANIZATION_ADMIN,
    GROUP_TYPE_ROBOTS,
    GROUP_TYPE_DEPARTMENT_HEAD,
    GROUP_TYPE_ORGANIZATION_DEPUTY_ADMIN,
)
GROUP_FIELDS = [
    'id',
    'description',
    'org_id',
    'name',
    'type',
    'members'  # это на самом деле связь с дочерними группами
]


def ref_to_relation(ref):
    """Преобразует ссылки на объекты типа {'type': 'user', 'id': 123} и
    {'type': 'user', 'object': {'id': ..., 'name': ...}}
    в словари, годные для создания отношения с этим объектом: {'user_id': 123}

    Если поле type отсутствует, объект подается на выход как есть.
    Это нужно потому, что некоторый легаси-код оперирует {'user_id': 124}
    как ссылками. Со временем надо будет это выпилить отовсюду.
    """

    if 'type' in ref:
        # это правильный, новый стиль мембера
        obj_type = ref['type']
        return {obj_type + '_id': ref.get('id') or ref.get('object', {}).get('id')}

    return ref

REL_TUPLE_LABELS = ('user_id', 'group_id', 'department_id')

DEFAULT_DESCRIPTION = {'ru': '', 'en': ''}


def rel_to_tuple(rel):
    """Возвращает вместо словаря tuple:
    (user_id, group_id, department_id)

    При этом `rel` должен содержать одно из полей.
    """
    return tuple(map(rel.get, REL_TUPLE_LABELS))


def tuple_to_rel(tup):
    """Делает процедуру, обратную rel_to_tuple,
    превращая (user_id, group_id, department_id)
    в словарик, где будет установлено одно из полей.
    Для полей в которых прописаны None, соответствующие
    поля в словаре не создаются.
    """
    zipped = list(zip(REL_TUPLE_LABELS, tup))
    return dict([i for i in zipped if i[1] is not None])


class GroupModel(BaseModel):
    db_alias = 'main'
    table = 'groups'
    json_fields = [
        'name',
        'description',
        'aliases',
    ]
    all_fields = [
        'org_id',
        'id',
        'name',
        'type',
        'description',
        'resource_id',
        'label',
        'author_id',
        'created',
        'removed',
        'members_count',
        'aliases',
        'external_id',
        'maillist_type',
        'uid',  # uid аккаунта в паспорте
        'name_plain',
        'description_plain',
        # not from db
        'admins',
        'members',
        'all_users',
        'author',
        'email',
        'member_of',
        'tracker_license',
    ]
    select_related_fields = {
        'author': 'UserModel',
        'tracker_license': 'ResourceRelationModel'
    }
    prefetch_related_fields = {
        'admins': 'UserModel',
        'members': 'GroupMember',
        'all_users': 'UserModel',
        'email': 'DomainModel',
        'member_of': 'GroupModel',
    }
    field_dependencies = {
        'members': ['resource_id', 'org_id'],
        'admins': ['resource_id', 'org_id'],
        'all_users': ['org_id'],
        'email': ['org_id'],
        'member_of': ['org_id'],
        'tracker_license': ['org_id'],
    }

    def _build_model_field(self, field):
        if field == 'tracker_license':
            return None
        return super()._build_model_field(field)

    def create(self,
               name,
               org_id,
               label=None,
               email=None,
               aliases=None,
               type=GROUP_TYPE_GENERIC,
               id=None,
               description=None,
               author_id=None,
               members=[],
               admins=[],
               external_id=None,
               generate_action=True,
               action_author_id=None,
               maillist_type=None,
               uid=None,
               ):
        """Создать новую группу.

        Если в members список, то входящие в него объекты включаются в группу.
        Каждый member должен быть словарем типа:
        {'type': 'department', 'id': 123}
        {'type': 'user', 'id': 124}
        {'type': 'group', 'id': 654}

        В списке admins могут быть переданы uid людей, которые имеют права на
        редактирование этой группы.
        """
        conn = self._connection

        check_objects_exists(conn, org_id, members)
        member_relations = list(map(ref_to_relation, members))
        relation_tuples = set(map(rel_to_tuple, member_relations))
        relations = list(map(tuple_to_rel, relation_tuples))

        def add_include_relation(r):
            r['name'] = relation_name.include
            return r

        list(map(add_include_relation,  relations))

        for admin_uid in admins:
            relation = dict(
                name=relation_name.admin,
                user_id=admin_uid,
            )
            relations.append(relation)

        resource = self.create_resource_for_group(
            org_id=org_id,
            relations=relations
        )

        if aliases is None:
            aliases = []

        label = to_lowercase(label)
        email = to_lowercase(email)
        if not maillist_type:
            maillist_type = get_organization_maillist_type(conn, org_id)
        query_kwargs = {
            'org_id': org_id,
            'label': label,
            'email': email or build_email(
                conn,
                label,
                org_id=org_id,
            ),
            'aliases': aliases,
            'name': name,
            'type': type,
            'description': description or DEFAULT_DESCRIPTION,
            'resource_id': resource['id'],
            'author_id': author_id,
            'external_id': external_id,
            'maillist_type': maillist_type,
            'uid': uid,
            'name_plain': make_simple_strings(name),
            'description_plain':  make_simple_strings(description) or None,
        }

        if id:
            query_kwargs['id'] = id  # todo: test id
            query = queries.CREATE_GROUP_WITH_ID['query']
        else:
            query = queries.CREATE_GROUP['query']
        new_group = self.insert(
            query,
            query_kwargs,
        )
        members_count = self.update_leafs_cache_and_member_count(org_id=org_id, group_id=new_group['id'])
        new_group['members'] = members
        new_group['members_count'] = members_count
        group = self.get_extend_info(new_group['id'], org_id)

        # генерируем действие
        from intranet.yandex_directory.src.yandex_directory.core.actions import action_group_add
        if generate_action:
            action_group_add(
                self._connection,
                org_id=org_id,
                author_id=action_author_id or author_id,
                object_value=group,
            )
        return group

    def get_extend_info(self, group_id, org_id):
        """
        Расширенная информация по группе
        В этом формате данные пригодны для функции prepare_group
        """

        group = self.get(
            org_id=org_id,
            group_id=group_id,
            fields=[
                '**',
            ],
        )
        group['member_of'] = self.get_parents(
            org_id=org_id,
            group_id=group_id,
        )
        return group

    def create_resource_for_group(self, org_id, relations=[]):
        conn = self._connection
        return ResourceModel(connection=conn).create(
            org_id=org_id,
            service=app.config['DIRECTORY_SERVICE_NAME'],
            relations=relations,
        )

    def get_or_create_admin_group(self, org_id):
        """
        Создание админской группы
        """
        group = self._connection.execute(
            queries.GET_ORGANIZATION_ADMIN_GROUP['query'],
            {'org_id': org_id}
        ).fetchone()

        if not group:
            group = self.create(
                name={
                    'ru': 'Администратор организации',
                    'en': 'Organization administrator',
                },
                org_id=org_id,
                type=GROUP_TYPE_ORGANIZATION_ADMIN,
                action_author_id=1,
            )
        return group

    def get_or_create_deputy_admin_group(self, org_id):
        """
        Создание группы заместителей администраторов
        """
        group = self.find({'type': GROUP_TYPE_ORGANIZATION_DEPUTY_ADMIN, 'org_id': org_id}, one=True)

        if not group:
            group = self.create(
                name={
                    'ru': 'Заместители администраторов организации',
                    'en': 'Organization deputy administrators',
                },
                org_id=org_id,
                type=GROUP_TYPE_ORGANIZATION_DEPUTY_ADMIN,
                action_author_id=1,
            )
        return group

    def get_robot_group(self, org_id):
        """
        Получение роботной группы
        """
        return self.find({'type': GROUP_TYPE_ROBOTS, 'org_id': org_id}, one=True)

    def get_or_create_robot_group(self, org_id):
        """
        Создание роботной группы или получение существующей, если она есть.
        """
        group = self.find({'type': GROUP_TYPE_ROBOTS, 'org_id': org_id}, one=True)
        if not group:
            group = self.create(
                name={
                    'ru': 'Роботы организации',
                    'en': 'Organization robots',
                },
                description={
                    'ru': 'Роботы организации',
                    'en': 'Organization robots',
                },
                org_id=org_id,
                type=GROUP_TYPE_ROBOTS,
            )
        return group

    def is_subgroup(self, org_id, group_id, subgroup_id):
        """Проверяет, действительно ли группа с subgroup_id входит в группу group_id.
        Если входит, то возвращает True.
        """
        group_id = int(group_id)
        subgroup_id = int(subgroup_id)
        if group_id == subgroup_id:
            return True
        parents = only_ids(
            self.get_parents(
                org_id=org_id,
                group_id=subgroup_id
            )
        )

        for parent_id in parents:
            if self.is_subgroup(org_id, group_id, parent_id):
                return True
        return False

    def _get_diff_relations(self,
                            collaborators,
                            relation_type,
                            org_id,
                            resource_id):
        """
        Этот метод для ресурса resource_id считает diff отношений.
        После наложения этого диффа, ресурс будет иметь связь только
        с сущностями из списка.
        Некоторые из связей могут быть разорваны (to_delete),
        а некоторые созданы (to_add), а некоторые останутся как есть.
        И всё это касается только связей типа relation_type (
        названия relation_type: include - для мемберов, admin - для админов)

        (возвращает кортеж: (
            to_delete, #  какие отношения нужно удалить,
            to_add, #  какие - добавить
        )
        :param collaborators: мемберы (список dict-ов) или админы (список uid-ов)
        :param relation_type: include- или admin-связь
        :return: tuple(to_delete, to_add)
        """

        # collaborators - мемберы или админы группы
        collaborators = list(map(ref_to_relation, collaborators))
        new_relations = set(map(rel_to_tuple, collaborators))

        current_relations = ResourceModel(self._connection).get_relations(
            resource_id=resource_id,
            org_id=org_id,
            name=relation_type,
        )
        current_relations = set(map(rel_to_tuple, current_relations))

        # теперь, когда у нас есть два множества отношений,
        # мы можем вычислить какие из них необходимо удалить
        # а какие необходимо создать
        to_delete = list(map(tuple_to_rel, current_relations - new_relations))
        to_add = list(map(tuple_to_rel, new_relations - current_relations))

        return to_delete, to_add

    def _update_members(self, org_id, group_id, data_members):
        """
        Обновляем членов одной группы
        :param org_id:
        :param group_id:
        :param data_members: список словарей контейнеров
        (групп, пользователей, депаратаментов)
        """

        # getting group member relations
        group = self.get(org_id=org_id, group_id=group_id)
        if not group:
            return
        resource_id = group['resource_id']

        # проверяем валидность контейнеров
        conn = self._connection
        check_objects_exists(conn, org_id, data_members)
        # понимаем, какие relations нужно удалить, а какие - добавить
        to_delete, to_add = self._get_diff_relations(
            data_members,
            relation_name.include,
            org_id,
            resource_id
        )

        model = ResourceRelationModel(self._connection)
        for item in to_delete:
            item['org_id'] = org_id
            item['resource_id'] = resource_id
            # удаляем только include-связи
            item['name'] = relation_name.include
            model.delete(item)

        groups_to_add = [item for item in to_add if 'group_id' in item]
        for item in groups_to_add:
            subgroup_id = item['group_id']

            # проверяем, что subgroup_id не включает в себя текущую группу
            if self.is_subgroup(
                org_id=org_id,
                group_id=subgroup_id,
                subgroup_id=group_id
            ):
                self._raise_cycle_detected_error(
                    group_id=group_id,
                    subgroup_id=subgroup_id
                )

        for item in to_add:
            item['org_id'] = org_id
            item['name'] = relation_name.include
            item['resource_id'] = resource_id
            model.create(**item)

        if to_add or to_delete:
            self.update_leafs_cache_and_member_count(org_id=org_id, group_id=group_id)

    def _update_admins(self, org_id, group_id, data_admins):
        """
        Обновляем админов одной группы.
        :param org_id:
        :param group_id:
        :param data_admins: список словарей контейнеров
        (групп, пользователей, депаратаментов)
        """

        # getting group admin relations
        group = self.get(org_id=org_id, group_id=group_id)
        resource_id = group['resource_id']

        # если в админы передаются группы и департаменты, то кидаем ошибку
        # (пока так - потом посмотрим, подробнее - у Димы Тарабанько)
        for obj in data_admins:
            if obj['type'] != TYPE_USER:
                raise ConstraintValidationError(
                    'only_users_can_be_group_admins',
                    'Only users can be group admins',
                )

        check_objects_exists(self._connection, org_id, data_admins)
        to_delete, to_add = self._get_diff_relations(
            data_admins, relation_name.admin, org_id, resource_id
        )

        model = ResourceRelationModel(self._connection)
        for item in to_delete:
            item['org_id'] = org_id
            item['resource_id'] = resource_id
            # удаляем только admin-связи
            item['name'] = relation_name.admin
            model.delete(item)

        for item in to_add:
            item['org_id'] = org_id
            item['name'] = relation_name.admin
            item['resource_id'] = resource_id
            model.create(**item)

    def update_one(self, org_id, group_id, data, allow_empty_admins=False):
        """
        Обновляем одну группу вместе с кешом и счетчиками
        :param org_id:
        :param group_id:
        :param data: словарик - PATCH-data
        """
        data = unfreeze_or_copy(data)

        if 'members' in data:
            members = data.pop('members')
            self._update_members(org_id, group_id, members)
        if 'admins' in data:
            admins = data.pop('admins')
            # не даем удалить всех админов если только это явно не разрешено в allow_empty_admins
            if not admins and not allow_empty_admins:
                raise ConstraintValidationError(
                    'group_should_have_admin',
                    'Group should have at least one admin',
                )
            self._update_admins(org_id, group_id, admins)

        if data:
            old_groups = self.find(filter_data=dict(
                org_id=org_id,
                id=group_id,
                removed=Ignore,
            ))
            # по какой-то причине группа не найдена
            if not old_groups:
                raise ConstraintValidationError(
                    'group_does_not_exist',
                    'Group with id={id} does not exist',
                    id=group_id,
                )

            if 'name' in data:
                data['name_plain'] = make_simple_strings(data['name'])
            if 'description' in data:
                data['description_plain'] = make_simple_strings(data['description'])

            self.update(
                update_data=data,
                filter_data={
                    'id': group_id,
                    'org_id': org_id,
                }
            )

    def get_select_related_data(self, select_related):
        if not select_related:
            return [self.default_all_projection], [], []

        select_related = select_related or []
        projections = set()
        joins = []
        processors = []

        if 'author' in select_related:
            projections.update([
                'groups.author_id',
                'author.id AS "author.id"',
                'author.org_id AS "author.org_id"',
                'author.nickname AS "author.nickname"',
                'author.aliases AS "author.aliases"',
                'author.name AS "author.name"',
                'author.email AS "author.email"',
                'author.gender AS "author.gender"',
                'author.about AS "author.about"',
                'author.birthday AS "author.birthday"',
                'author.contacts AS "author.contacts"',
                'author.position AS "author.position"',
                'author.department_id AS "author.department_id"',
                'author.is_dismissed AS "author.is_dismissed"',
                'author.role AS "author.role"',
            ])
            joins.append("""
            LEFT OUTER JOIN users as author ON (
                groups.author_id = author.id
                and groups.org_id = author.org_id
            )
            """)
            processors.append(set_to_none_if_no_id('author'))

        if 'tracker_license' in select_related:
            projections.update([
                'case when resource_relations.group_id is not null then True else False end as tracker_license',
            ])
            joins.append("""
            LEFT OUTER JOIN resource_relations ON (
                resource_relations.org_id = groups.org_id
                AND resource_relations.group_id = groups.id
            )
            LEFT OUTER JOIN organization_services ON (
                organization_services.resource_id = resource_relations.resource_id
            )
            LEFT OUTER JOIN services ON (
                services.id = organization_services.service_id
                AND services.slug = 'tracker'
            )
            """)

        return projections, joins, processors

    def prefetch_related(self, items, prefetch_related):
        from intranet.yandex_directory.src.yandex_directory.core.models.department import DepartmentModel
        from intranet.yandex_directory.src.yandex_directory.core.models.user import UserModel

        conn = self._connection
        m_departments = DepartmentModel(connection=conn)
        m_resource_relations = ResourceRelationModel(connection=conn)
        m_users = UserModel(connection=conn)
        m_groups = GroupModel(connection=conn)
        m_user_group_memberships = UserGroupMembership(connection=conn)

        if not prefetch_related or not items:
            return

        org_id = set(item['org_id'] for item in items)
        if len(org_id) > 1:
            raise RuntimeError('All groups should belong to one organization')
        org_id, = org_id

        prefetch_members = 'members' in prefetch_related
        prefetch_admins = 'admins' in prefetch_related

        if prefetch_members or prefetch_admins:
            # нас попросили подгрузить список всех непосредственных
            # членов групп. Каждый элемент списка будет представлять словарь:
            # {type: 'group', object: {id: 124, ...}}

            # сначала мы определяем какие отношения есть у каждой из групп
            resource_relations_filter = {
                'org_id': org_id,
                'resource_id__in': [g['resource_id'] for g in items],
            }
            resource_relations = defaultdict(list)
            # получаем все отношения данных групп
            for resource_relation in m_resource_relations.find(filter_data=resource_relations_filter):
                relation = (
                    resource_relation.get('name'),
                    resource_relation.get('user_id'),
                    resource_relation.get('group_id'),
                    resource_relation.get('department_id'),
                )
                resource_relations[resource_relation['resource_id']].append(relation)

            # формируем словарь вида group_id: set(relations)
            group_relations = {}
            for group in items:
                group_relations[group['id']] = set(resource_relations[group['resource_id']])

            flattened = list(chain(*list(group_relations.values())))
            user_ids = [_f for _f in map(itemgetter(1), flattened) if _f]
            group_ids = [_f for _f in map(itemgetter(2), flattened) if _f]
            department_ids = [_f for _f in map(itemgetter(3), flattened) if _f]

            def get_objects_map(iterable):
                return {obj['id']: obj for obj in iterable}

            users = {}
            departments = {}
            groups = {}

            if user_ids:
                users = get_objects_map(
                    m_users.find(
                        filter_data={
                            'org_id': org_id,
                            'id': user_ids
                        },
                        fields=ALL_SIMPLE_FIELDS,
                    )
                )

            if group_ids:
                groups = get_objects_map(
                    m_groups.find(
                        filter_data={
                            'org_id': org_id,
                            'id': group_ids,
                        },
                        fields=ALL_SIMPLE_FIELDS,
                    )
                )

            if department_ids:
                departments = get_objects_map(
                    m_departments.find(
                        filter_data={
                            'org_id': org_id,
                            'id': department_ids,
                        },
                        fields=ALL_SIMPLE_FIELDS,
                    )
                )

            def get_object(relation_name, user_id, group_id, department_id):
                if user_id:
                    return {'type': 'user', 'object': users[user_id]}
                if group_id:
                    return {'type': 'group', 'object': groups[group_id]}
                if department_id:
                    return {'type': 'department', 'object': departments[department_id]}
                raise RuntimeError('This should never happen, but who knows...')

            for group in items:
                relations = group_relations[group['id']]
                if prefetch_members:
                    member_relations = [r for r in relations
                                        if r[0] == relation_name.include]
                    group['members'] = list(starmap(get_object, member_relations))

                if prefetch_admins:
                    admin_relations = [r for r in relations
                                       if r[0] == relation_name.admin]
                    # на всякий случай проверяем, что админская связь
                    # установлена с пользователем а не отделом или
                    # группой
                    admins = [
                        i['object']
                        for i in starmap(get_object, admin_relations)
                        if i['type'] == 'user'
                    ]
                    group['admins'] = admins

        if 'all_users' in prefetch_related:
            # нас попросили создать в каждой группе развернутый список всех
            # сотрудников, которые в неё входят
            group_ids = [item['id'] for item in items]

            # что ж, возьмём их айдишники из кэша
            leafs = m_user_group_memberships.find(
                filter_data={
                    'org_id': org_id,
                    'group_id': group_ids
                },
                fields=['group_id', 'user_id'],
            )
            user_ids = [leaf['user_id'] for leaf in leafs]
            # словарь group_id -> users
            user_lists = defaultdict(list)

            if user_ids:
                # затем по айдишникам вытянем самих пользователей
                users = m_users.find(filter_data={
                    'org_id': org_id,
                    'id': user_ids
                })
                # чтобы удобно было к ним образаться, сделаем такой словарик
                user_by_id = {user['id']: user for user in users}

                # сгруппируем конечных пользователей по group_id
                for leaf in leafs:
                    user_lists[leaf['group_id']].append(
                        user_by_id[leaf['user_id']]
                    )

            for group in items:
                group['all_users'] = user_lists[group['id']]

        if 'email' in prefetch_related:
            group_by_id = {}
            for group in items:
                group_by_id[group['id']] = group

            filter_data = {
                'id': list(group_by_id.keys()),
                'org_id': org_id,
            }
            fields = ['label']
            for group in self.find(
                filter_data=filter_data,
                fields=fields,
            ):
                group_by_id[group['id']]['email'] = build_email(
                    self._connection,
                    group['label'],
                    org_id
                )

        if 'member_of' in prefetch_related:
            for item in items:
                item['member_of'] = self.get_parents(org_id=org_id, group_id=item['id'])

    def get_filters_data(self, filter_data):
        distinct = False

        if filter_data is None:
            filter_data = {}

        # если filter_data меняется внутри функции,
        # то изменится и исходный filter_data-словарь
        filter_data_removed = deepcopy(filter_data)
        if 'removed' not in filter_data_removed:
            filter_data_removed['removed'] = False

        filter_parts = []
        joins = []
        used_filters = []

        if filter_data.get('removed', False) is Ignore:  # убираем из фильтра
            del filter_data_removed['removed']

        if 'removed' in filter_data_removed:
            filter_parts.append(
                self.mogrify(
                    'groups.removed = %(removed)s',
                    {
                        'removed': filter_data_removed.get('removed')
                    }
                )
            )
            used_filters.append('removed')

        # removed - не учитываем при фильтрации внутри filter_data
        # (только внутри filter_data_removed),
        # но добавляем в used_filters, чтоб проходила проверка
        # get_filters_and_check
        if 'removed' in filter_data:
            used_filters.append('removed')

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('id', can_be_list=True) \
            ('org_id') \
            ('resource_id', can_be_list=True) \
            ('external_id') \
            ('type', can_be_list=True) \
            ('maillist_type', can_be_list=True) \
            ('label', can_be_list=True) \
            ('uid', can_be_list=True)

        if 'admin_uid' in filter_data:
            admin_uid = filter_data.get('admin_uid')
            joins.append("""
            LEFT OUTER JOIN resource_relations ON (
                groups.org_id = resource_relations.org_id
            AND
                groups.resource_id = resource_relations.resource_id
            )
            """)
            filter_parts.append(
                self.mogrify(
                    ('resource_relations.name = %(relation_name)s '
                     'AND resource_relations.user_id = %(admin_uid)s'),
                    {
                        'relation_name': relation_name.admin,
                        'admin_uid': admin_uid,
                    }
                )
            )
            used_filters.append('admin_uid')

        if 'suggest' in filter_data:
            # производим полнотекстовый поиск по строке
            # "`Название на русском` +' '+`Название на английском` +' '+  `label команды`"
            # ищем искомый текст по префиксам слов "слово1* AND слово2*'
            condition = (
                "(to_tsvector('simple', "
                "coalesce(groups.name ->> 'ru', '') || ' ' || "
                "coalesce(groups.name ->> 'en', '') || ' ' || "
                "coalesce(groups.label, ''))"
                " @@ to_tsquery('simple', %(text)s))"

            )
            text = prepare_for_tsquery(filter_data.get('suggest'))
            mogrify_condition = self.mogrify(
                condition,
                {'text': text}
            )
            filter_parts.append(mogrify_condition)
            used_filters.append('suggest')

        if 'user_id__in' in filter_data:
            joins.append("""
            LEFT OUTER JOIN user_group_membership ON (
                groups.org_id = user_group_membership.org_id AND groups.id = user_group_membership.group_id
            )
            """)
            filter_parts.append(
                self.mogrify(
                    'user_group_membership.user_id IN %(user_ids)s',
                    {
                        'user_ids': tuple(filter_data.get('user_id__in'))
                    }
                )
            )
            used_filters.append('user_id__in')

        if 'user_id' in filter_data:
            joins.append("""
            LEFT OUTER JOIN user_group_membership ON (
                groups.org_id = user_group_membership.org_id AND groups.id = user_group_membership.group_id
            )
            """)
            filter_parts.append(
                self.mogrify(
                    'user_group_membership.user_id = %(user_id)s',
                    {
                        'user_id': filter_data.get('user_id')
                    }
                )
            )
            used_filters.append('user_id')

        # TODO: надо придумать, как это переназвать, чтобы было понятно,
        # что фильтрация идет именно по ресурсу, с которым группа имеет связь
        #
        # для группы, этот фильтр выбирает только те группы, которые
        # непосредственно связаны с указанным ресурсом,
        # ИЕРАРХИЯ ГРУПП НЕ УЧИТЫВАЕТСЯ
        if 'resource' in filter_data or 'resource_relation_name' in filter_data:
            joins.append("""
            LEFT OUTER JOIN resource_relations ON (
                groups.org_id = resource_relations.org_id
            )
            """)

            # сюда будем складывать фильтры для данного джойна
            top_filters = []

            if 'resource' in filter_data:
                resources = filter_data.get('resource')
                if hasattr(resources, '__iter__') and not isinstance(resources, (str, bytes)):
                    resources = tuple(resources)
                    operator = 'IN'
                else:
                    operator = '='
                top_filters.append(
                    self.mogrify(
                        ('resource_relations.resource_id '
                         '{operator} %(resources)s').format(operator=operator),
                        {'resources': resources}
                    )
                )
                used_filters.append('resource')

            if 'resource_relation_name' in filter_data:
                names = filter_data.get('resource_relation_name')
                if hasattr(names, '__iter__') and not isinstance(names, (str, bytes)):
                    names = tuple(names)
                    operator = 'IN'
                else:
                    operator = '='
                top_filters.append(
                    self.mogrify(
                        ('resource_relations.name '
                         '{operator} %(names)s').format(operator=operator),
                        {'names': names}
                    )
                )
                used_filters.append('resource_relation_name')

            top_filters.append('resource_relations.group_id = groups.id')
            top_filters = ' AND '.join(top_filters)

            filter_parts.append(top_filters)

        if 'alias' in filter_data:
            filter_parts.append(
                self.mogrify(
                    'groups.aliases::jsonb @> %(alias)s',
                    {
                        'alias': json.dumps(filter_data.get('alias'))
                    }
                )
            )
            used_filters.append('alias')

        if 'uid__isnull' in filter_data:
            uid__isnull = 'NULL' if bool(filter_data.get('uid__isnull')) else 'NOT NULL'
            filter_parts.append(
                self.mogrify(
                    'groups.uid IS {}'.format(uid__isnull)
                )
            )
            used_filters.append('uid__isnull')

        return distinct, filter_parts, joins, used_filters

    def get(self,
            group_id=None,
            org_id=None,
            resource_id=None,
            fields=None):
        filter_data = {
            'org_id': org_id
        }
        if group_id is not None:
            filter_data['id'] = group_id

        if resource_id is not None:
            filter_data['resource_id'] = resource_id

        return self.find(
            filter_data=filter_data,
            fields=fields,
            one=True,
        )

    def get_by_external_id(self, org_id, id):
        return self.find(
            filter_data={
                'org_id': org_id,
                'external_id': id,
            },
            one=True,
        )

    def get_parents(self, org_id, group_id):
        """
        Возвращает список родительских групп.
        """
        # todo: перенести это в prefetch_related
        relations = ResourceRelationModel(connection=self._connection).find(
            filter_data=dict(
                org_id=org_id,
                group_id=group_id,
                name=relation_name.include,
            )
        )
        resource_ids = [rel['resource_id'] for rel in relations]

        if resource_ids:
            return GroupModel(connection=self._connection).find(
                filter_data={
                    'org_id': org_id,
                    'resource_id': resource_ids,
                }
            )
        else:
            return []

    def get_subgroups(self, org_id, group=None, group_id=None):
        """Возвращает список дочерних групп.
        """
        conn = self._connection

        if group is None:
            assert group_id is not None
            group = self.get(org_id=org_id, group_id=group_id)

        resource_id = group['resource_id']

        relations = ResourceModel(connection=conn).get_relations(
            org_id=org_id,
            resource_id=resource_id,
            name=relation_name.include,
        )
        subgroup_ids = [rel['group_id']
                        for rel in relations
                        if rel['group_id'] is not None]

        if subgroup_ids:
            return self.find(
                filter_data={
                    'org_id': org_id,
                    'id': subgroup_ids,
                }
            )
        else:
            return []

    def get_all_subgroups(self, org_id, group=None, group_id=None):
        """Возвращает список всех подгрупп, включая вложенные.
        """
        all_groups = []
        childs = self.get_subgroups(
            org_id=org_id,
            group=group,
            group_id=group_id
        )
        all_groups.extend(childs)

        for child in childs:
            all_groups.extend(
                self.get_subgroups(
                    org_id=org_id,
                    group=child,
                )
            )

        return all_groups

    def update_leafs_cache_and_member_count(self, org_id, group_id):
        members_count = self.update_leafs_cache(org_id=org_id, group_id=group_id)
        self.update(
            update_data={
                'members_count': members_count,
            },
            filter_data={
                'id': group_id,
                'org_id': org_id,
            }
        )
        return members_count

    def update_leafs_cache(self, org_id, group_id):
        """
        Обновляет в базе табличку, в которой хранится полный список
        пользователей, входящих в данную группу.

        Для обновления кэша, мы собираем в список ресурсы всех подгрупп,
        включая вложенные, а затем запрашиваем всех пользователей,
        связанных с ресурсами этих подгрупп, а так же c ресурсом текущей группы.
        """
        from intranet.yandex_directory.src.yandex_directory.core.models.user import UserModel
        group = self.get(group_id=group_id, org_id=org_id)
        # Правильно я понимаю, что обновлять кеш для детей не нужно?
        resource_id = group['resource_id']

        connection = self._connection
        uids = UserModel(connection).get_uids_by_resource(
            org_id=org_id,
            resource_service=app.config['DIRECTORY_SERVICE_NAME'],
            resource_external_id=resource_id,
            resource_relation_name=relation_name.include,
        )
        # теперь удаляем весь кэш для этой группы и создаем заново
        # это может быть медленно, но пока и так сойдет
        model = UserGroupMembership(connection=connection)
        to_be_deleted = model.find(
            filter_data={'org_id': org_id, 'group_id': group_id}, fields=('user_id', 'org_id', 'group_id')
        )
        user_list = [{'user_id': uid,'org_id': org_id,'group_id': group_id} for uid in uids]
        # гарантировать один порядок ключей, чтобы корректно считать разницу set
        set_add, set_delete = (
            {(
                ('group_id', x['group_id']), ('org_id', x['org_id']), ('user_id', x['user_id'])
            ) for x in user_list},
            {(
                ('group_id', x['group_id']), ('org_id', x['org_id']), ('user_id', x['user_id'])
            ) for x in to_be_deleted},
        )
        # в формате {(('group_id', 5), ('org_id', 7), ('user_id', 10)), (...)}
        pure_add = [dict(x) for x in set_add - set_delete]
        pure_delete = [dict(x) for x in set_delete - set_add]

        for one_filter in pure_delete:
            model.delete(filter_data=one_filter)
        if pure_add:
            model.bulk_create(pure_add, strategy=Values(on_conflict=Values.do_nothing))

        # а теперь обновляем так же и кэши родителей
        # поскольку их полный состав от этой группы зависит
        parents = self.get_parents(org_id=org_id, group_id=group['id'])
        for parent in parents:
            self.update_leafs_cache(
                org_id=org_id,
                group_id=parent['id']
            )
        filter_data = {
            'org_id': org_id,
            'group_id': group_id,
        }
        return UserGroupMembership(connection=connection).count(filter_data)

    # operations
    # операции над группами

    def add_member(self, org_id, group_id, member, update_leafs_cache=True):
        """Добавляет в группу человека, отдел или вложенную группу.
        """
        group = self.get(group_id=group_id, org_id=org_id)
        resource_id = group['resource_id']

        conn = self._connection
        relations = ResourceRelationModel(conn)
        item = ref_to_relation(member)
        item['org_id'] = org_id
        item['resource_id'] = resource_id
        item['name'] = relation_name.include

        conn = self._connection
        check_objects_exists(conn, org_id, (member, ))
        if member['type'] == 'group':
            # проверяем, что member не включает в себя текущую группу
            if self.is_subgroup(
                org_id=org_id,
                group_id=member['id'],
                subgroup_id=group_id
            ):
                self._raise_cycle_detected_error(
                    group_id=group_id,
                    subgroup_id=member['id']
                )

        try:
            # если не начать тут вложенную транзакцию, то в результате возможной
            # IntegrityError ломается основная транзакция

            with conn.begin_nested():
                relations.create(**item)
        except IntegrityError:
            # если такая связь уже была, то просто игнорируем ошибку
            pass
        else:
            if update_leafs_cache:
                self.update_leafs_cache_and_member_count(org_id=org_id, group_id=group_id)

    def _raise_cycle_detected_error(self, group_id, subgroup_id):
        raise ConstraintValidationError(
            'cycle_detected',
            'Group {subgroup_id} already includes {group_id}',
            group_id=group_id,
            subgroup_id=subgroup_id
        )

    def remove_member(self, org_id, group_id, member, update_leafs_cache=True):
        """Удаляет из группы человека, отдел или вложенную группу.
        """
        group = self.get(group_id=group_id, org_id=org_id)
        resource_id = group['resource_id']

        relations = ResourceRelationModel(self._connection)
        item = ref_to_relation(member)
        item['org_id'] = org_id
        item['resource_id'] = resource_id
        relations.delete(filter_data=item)
        if update_leafs_cache:
            self.update_leafs_cache_and_member_count(org_id=org_id, group_id=group_id)

    def remove_admin_from_all_groups(self, org_id, uid):
        """Удаляет указанного пользователя из администраторов всех команд.

        Тут мы просто удаляем связи всех "командных" ресурсов с этим человеком.
        """
        with log.fields(org_id=org_id, uid=uid):
            log.info('Removing admin from all groups')

            self._connection.execute(
                """DELETE FROM resource_relations
                   USING groups
                WHERE
                      resource_relations.org_id = %(org_id)s AND
                      groups.org_id = %(org_id)s AND
                      resource_relations.resource_id = groups.resource_id AND
                      resource_relations.name = 'admin' AND
                      resource_relations.user_id = %(uid)s
                """,
                {'org_id': org_id, 'uid': uid}
            )

    # операции над группами и пользователями

    def get_all_users(self, org_id, group_id):
        """Разворачивает группу в список всех сотрудников
        """
        from intranet.yandex_directory.src.yandex_directory.core.models.user import UserModel

        # возьмём айдишники пользователей из кэша
        leafs = UserGroupMembership(self._connection).find(
            filter_data={
                'org_id': org_id,
                'group_id': group_id
            }
        )
        user_ids = [leaf['user_id'] for leaf in leafs]

        # затем по айдишникам вытянем самих пользователей
        if user_ids:
            filter_data = {
                'org_id': org_id,
                'id': user_ids,
            }
            return UserModel(self._connection).find(filter_data)
        else:
            return []

    def get_member_groups(self, org_id, member, group_types=None):
        """
        Возвращает список групп, в которые пользователь/департамент/группа
        входит непосредственно.
        member: dict(id=d+, type=['user'|'department'|'group'])
        """
        # todo: test me
        member_id = member['id']
        member_type = member['type'] + '_id'
        connection = self._connection
        relations = ResourceRelationModel(connection).find(
            filter_data={
                'org_id': org_id,
                member_type: member_id,
                'name': relation_name.include,
            }
        )
        resource_ids = [rel['resource_id'] for rel in relations]
        if resource_ids:
            groups = GroupModel(connection).find(
                filter_data={
                    'org_id': org_id,
                    'resource_id': resource_ids,
                }
            )
        else:
            groups = []
        if group_types != None:
            groups = [group for group in groups if group['type'] in group_types]
        return groups

    def get_user_groups(self, org_id, user_id, group_types=None):
        """Отдаёт список групп пользователя, при чём отдаются только
        группы заданного типа.

        Если в group_types передан None (по-умолчанию это так),
        то отдаются все группы.
        """
        user_member = dict(id=user_id, type=TYPE_USER)
        return self.get_member_groups(
            org_id,
            user_member,
            group_types=group_types,
        )

    def set_user_groups(self, org_id, user_id, groups, group_types=(GROUP_TYPE_GENERIC,), update_leafs_cache=True):
        """Изменяет список пользовательских generic групп на
        группы с указанными id. Поле groups должно быть списком
        с id групп.

        По умолчанию, заменяются только группы generic типа.
        """
        current_groups = self.get_user_groups(
            org_id,
            user_id,
            group_types=group_types,
        )
        current_groups = set(only_ids(current_groups))
        new_groups = set(groups)
        to_delete = current_groups - new_groups
        to_add = new_groups - current_groups

        member = {'type': 'user', 'id': user_id}
        for group_id in to_delete:
            self.remove_member(
                org_id=org_id,
                group_id=group_id,
                member=member,
                update_leafs_cache=update_leafs_cache,
            )

        for group_id in to_add:
            self.add_member(
                org_id=org_id,
                group_id=group_id,
                member=member,
                update_leafs_cache=update_leafs_cache,
            )

    def delete(self,
               filter_data=None,
               force=False,
               force_remove_all=False,
               generate_action=True,
               action_author_id=None):
        org_id = filter_data.get('org_id')
        group_id = filter_data.get('id')

        filter_data_del = {
            'org_id': org_id
        }
        if group_id:
            filter_data_del['group_id'] = group_id
            if generate_action:
                group = self.get_extend_info(group_id, org_id)
                from intranet.yandex_directory.src.yandex_directory.core.actions import action_group_delete
                action_group_delete(
                    self._connection,
                    org_id=org_id,
                    author_id=action_author_id,
                    object_value=group,
                )

            # если включен сервис рассылок, удаляем рассылку из паспорта и у себя
            if OrganizationServiceModel(self._connection).is_service_enabled(org_id, MAILLIST_SERVICE_SLUG):
                group = self.filter(org_id=org_id, id=group_id, removed=Ignore).one()
                if group['uid']:
                    app.passport.maillist_delete(group['uid'])
                    force = True

            data = {
                'label': None,
                'members': [],
                'email': None,
            }
            self.update_one(org_id=org_id, group_id=group_id, data=data)

        # Эта группа больше не входит ни в какие другие
        ResourceRelationModel(self._connection).delete(
            filter_data=filter_data_del
        )

        if force:
            # реально удаляем данные из базы
            data_with_remove_ignored = deepcopy(filter_data)
            data_with_remove_ignored['removed'] = Ignore
            # force_remove_all = True не проверяется наличие фильтра для удаления
            super(GroupModel, self).delete(data_with_remove_ignored, force_remove_all=force_remove_all)
        else:
            # пометим группу как удаленную
            self.update_one(org_id=org_id, group_id=group_id, data={'removed': True})


class GroupMemberEnum(PseudoModel):
    primary_key = 'id'
    simple_fields = UserModel.simple_fields | \
                    GroupModel.simple_fields | \
                    DepartmentModel.simple_fields
    all_fields = UserModel.all_fields | \
                 GroupModel.all_fields | \
                 DepartmentModel.all_fields
    nested_fields = join_dicts(
        UserModel.nested_fields,
        GroupModel.nested_fields,
        DepartmentModel.nested_fields,
    )


class GroupMember(PseudoModel):
    primary_key = 'object'
    simple_fields = set(['type', 'object'])
    all_fields = simple_fields
    nested_fields = {'object': GroupMemberEnum}

# Эту псевдо-модель надо поместить в registry,
# чтобы код выборки полей из базы понимал, какие поля
# могут быть у вложенных в группу объектов
_model_registry['GroupMemberEnum'] = GroupMemberEnum
_model_registry['GroupMember'] = GroupMember


class UserGroupMembership(BaseModel):
    """Эта табличка служит для представления всех
    сотрудников, входящих в группу, независимо от того,
    включены они в неё непосредственно, или через членство
    в других группах или отделах.
    """
    db_alias = 'main'
    table = 'user_group_membership'
    order_by = 'user_group_membership.org_id, user_group_membership.user_id, user_group_membership.group_id'
    primary_key = 'user_id'
    all_fields = [
        'org_id',
        'user_id',
        'group_id',
        # not from db
        'groups',
    ]
    select_related_fields = {
        'groups': 'GroupModel',
    }

    def create(self, user_id, org_id, group_id):
        # todo: test me
        return dict(
            self._connection.execute(
                queries.CREATE_USER_GROUP_MEMBERSHIP['query'],
                {
                    'user_id': user_id,
                    'org_id': org_id,
                    'group_id': group_id,
                }
            ).fetchone()
        )

    def get(self, user_id, org_id, group_id):
        # todo: test me
        response = self.find(
            {
                'user_id': user_id,
                'org_id': org_id,
                'group_id': group_id,
            },
            limit=1,
            fields=ALL_FIELDS,
        )
        if response:
            return response[0]

    def get_select_related_data(self, select_related):
        select_related = select_related or []
        projections = set()
        joins = []
        processors = []

        if 'groups' in select_related:
            projections.update([
                'user_group_membership.group_id',
                'user_group_membership.org_id',
                'groups.id as "groups.id"',
                'groups.org_id as "groups.org_id"',
                'groups.description as "groups.description"',
                'groups.name as "groups.name"',
                'groups.email as "groups.email"',
                'groups.label as "groups.label"',
                'groups.type as "groups.type"',
                'groups.created as "groups.created"',
                'groups.members_count as "groups.members_count"',
            ])
            joins.append("""
            LEFT OUTER JOIN groups ON (
                groups.id = user_group_membership.group_id and groups.org_id = user_group_membership.org_id
            )
            """)
        return projections, joins, processors

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        # фильтр по user_id и org_id/group_id позволяет найти
        # группы, в которые входит
        for field in ('user_id', 'org_id', 'group_id'):
            if field in filter_data:
                value = filter_data[field]
                if isinstance(value, (list, tuple)):
                    value = tuple(value)
                    # чтобы исключить ситуцию, когда список почему то пустой
                    # и из-за этого получился невалидный SQL, добавим в него
                    # несуществующий -1 id.
                    if len(value) == 0:
                        value = (-1,)

                    op = 'IN'
                else:
                    op = '='

                filter_parts.append(
                    self.mogrify(
                        'user_group_membership.{field} {op} %(value)s'.format(
                            field=field,
                            op=op
                        ),
                        {'value': value}
                    )
                )
                used_filters.append(field)

        if 'group__type' in filter_data:
            joins.append("""
                LEFT OUTER JOIN groups ON (
                    user_group_membership.group_id = groups.id
                    AND user_group_membership.org_id = groups.org_id
                )
            """)

            filter_parts.append(
                self.mogrify(
                    'groups.type = %(value)s'.format(
                        field=field,
                    ),
                    {'value': filter_data['group__type']},
                )
            )
            used_filters.append('group__type')

        return distinct, filter_parts, joins, used_filters
