# -*- coding: utf-8 -*-

import datetime as dt

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.models.base import BaseModel
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    utcnow,
    generate_id,
)


class invite_type:
    """
    Типы инвайтов
    """
    personal = 'personal'
    department = 'department'


class InviteModel(BaseModel):
    db_alias = 'meta'
    table = 'invites'
    primary_key = 'code'
    order_by = 'code'

    all_fields = [
        'code',
        'org_id',
        'department_id',
        'created_at',
        'last_use',
        'counter',
        'enabled',
        'author_id',
        'invite_type',
        'wait',
        'mail_campaign_slug',
        'service_slug',
        'valid_to',
        'add_license',
    ]

    def create(
            self,
            org_id,
            department_id,
            author_id,
            counter=None,
            invite_type=invite_type.personal,
            wait=5,
            mail_campaign_slug=None,
            service_slug='portal',
            ttl=None,
            add_license=False,
    ):
        """
        Генерация нового приглашения.
        :return: код приглашения
        :rtype: str
        """

        if counter is None:
            counter = app.config['DEFAULT_INVITE_LIMIT']
        if ttl is not None:
            valid_to = utcnow() + dt.timedelta(seconds=int(ttl))
        else:
            valid_to = None

        code = generate_id()

        create_params = {
            'code': code,
            'created_at': utcnow(),
            'org_id': org_id,
            'department_id': department_id,
            'author_id': author_id,
            'invite_type': invite_type,
            'wait': wait,
            'mail_campaign_slug': mail_campaign_slug,
            'service_slug': service_slug,
            'valid_to': valid_to,
            'add_license': add_license,
        }
        if counter:
            create_params['counter'] = int(counter)

        self.insert_into_db(**create_params)
        return code

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts, joins, used_filters = [], [], []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('code', can_be_list=True) \
            ('org_id', can_be_list=True) \
            ('author_id', can_be_list=True) \
            ('last_use') \
            ('created_at') \
            ('enabled') \
            ('invite_type') \
            ('department_id')

        # valid_to is optional, so, we implement __gt/__lt/etc. operators
        # for ourself with taking into account valid_to_opt option
        valid_to_filters = {
            'valid_to__gt': '>',
            'valid_to__lt': '<',
            'valid_to__gte': '>=',
            'valid_to__lte': '<=',
        }

        valid_to_opt = False
        if 'valid_to_opt' in filter_data:
            valid_to_opt = filter_data.get('valid_to_opt')
            used_filters.append('valid_to_opt')

        for filter_name, comp_sym in valid_to_filters.items():
            if filter_name in filter_data:
                dt_value = filter_data.get(filter_name)
                if isinstance(dt_value, dt.datetime):
                    value = dt_value.isoformat()
                else:
                    value = dt_value
                where_stmt = 'valid_to {} %(timestamp)s'.format(comp_sym)
                if valid_to_opt:
                    where_stmt += ' OR valid_to IS NULL'
                filter_parts.append(
                    self.mogrify(where_stmt, {'timestamp': value})
                )
                used_filters.append(filter_name)

        return distinct, filter_parts, joins, used_filters

    def use(self, code):
        """
        Уменьшаем counter на 1. Если при этом counter станет = 0, проставляем прзнак enabled = False
        :param code: код приглашения
        :type code: str
        :param uid: uid использовавшего пользователя
        :type uid: int
        """

        query = """
            UPDATE
                invites
            SET counter = counter - 1,
                last_use = %(last_use)s,
                enabled = CASE WHEN counter = 1 THEN FALSE ELSE TRUE END
            WHERE code=%(code)s
            AND enabled
            AND counter > 0
            """
        self._connection.execute(
            query,
            self.prepare_dict_for_db({
                'code': code,
                'last_use': utcnow(),
                }
            )
        )

    def get(self, code, for_update=False):
        """
        Информация по коду приглашения.
        :param code: код приглашения
        :type code: str
        :return:
        """
        return self.find(filter_data={'code': code}, one=True, for_update=for_update)


class UsedInviteModel(BaseModel):
    db_alias = 'meta'
    table = 'used_invites'
    primary_key = 'code'
    order_by = 'code'

    all_fields = [
        'code',
        'user_id',
        'org_id',
        'created_at',
    ]

    def create(self, code, user_id, org_id):
        """
        Добавление связки "код-приглашенный uid"
        :return: код приглашения
        :rtype: str
        """
        return self.insert_into_db(
            code=code,
            user_id=user_id,
            org_id=org_id,
        )

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts, joins, used_filters = [], [], []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('code', can_be_list=True) \
            ('user_id', can_be_list=True) \
            ('org_id', can_be_list=True) \
            ('created_at') \

        return distinct, filter_parts, joins, used_filters

    def get(self, code, user_id):
        return self.find(
            filter_data={
                'code': code,
                'user_id': user_id,
            },
            one=True,
        )
