# -*- coding: utf-8 -*-

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.models.base import BaseModel
from intranet.yandex_directory.src.yandex_directory.common.utils import utcnow
from intranet.yandex_directory.src.yandex_directory.core.models import ServiceModel
from intranet.yandex_directory.src.yandex_directory.core.models.service import MAILLIST_SERVICE_SLUG


class MaillistCheckModel(BaseModel):
    db_alias = 'main'
    table = 'maillist_checks'
    all_fields = [
        'org_id',
        'revision',
        'ml_is_ok',
        'updated_at',
        'problems',
    ]
    order_by = 'org_id'
    primary_key = 'org_id'

    def create(self, org_id, revision):
        params = dict(
            org_id=org_id,
            revision=revision,
            updated_at=utcnow(),
        )
        maillist_check = self.insert_into_db(
            **params
        )
        return maillist_check

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts, joins, used_filters = [], [], []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('org_id', can_be_list=True) \
            ('ml_is_ok') \
            ('updated_at__lt') \
            ('updated_at__lte') \
            ('updated_at__gt') \
            ('updated_at__gte')

        return distinct, filter_parts, joins, used_filters

    def get_organizations_for_sync(self):
        maillist_service_id = ServiceModel(self._connection).get_by_slug(MAILLIST_SERVICE_SLUG)['id']
        query = """
            SELECT organization_services.org_id, counter.revision as last_revision
            FROM organization_services
            LEFT OUTER JOIN maillist_checks USING(org_id)
            LEFT OUTER JOIN revision_counters as counter USING(org_id)
            INNER JOIN organizations on organizations.id = organization_services.org_id
            WHERE organization_services.service_id = %(maillist_service_id)s
            AND organizations.environment = %(environment)s
            AND organization_services.enabled = TRUE
            AND (maillist_checks.revision is NULL OR
            counter.revision > maillist_checks.revision);
        """
        query = self.mogrify(
            query,
            {
                'maillist_service_id': maillist_service_id,
                'environment': app.config['ENVIRONMENT'],
            }
        )
        result = self._connection.execute(query).fetchall()
        return list(map(dict, result))

    def insert_or_update(self, org_id, data):
        query = '''
        INSERT INTO {0} (org_id, revision, ml_is_ok, updated_at, problems) VALUES
            (%(org_id)s, %(revision)s, %(ml_is_ok)s, NOW(), %(problems)s)
            ON CONFLICT (org_id) DO UPDATE
            SET revision=%(revision)s, updated_at=NOW(), ml_is_ok=%(ml_is_ok)s, problems=%(problems)s;
        '''.format(self.table)
        query = self.mogrify(
            query,
            {
                'org_id': org_id,
                'revision': data.get('revision'),
                'ml_is_ok': data.get('ml_is_ok'),
                'problems': data.get('problems'),
            }
        )
        self._connection.execute(query)
