# -*- coding: utf-8 -*-
import datetime
import json
import base64
import os
import re
import urllib.parse
import collections
from collections import defaultdict
from itertools import groupby
from copy import deepcopy
from datetime import timedelta
from random import random
from threading import RLock
from random import choice
from time import sleep

import six
from cachetools import TTLCache, cached
from flask import g
from sqlalchemy.exc import IntegrityError


from intranet.yandex_directory.src import settings
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_shards_by_org_type,
    get_main_connection,
    get_meta_connection,
    get_shards_with_weight,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    EmptyOrganizationName,
    MasterDomainNotFound,
)
from intranet.yandex_directory.src.yandex_directory.common.models.base import (
    BaseModel,
    PseudoModel,
    get_model,
    set_to_none_if_no_id,
    BaseAnalyticsModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models.domain import DomainModel
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    to_punycode,
    ensure_date,
    utcnow,
    prepare_for_tsquery,
    prepare_for_query,
    Ignore,
    make_simple_strings,
    get_user_id_from_passport_by_login,
)
from intranet.yandex_directory.src.yandex_directory.core.db import queries
from intranet.yandex_directory.src.yandex_directory.core.db.utils import batch_insert
from intranet.yandex_directory.src.yandex_directory.core.exceptions import (
    OrganizationIsWithoutContract,
    OrganizationHasDebt,
    OrganizationAlreadyHasContract,
    OrganizationUnknown,
    OrganizationDeleted,
    PromocodeExpiredException,
    PromocodeInvalidException,
    OrganizationIsNotPartner,
    NoActivePerson,
    TooManyPersonsFromBilling,
)
from intranet.yandex_directory.src.yandex_directory.core.features import get_enabled_organizations_features
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    only_attrs,
    only_ids,
    get_master_domain,
    int_or_none,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.subscription import (
    format_promocode,
    get_promocode_expires_at,
)
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

# дни, когда посылаем напоминания о необходимости оплаты при наличии задолженности

SEND_DEBT_MAIL_DAYS = [14, 21, 28, 29, 30, 31]

DEFAULT_QUERY_TYPE = 'organization'


class subscription_plan:
    paid = 'paid'
    free = 'free'


class vip_reason:
    """
       Признаки VIP организации
    """
    paid = 'paid'
    trial_service = 'trial_service'
    partner = 'partner'
    many_users = 'many_users'
    whitelist = 'whitelist'


class organization_type:
    """
    Типы организаций
    """
    common = 'common'
    education = 'education'
    test = 'test'
    yandex_project = 'yandex_project'
    partner_organization = 'partner_organization'
    portal = 'portal'
    cloud = 'cloud'
    cloud_partner = 'cloud_partner'
    charity = 'charity'

    all_types = [
        common,
        education,
        test,
        yandex_project,
        portal,
        partner_organization,
        cloud,
        cloud_partner,
        charity,
    ]

    # для этих типов не отгружаем инфу в биллинг
    free_types = [
        test,
        yandex_project,
        partner_organization,
        cloud,
        cloud_partner,
        education,
        charity,
    ]

    cloud_types = [
        cloud,
        cloud_partner,
    ]

    partner_types = [
        partner_organization,
        cloud_partner,
    ]

    @classmethod
    def get_type_settings(cls, org_type):
        # подробное описание настроек здесь:
        # https://wiki.yandex-team.ru/ws/obrazovatelnye-i-testovye-organizacii/
        type_settings = {
            cls.common: {},
            cls.test: {},
            cls.portal: {},
            cls.partner_organization: {},
            cls.cloud_partner: {},
            cls.yandex_project: {
                'promocode': app.config['PROMOCODE_FOR_YANDEX_PROJECT_ORG'],
            },
            cls.education: {
                'promocode': app.config['PROMOCODE_FOR_EDUCATION_ORG'],
            },
            cls.cloud: {},
            cls.charity: {
                'promocode': app.config['PROMOCODE_FOR_EDUCATION_ORG'],
            }
        }
        return type_settings.get(org_type)


def get_name_for_organization(name, language):
    """
    """
    if isinstance(name, str):
        return name

    return name.get(language) or name.get('en') or name.get('ru')


def check_has_debt(client_id=None, first_debt_act_date=None, balance=0):
    """
    Проверяет наличие задолженности по номеру договора или дате задолженности и балансу.
    Возвращает True/False и количество дней с даты первого неоплаченного акта.
    """
    days_since_first_act = 0
    if client_id:
        balance_info = app.billing_client.get_balance_info(client_id)
        first_debt_act_date = balance_info['first_debt_act_date']
        balance = balance_info['balance']
    if first_debt_act_date:
        days_since_first_act = (utcnow().date() - first_debt_act_date).days
    # считаем, что если баланс >= 0, то задолженности нет, даже если есть непогашенный акт
    has_debt = days_since_first_act > app.config['BILLING_PAYMENT_TERM'] and balance < 0
    return has_debt, days_since_first_act


class OrganizationDomainsModel(PseudoModel):
    """Эта модель используется исключительно для валидации
    вложенных полей поля 'domains' у организации.
    """
    primary_key = 'master'
    simple_fields = set(['master', 'display', 'all', 'owned'])
    all_fields = simple_fields


class OrganizationSsoSettingsModel(BaseModel):
    db_alias = 'main'
    table = 'organization_sso_settings'
    primary_key = 'org_id'
    order_by = 'org_id'
    all_fields = [
        'org_id',
        'enabled',
        'provisioning_enabled',
        'last_sync'
    ]

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('org_id', can_be_list=True) \
            ('enabled')

        return distinct, filter_parts, joins, used_filters

    def is_enabled(self, org_id):
        return self.filter(org_id=org_id, enabled=True).one() is not None

    def insert_or_update(self, org_id, enabled, provisioning_enabled):
        query = '''
        INSERT INTO {0} (org_id, enabled, provisioning_enabled) VALUES
            (%(org_id)s, %(enabled)s, %(provisioning_enabled)s)
            ON CONFLICT (org_id) DO UPDATE
            SET enabled=%(enabled)s, provisioning_enabled=%(provisioning_enabled)s;
        '''.format(self.table)
        query = self.mogrify(
            query,
            {
                'org_id': org_id,
                'enabled': enabled,
                'provisioning_enabled': provisioning_enabled,
            }
        )
        self._connection.execute(query)

    def update_last_sync_date(self, org_id):
        self._connection.execute(
            '''
                UPDATE {0} SET last_sync = %(last_sync)s WHERE org_id = %(org_id)s
            '''.format(self.table),
            {'last_sync': utcnow(), 'org_id': org_id}
        )

    def get_max_delay(self):
        return self._connection.execute(
            '''SELECT MAX(NOW() - last_sync) FROM {} WHERE enabled = True'''.format(self.table)
        ).fetchall()[0][0]


class OrganizationModel(BaseModel):
    db_alias = 'main'
    table = 'organizations'
    json_fields = [
        'name'
    ]
    all_fields = [
        'id',
        'label',
        'name',
        'admin_uid',
        'ogrn',
        'inn',
        'trc',
        'corr_acc',
        'account',
        'law_address',
        'real_address',
        'head_id',
        'phone_number',
        'fax',
        'email',
        'language',
        'source',
        'header',
        'created',
        'tld',
        'country',
        'disk_usage',
        'disk_limit',
        'subscription_plan',
        'subscription_plan_changed_at',
        'subscription_plan_expires_at',
        'partner_id',
        'billing_info',
        'logo_id',
        'shared_contacts',
        'environment',
        'maillist_type',
        'organization_type',
        'preset',
        'name_plain',
        'user_count',
        'vip',
        'max_group_id',
        'registrar_id',
        'karma',
        'ip',
        'clouds',
        'last_passport_sync',
        'is_sso_enabled',
        'is_provisioning_enabled',
        # не из базы
        'revision',
        'head',
        'services',
        'domains',
        'logo',
        'admin_id',
        'has_owned_domains',
        'root_departments',
        'default_uid',
        'can_users_change_password',
        'is_blocked',
    ]
    select_related_fields = {
        'revision': 'OrganizationRevisionCounterModel',
        'head': 'UserModel',
        'billing_info': 'OrganizationBillingInfoModel',
        'logo': 'ImageSizesModel',
        'admin_id': None,
        'has_owned_domains': None,
        'is_sso_enabled': None,
        'is_provisioning_enabled': None,
    }
    prefetch_related_fields = {
        'services': 'OrganizationServiceModel',
        'domains': 'OrganizationDomainsModel',
        'root_departments': 'DepartmentModel',
        'default_uid': None,
        'can_users_change_password': None,
    }
    field_dependencies = {
        'admin_id': 'admin_uid',
    }
    date_fields = [
        'subscription_plan_expires_at'
    ]

    def create(self,
               id,
               name,
               label,
               admin_uid,
               language,
               clouds=None,
               **additional_fields):
        from intranet.yandex_directory.src.yandex_directory.core.models.event import LastProcessedRevisionModel

        if 'karma' not in additional_fields:
            additional_fields['karma'] = 50

        if clouds:
            additional_fields['clouds'] = ','.join(clouds)

        additional_fields_names = ','.join(list(additional_fields.keys()))
        if additional_fields_names:
            additional_fields_names = ',' + additional_fields_names

        create_placeholder = '%({0})s'.format
        placeholders = list(map(create_placeholder, list(additional_fields.keys())))
        additional_fields_placeholders = ','.join(placeholders)
        if additional_fields_placeholders:
            additional_fields_placeholders = ',' + additional_fields_placeholders

        params = {
            'id': id,
            'label': label,
            'name': name,
            'admin_uid': admin_uid,
            'language': language,
            'environment': app.config['ENVIRONMENT'],
            'name_plain': make_simple_strings(name),
        }
        params.update(additional_fields)

        result = dict(
            self._connection.execute(
                queries.CREATE_ORGANIZATION['query'].format(
                    additional_fields_names=additional_fields_names,
                    additional_fields_placeholders=additional_fields_placeholders
                ),
                self.prepare_dict_for_db(params)
            ).fetchone()
        )
        result['revision'] = 0

        # создадим счетчик ревизий для новой организации
        OrganizationRevisionCounterModel(self._connection).create(org_id=result['id'])
        LastProcessedRevisionModel(self._connection).create(org_id=result['id'])

        return self.remove_private_fields(result)

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('id', can_be_list=True) \
            ('id__gt') \
            ('source') \
            ('subscription_plan') \
            ('organization_type', can_be_list=True) \
            ('environment') \
            ('admin_uid') \
            ('maillist_type') \
            ('user_count') \
            ('preset') \
            ('vip', array=True, cast='text[]') \
            ('registrar_id')

        # Этот фильтр нужен для того, чтобы выбрать организации,
        # в которых не подключен сервис с заданным id
        if 'service_id__not_enabled' in filter_data:
            subquery = """
                NOT EXISTS (
                    SELECT 1
                        FROM organization_services
                       WHERE organizations.id = organization_services.org_id
                         AND organization_services.service_id = %(service_id)s)
            """
            filter_parts.append(
                self.mogrify(
                    subquery,
                    {
                        'service_id': filter_data.get('service_id__not_enabled')
                    }
                )
            )
            used_filters.append('service_id__not_enabled')

        # поиск по текстовым полям, домену, внешнему и внутреннему админу организации
        if 'text' in filter_data:
            query_type = filter_data.get('type') or DEFAULT_QUERY_TYPE
            # DIR-5684 для ускорения работы поиска мы заменили ilike на like при поиске доменов.
            # поэтому переводим наш текст в нижний регистр
            query = filter_data.get('text', '').lower()
            if query_type == 'organization':
                subquery = """
                    organizations.id in (
                        (
                            SELECT domains.org_id as id
                            FROM domains
                            WHERE domains.name like %(domain_in_punycode)s
                            {query_limit}
                        )

                        UNION ALL
                        (
                            SELECT id
                            FROM organizations
                            WHERE make_organization_search_ts_vector(organizations.name_plain, organizations.label) @@ %(organization)s::tsquery
                            {query_limit}
                        )
                    )
                """

                query_limit = str(filter_data.get('limit', ''))
                if query_limit == '':
                    subquery = subquery.format(query_limit='')
                else:
                    subquery = subquery.format(query_limit='LIMIT %(query_limit)s')

                try:
                    domain_in_punycode = to_punycode(query)
                except:
                    domain_in_punycode = query

                domain_in_punycode = prepare_for_query('{}%'.format(domain_in_punycode))  # like

                filter_parts.append(
                    self.mogrify(
                        subquery,
                        {
                            'organization': prepare_for_tsquery(query),
                            'domain_in_punycode': domain_in_punycode,
                            'query_limit': query_limit,
                        }
                    )
                )

            elif query_type == 'internal_admin':
                subquery = """
                    EXISTS (
                        SELECT 1 FROM users
                        INNER JOIN user_group_membership ON user_group_membership.user_id=users.id
                        INNER JOIN groups ON user_group_membership.group_id = groups.id
                        WHERE organizations.id = users.org_id AND
                            NOT users.is_dismissed AND
                            groups.type = 'organization_admin' AND
                            user_group_membership.org_id = organizations.id AND
                            {user_text_search}
                    )
                """
                user_text_search = '(make_user_search_ts_vector2(users.first_name, users.last_name, users.nickname) @@ %(user)s::tsquery)'
                subquery = subquery.format(user_text_search=user_text_search)
                # у нас хранятся логины в нормализованной форме, через -
                user = re.sub(r'\.', '-', query)
                filter_parts.append(
                    self.mogrify(
                        subquery,
                        {
                            'user': prepare_for_tsquery(user),
                        }
                    )
                )

            elif query_type == 'outer_admin':
                uid = get_user_id_from_passport_by_login(query)
                # хак, чтобы фильтр не был пустым, если мы не нашли uid в нашей базе,
                # и чтобы мы гарантированно ничего по этому фильтру не вернули
                filter_data = {'id': (-1,)}
                if uid:
                    with get_meta_connection() as meta:
                        from intranet.yandex_directory.src.yandex_directory.core.models.user import UserMetaModel
                        org_ids = only_attrs(
                            UserMetaModel(meta).find(filter_data={'id': uid}, fields=['org_id']),
                            'org_id'
                        )
                        if org_ids:
                            filter_data = {'id': org_ids}
                self.filter_by(filter_data, filter_parts, used_filters)('id', can_be_list=True)
            else:
                raise RuntimeError('Unknown query type: {0}'.format(query_type))
            used_filters.append('text')
            used_filters.append('type')

        return distinct, filter_parts, joins, used_filters

    def exists(self, id):
        return self.count(filter_data=dict(id=id)) > 0

    def get_update_set_data_params(self, update_data):
        # Если название организации пустая строка - кидаем исключение
        if update_data.get('name') == '':
            raise EmptyOrganizationName()
        return super(OrganizationModel, self).get_update_set_data_params(update_data)

    def update_one(self, org_id, update_data):
        return self.update(
            filter_data={
                'id': org_id,
            },
            update_data=update_data,
        )

    def get(self, id, fields=None):
        return self.find(
            {'id': id},
            fields=fields,
            one=True,
        )

    def get_admins(self, org_id):
        """
        Возвращает список администраторов организации (обычных, не внешних)
        """
        from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel

        group_model = GroupModel(self._connection)
        admin_group = group_model.get_or_create_admin_group(org_id=org_id)
        return group_model.get_all_users(org_id=org_id, group_id=admin_group['id'])

    def get_deputy_admins(self, org_id):
        """
        Возвращает список внутренних заместителей
        """
        from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel

        group_model = GroupModel(self._connection)
        deputy_admin_group = group_model.get_or_create_deputy_admin_group(org_id=org_id)
        return group_model.get_all_users(org_id=org_id, group_id=deputy_admin_group['id'])

    def get_robots(self, org_id):
        """
        Возвращает список роботов организации.
        Теперь возвращаем его не только через группу "Роботы",
        но и через Сервисные роботы
        """
        from intranet.yandex_directory.src.yandex_directory.core.models import GroupModel, RobotServiceModel, UserModel
        robots_uids = set()
        # Выбираем роботов из роботной группы (Зачем? потому что пока что Yamb
        # - это не отдельный сервис Директории, но у Yamb-a есть роботный
        # пользователь, которого можно найти только в роботной группе)
        group_model = GroupModel(self._connection)
        robot_group = group_model.get_robot_group(org_id=org_id)
        if robot_group:
            robot_uids_from_group = only_attrs(
                group_model.get_all_users(
                    org_id=org_id,
                    group_id=robot_group['id']
                ),
                'id'
            )
            robots_uids |= set(robot_uids_from_group)

        # Посмотрим на сервисных роботов (они приоритетнее,
        # а группу <Роботы> когда-нибудь выпилим)
        robot_service_uids = only_attrs(
            RobotServiceModel(self._connection).find(
                filter_data={'org_id': org_id}
            ),
            'uid'
        )

        robots_uids |= set(robot_service_uids)
        # по роботным айдишникам получаем всю информацию о роботах
        if robots_uids:
            filter_data = {
                'org_id': org_id,
                'id': robots_uids,
            }
            return UserModel(self._connection).find(filter_data)

        return []

    def increment_revision(self, org_id):
        """
        Получает номер для следующей ревизии организации
        """
        return OrganizationRevisionCounterModel(self._connection).increment_revision(org_id)

    def get_select_related_data(self, select_related):
        if not select_related:
            return [self.default_all_projection], [], []

        select_related = select_related or []
        projections = set()
        joins = []
        processors = []

        if any(x in select_related for x in ['is_sso_enabled', 'is_provisioning_enabled']):
            if 'is_sso_enabled' in select_related:
                projections.update([
                    'COALESCE(sso_settings.enabled, False) AS is_sso_enabled',
                ])
            if 'is_provisioning_enabled' in select_related:
                projections.update([
                    'COALESCE(sso_settings.provisioning_enabled, False) AS is_provisioning_enabled',
                ])

            sql = """
                LEFT OUTER JOIN organization_sso_settings AS sso_settings ON (
                    organizations.id = sso_settings.org_id
                )
            """
            joins.append(sql)

        if 'revision' in select_related:
            projections.update([
                'organizations.id',
                'counter.revision AS "revision"',
            ])
            sql = """
                LEFT OUTER JOIN revision_counters as counter ON (
                    organizations.id = counter.org_id
                )
            """
            joins.append(sql)

        if 'billing_info' in select_related:
            projections.update([
                'organizations.*',
                'billing_info.org_id AS "billing_info.org_id"',
                'billing_info.client_id AS "billing_info.client_id"',
                'billing_info.person_id AS "billing_info.person_id"',
                'billing_info.is_contract_active AS "billing_info.is_contract_active"',
                'billing_info.balance AS "billing_info.balance"',
                'billing_info.first_debt_act_date AS "billing_info.first_debt_act_date"',
                'billing_info.person_type AS "billing_info.person_type"',
            ])
            sql = """
                LEFT OUTER JOIN organizations_billing_info as billing_info ON (
                    organizations.id = billing_info.org_id
                )
            """
            joins.append(sql)

        if 'head' in select_related:
            head_fields = select_related['head']

            model_class = get_model(self.select_related_fields['head'])
            model_table = model_class.table
            for name in head_fields:
                projections.add(
                    '{table}.{nested_field} AS "{field}.{nested_field}"'.format(
                        table=model_table,
                        field='head',
                        nested_field=name
                    )
                )
            joins.append("""
            LEFT OUTER JOIN users
                         ON (organizations.id = users.org_id AND
                             organizations.head_id = users.id)
            """)

            processors.append(set_to_none_if_no_id('head'))

        if 'logo' in select_related:
            projections.add(
                'images.meta AS logo_meta'
            )
            joins.append("""
            LEFT OUTER JOIN images
                         ON (organizations.id = images.org_id AND
                             organizations.logo_id = images.id)
            """)

            def prepare_organization_logo(organization):
                """
                Оставляет от метаданных картинки только размеры
                и заменяет path на полный URL до картинки.
                """

                def replace_path(size):
                    path = size.pop('path')
                    size['url'] = urllib.parse.urljoin(
                        app.config['MDS_READ_API'],
                        path,
                    )
                    return size

                meta = organization.pop('logo_meta', None)
                if meta:
                    sizes = {
                        size_name: replace_path(size)
                        for size_name, size
                        in list(meta.get('sizes', {}).items())
                    }
                else:
                    sizes = None

                organization['logo'] = sizes
                return organization

            processors.append(prepare_organization_logo)

        if 'admin_id' in select_related:
            def prepare_organization_admin_id(organization):
                organization.pop('admin_id', None)
                organization['admin_id'] = organization.pop('admin_uid', None)
                return organization

            processors.append(prepare_organization_admin_id)

        if 'has_owned_domains' in select_related:
            def prepare_organization_has_owned_domains(organization):
                organization['has_owned_domains'] = self.has_owned_domains(organization['id'])
                return organization

            processors.append(prepare_organization_has_owned_domains)

        return projections, joins, processors



    def prefetch_related(self, items, prefetch_related):
        """Подгружает вложенные объекты.
        """
        from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationServiceModel

        if not prefetch_related or not items:
            return

        org_ids = only_ids(items)

        if 'services' in prefetch_related:
            nested_fields = prefetch_related['services'].copy()
            # ID организации нам понадобится позже, для склейки данных
            nested_fields['org_id'] = True
            services = defaultdict(list)

            find_services_kwargs = {
                'filter_data': {
                    'org_id': org_ids,
                    'enabled': Ignore,
                },
                'fields': nested_fields,
            }
            # Не даём внешнему API увидеть внутренние сервисы.
            if not app.config['INTERNAL']:
                find_services_kwargs['filter_data']['internal'] = False

            organizations_services = OrganizationServiceModel(self._connection).find(**find_services_kwargs)

            for service in organizations_services:
                org_id = service['org_id']
                services[org_id].append(service)

            for organization in items:
                organization['services'] = services.get(organization['id'], [])

        if 'domains' in prefetch_related:
            from intranet.yandex_directory.src.yandex_directory.core.models.domain import DomainModel

            # В результатах всегда есть словарь с доменами для всех переданных
            # организаций. Даже если у организации почему-то нет ни одного.
            domains_by_org_id = {
                org_id: {'all': [], 'owned': [], 'master': None, 'display': None}
                for org_id in org_ids
            }

            from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
                get_domains_from_db_or_domenator,
                DomainFilter,
            )
            with get_meta_connection() as meta_connection:
                domains = get_domains_from_db_or_domenator(
                    meta_connection=meta_connection,
                    domain_filter=DomainFilter(org_id=org_ids),
                    main_connection=self._connection
                )

            for domain in domains:
                domains_info = domains_by_org_id[domain['org_id']]

                domains_info['all'].append(domain['name'])
                if domain.get('master'):
                    domains_info['master'] = domain['name']
                    domains_info['display'] = domain['name']
                if domain.get('owned'):
                    domains_info['owned'].append(domain['name'])

            get_domains = domains_by_org_id.get

            for organization in items:
                organization['domains'] = get_domains(organization['id'])

        if 'root_departments' in prefetch_related:
            from intranet.yandex_directory.src.yandex_directory.core.models import DepartmentModel

            nested_fields = prefetch_related['root_departments'].copy()
            # ID организации нам понадобится позже, для склейки данных
            nested_fields['org_id'] = True

            by_org_id = defaultdict(list)

            domains = DepartmentModel(self._connection) \
                      .filter(org_id=org_ids, parent_id__isnull=True) \
                      .fields(*nested_fields)

            for domain in domains:
                by_org_id[domain['org_id']].append(domain)

            for organization in items:
                organization['root_departments'] = by_org_id.get(organization['id'])


        if 'default_uid' in prefetch_related or 'can_users_change_password' in prefetch_related:
            for organization in items:
                master_domain = get_master_domain(self._connection, organization['id'], domain_is_required=False)
                if master_domain:
                    try:
                        domain_info = self._get_domain_info(master_domain)
                    except RuntimeError:
                        # Не нашли мастер в паспорте, данные у нас разъехались
                        # починим рассинхрон
                        from intranet.yandex_directory.src.yandex_directory.common.crutches import sync_master_domain
                        with get_main_connection(shard=g.shard, for_write=True) as write_main_connection:
                            sync_master_domain(
                                write_main_connection,
                                master_domain,
                                organization['id'],
                            )
                        # и попробуем заново
                        master_domain = get_master_domain(
                            self._connection,
                            organization['id'],
                            domain_is_required=False,
                            force_refresh=True,
                        )
                        if not master_domain:
                            if 'default_uid' in prefetch_related:
                                organization['default_uid'] = None
                            if 'can_users_change_password' in prefetch_related:
                                organization['can_users_change_password'] = None
                            return
                        domain_info = self._get_domain_info(master_domain)

                    domain_info = domain_info['hosted_domains'][0]
                    if 'default_uid' in prefetch_related:
                        organization['default_uid'] = int_or_none(domain_info.get('default_uid'))
                    if 'can_users_change_password' in prefetch_related:
                        options = domain_info.get('options')
                        if options:
                            try:
                                options = json.loads(options)
                            except (ValueError, TypeError):
                                log.warning('Unable to parse json')
                        else:
                            options = {}
                        # по умолчанию can_users_change_password для доменов true
                        # https://wiki.yandex-team.ru/passport/api/bundle/mdapi/
                        # в ответе бб может придти как строка "0", так и число 0, поэтому заворачиваем в int
                        organization['can_users_change_password'] = bool(int(options.get('can_users_change_password', 1)))
                else:
                    if 'default_uid' in prefetch_related:
                        organization['default_uid'] = None
                    if 'can_users_change_password' in prefetch_related:
                        organization['can_users_change_password'] = None

    def _get_domain_info(self, master_domain):
        domain_info = app.blackbox_instance.hosted_domains(domain=master_domain)
        if not domain_info['hosted_domains']:
            log.error('No master domain in Passport')
            raise RuntimeError('No master domain in Passport')
        return domain_info

    def enable_paid_mode_for_initiated_in_billing_organization(self, org_id, author_id):
        """
        Включает платный режим для заведенной в Биллинге организации.
        Мы уже знаем client_id для неё, остаётся только перевести её в платный режим.
        """
        with log.name_and_fields('billing', org_id=org_id, author_id=author_id):
            log.info('Trying to enable paid mode for organization')
            # если в биллинге организация уже есть, просто переключаем её в платный режим
            # и генерируем событие его переключения

            # проверяем, что можно включать платный режим
            self._check_can_create_contract_info_or_reenable_paid_mode(org_id, re_enabling_paid_mode=True)
            self._save_paid_subscription_plan(org_id=org_id, author_id=author_id)

    def create_contract_info_for_natural_person(self,
                                                org_id,
                                                author_id,
                                                first_name,
                                                last_name,
                                                middle_name,
                                                phone,
                                                email):
        """
        Создаёт необходимые сущности в Биллинге и сохраняет полученные id клиента и плательщика в
        OrganizationBillingInfoModel.
        """
        # пока мы не умеем изменять способ подключения Биллинга
        # (например, если клиент уже подключился как физ.лицо и хочет переподключиться как юр.лицо)
        log_fields = {
            'org_id': org_id,
            'author_id': author_id,
            'first_name': first_name,
            'last_name': last_name,
            'middle_name': middle_name,
            'phone': phone,
            'email': email,
        }
        with log.name_and_fields('billing', **log_fields):
            log.info('Trying to create billing info for natural person')
            # проверяем, что можно заводить биллинговую информацию и включать платный режим
            self._check_can_create_contract_info_or_reenable_paid_mode(
                org_id,
                re_enabling_paid_mode=False
            )

            organization = self.get(org_id)

            client_id = app.billing_client.get_free_user_client(author_id)
            if not client_id:
                log.info('Creating client in Billing')
                client_id = app.billing_client.create_natural_person_client(
                    uid=author_id,
                    name=get_name_for_organization(organization['name'], organization['language']),
                    email=email,  # todo: поменять email на email отдельной рассылки?
                    phone=phone,
                )

            log.info('Creating person in Billing')
            person_id = app.billing_client.create_natural_person(
                uid=author_id,
                client_id=client_id,
                first_name=first_name,
                last_name=last_name,
                middle_name=middle_name,
                phone=phone,
                email=email,
            )

            log.info('Creating offer in Billing')
            app.billing_client.create_offer(
                uid=author_id,
                client_id=client_id,
                person_id=person_id,
            )

            self._complete_saving_billing_info(
                org_id=org_id,
                author_id=author_id,
                client_id=client_id,
                person_type='natural',
                contract_type='offer',
                person_id=person_id,
            )

    def create_contract_info_for_legal_person(self,
                                              org_id,
                                              author_id,
                                              long_name,
                                              phone,
                                              email,
                                              postal_code,
                                              postal_address,
                                              legal_address,
                                              inn,
                                              kpp,
                                              bik,
                                              account,
                                              contract=False):
        """
        Создаёт необходимые сущности в Биллинге и сохраняет полученные id клиента и плательщика в
        OrganizationBillingInfoModel.
        """
        # пока мы не умеем изменять способ подключения Биллинга
        # (например, если клиент уже подключился как физ.лицо и хочет переподключиться как юр.лицо)
        log_fields = {
            'org_id': org_id,
            'author_id': author_id,
            'long_name': long_name,
            'phone': phone,
            'postal_code': postal_code,
            'postal_address': postal_address,
            'legal_address': legal_address,
            'inn': inn,
            'kpp': kpp,
            'bik': bik,
            'account': account,
            'email': email,
            'contract': contract,
        }
        with log.name_and_fields('billing', **log_fields):
            log.info('Trying to create billing info for legal person')
            # проверяем, что можно заводить биллинговую информацию
            self._check_can_create_contract_info_or_reenable_paid_mode(org_id, re_enabling_paid_mode=False)

            organization = self.get(org_id, fields=['name', 'language'])
            organization_name = get_name_for_organization(
                organization['name'],
                organization['language'],
            )

            client_id = app.billing_client.get_free_user_client(author_id)
            if not client_id:
                client_id = app.billing_client.create_legal_person_client(
                    uid=author_id,
                    name=organization_name,
                    email=email,
                    phone=phone,
                )

            person_id = app.billing_client.create_legal_person(
                uid=author_id,
                client_id=client_id,
                name=organization_name,
                long_name=long_name,
                phone=phone,
                email=email,
                postal_code=postal_code,
                postal_address=postal_address,
                legal_address=legal_address,
                inn=inn,
                kpp=kpp,
                bik=bik,
                account=account,
            )

            if contract:
                # создаём договор в Биллинге, клиент должен будет его подписать
                app.billing_client.create_contract(
                    uid=author_id,
                    client_id=client_id,
                    person_id=person_id,
                )
            else:
                # создаём оферту в Биллинге
                app.billing_client.create_offer(
                    uid=author_id,
                    client_id=client_id,
                    person_id=person_id,
                )

            self._complete_saving_billing_info(
                org_id=org_id,
                author_id=author_id,
                client_id=client_id,
                person_type='legal',
                contract_type='contract' if contract else 'offer',
                person_id=person_id,
            )

    def _complete_saving_billing_info(self,
                                      org_id,
                                      author_id,
                                      client_id,
                                      person_type,
                                      contract_type,
                                      person_id,
                                      ):
        log_fields = {
            'org_id': org_id,
            'author_id': author_id,
            'client_id': client_id,
            'contract_type': contract_type,
            'person_type': person_type,
            'person_id': person_id,
        }
        with log.name_and_fields('billing', **log_fields):
            log.info('Saving billing data to DB')
            if OrganizationBillingInfoModel(self._connection).get(org_id):
                # В базе уже есть инфа по данной организации, делать ничего не нужно
                return
            OrganizationBillingInfoModel(self._connection).create(
                org_id=org_id,
                client_id=client_id,
                person_type=person_type,
                contract_type=contract_type,
                person_id=person_id,
            )
            OrganizationModel(self._connection).increment_revision(
                org_id=org_id,
            )

    def enable_paid_mode_for_legal_person(self,
                                          org_id,
                                          author_id,
                                          long_name,
                                          phone,
                                          email,
                                          postal_code,
                                          postal_address,
                                          legal_address,
                                          inn,
                                          kpp,
                                          bik,
                                          account,
                                          contract=False):
        """
        Включает платный режим в организации для юридических лиц.
        """
        self.create_contract_info_for_legal_person(
            org_id,
            author_id,
            long_name,
            phone,
            email,
            postal_code,
            postal_address,
            legal_address,
            inn,
            kpp,
            bik,
            account,
            contract,
        )
        self._save_paid_subscription_plan(org_id=org_id, author_id=author_id)

    def enable_paid_mode_for_natural_person(self,
                                            org_id,
                                            author_id,
                                            first_name,
                                            last_name,
                                            middle_name,
                                            phone,
                                            email):
        """
        Включает платный режим в организации для физических лиц.
        """
        self.create_contract_info_for_natural_person(
            org_id,
            author_id,
            first_name,
            last_name,
            middle_name,
            phone,
            email,
        )
        self._save_paid_subscription_plan(org_id=org_id, author_id=author_id)

    def enable_paid_mode_for_partner_organization(
            self,
            org_id,
            author_id,
            expires_at
    ):
        self._save_paid_subscription_plan(
            org_id=org_id,
            author_id=author_id,
            expires_at=expires_at,
        )

    def create_contract_info_for_existing_person(self,
                                                 org_id,
                                                 person_id,
                                                 author_id,
                                                 person_type):

        client_id = app.billing_client.get_and_valid_user_client(author_id)
        log_fields = {
            'org_id': org_id,
            'client_id': client_id,
            'person_id': person_id,
            'author_id': author_id,
            'person_type': person_type,
        }
        with log.name_and_fields('billing', **log_fields):
            log.info('Creating offer in Billing for existing client_id and person_id')
            app.billing_client.create_offer(
                uid=author_id,
                client_id=client_id,
                person_id=person_id,
            )

            self._complete_saving_billing_info(
                org_id=org_id,
                author_id=author_id,
                client_id=client_id,
                person_type=person_type,
                contract_type='offer',
                person_id=person_id,
            )

    def enable_paid_mode_for_existing_client_id(self,
                                                org_id,
                                                person_id,
                                                author_id,
                                                person_type):
        """
        Создает оферту в биллинге для существующих client_id и person_id и сохраняет данные в
        OrganizationBillingInfoModel.
        """
        self.create_contract_info_for_existing_person(org_id, person_id, author_id, person_type)
        self._save_paid_subscription_plan(org_id=org_id, author_id=author_id)

    def _check_can_create_contract_info_or_reenable_paid_mode(self, org_id, re_enabling_paid_mode):
        """
        Проверяет, что организация может завести биллинговую информацию и включить платный режим.

        Если платный режим включается повторно, нужно проверить,
        что у организации есть договор и нет задолженности.

        Если контракт создается первый раз (первый раз включается платный режим), проверяем только отсутствие информации
        об уже подписанных оферте или договоре (наличие записи OrganizationBillingInfoModel).
        """
        billing_info = OrganizationBillingInfoModel(self._connection).get(
            org_id=org_id,
            fields=['client_id'],
        )
        if re_enabling_paid_mode and billing_info:
            # повторное включение и есть биллинговая информация, проверяем наличие задолженности
            if check_has_debt(client_id=billing_info['client_id'])[0]:
                raise OrganizationHasDebt()
        elif re_enabling_paid_mode and not billing_info:
            # повторное включение и нет биллинговой информации
            raise OrganizationIsWithoutContract()
        elif not re_enabling_paid_mode and billing_info:
            # создание контракта и есть биллинговая информация
            raise OrganizationAlreadyHasContract()

    def has_preset(self, org_id, preset_name):
        return self.filter(
            id=org_id,
            preset=preset_name,
        ).count() > 0

    def is_partner_organization(self, org_id):
        return self.filter(
            id=org_id,
            organization_type=organization_type.partner_types
        ).count() > 0

    def is_cloud_organization(self, org_id):
        return self.filter(
            id=org_id,
            organization_type=organization_type.cloud_types
        ).count() > 0

    def require_partner(self, org_id):
        if not self.is_partner_organization(org_id):
            raise OrganizationIsNotPartner()

    def _save_paid_subscription_plan(self, org_id, author_id, expires_at=None):
        """
        Метод проставляет subscription_plan в базе равным paid и генерирует необходимое событие.

        Должен вызываться только при наличии соответствующей записи OrganizationBillingInfoModel
        """
        from intranet.yandex_directory.src.yandex_directory.core.actions import action_organization_subscription_plan_change
        from intranet.yandex_directory.src.yandex_directory.core.models.disk_usage import DiskUsageModel

        with log.fields(org_id=org_id, author_id=author_id):
            log.info('Trying to enable paid mode')

            has_billing_info = OrganizationBillingInfoModel(self._connection) \
                .count(filter_data={'org_id': org_id}) > 0
            is_partner_organization = self.is_partner_organization(org_id)
            if not has_billing_info and not is_partner_organization:
                raise OrganizationIsWithoutContract()

            updated_organization = dict(
                self.update(
                    filter_data={
                        'id': org_id,
                    },
                    update_data={
                        'subscription_plan': subscription_plan.paid,
                        'subscription_plan_changed_at': utcnow(),
                        'subscription_plan_expires_at': expires_at,
                    },
                )
            )

            # пересчитаем количество доступного места для организации
            DiskUsageModel(self._connection).update_organization_limits(org_id)

            action_organization_subscription_plan_change(
                self._connection,
                org_id=org_id,
                author_id=author_id,
                object_value=updated_organization,
                content={'subscription_plan': subscription_plan.paid},
            )
            log.info('Paid mode has been enabled')

    def get_balance(self, org_id):
        organization_billing_info = OrganizationBillingInfoModel(self._connection).get(
            org_id=org_id,
            fields=['client_id'],
        )
        if organization_billing_info:
            return app.billing_client.get_balance_info(organization_billing_info['client_id'])['balance']

    def disable_paid_mode(self, org_id, author_id=None):
        from intranet.yandex_directory.src.yandex_directory.core.actions import action_organization_subscription_plan_change
        from intranet.yandex_directory.src.yandex_directory.core.models.disk_usage import DiskUsageModel

        organization = self.get(
            org_id,
            fields=['admin_uid', 'subscription_plan_changed_at', 'subscription_plan'],
        )
        if organization['subscription_plan'] == subscription_plan.free:
            log.info('Organization already on free subscription_plan')
            return

        if author_id is None:
            author_id = organization['admin_uid']

        with log.name_and_fields('billing', org_id=org_id, author_id=author_id):
            log.info('Disabling paid mode for organization')

            # подсчитаем количество потребленных услуг, чтобы выключить платный режим
            consumed_info_model = OrganizationBillingConsumedInfoModel(self._connection)
            gaps = consumed_info_model.find_gaps_for_organization(
                from_date=organization['subscription_plan_changed_at'],
                to_date=utcnow(),
                org_id=org_id,
            )
            for date in gaps:
                OrganizationBillingConsumedInfoModel(self._connection).calculate(
                    org_id=org_id,
                    for_date=date,
                    subscription_plan_changed_at=organization['subscription_plan_changed_at'],
                )

            updated_organization = dict(
                self.update(
                    filter_data={
                        'id': org_id,
                    },
                    update_data={
                        'subscription_plan': subscription_plan.free,
                        'subscription_plan_changed_at': utcnow(),
                    }
                )
            )

            # пересчитаем количество доступного места для организации
            DiskUsageModel(self._connection).update_organization_limits(org_id)

            action_organization_subscription_plan_change(
                self._connection,
                org_id=org_id,
                author_id=author_id,
                object_value=updated_organization,
                content={'subscription_plan': subscription_plan.free},
            )
            log.info('Paid mode has been disabled')
            app.stats_aggregator.inc('paid_mode_disabled_summ')

    def get_url_for_paying_trust(self, org_id, author_id, amount):
        """
        Формирует ссылку, которую фронт открывает в iframe для оплаты (пополнения баланса) в
        Трасте
        """
        organization_billing_info = self._get_org_billing_data_for_payment(org_id)

        invoice_request = self._create_billing_request(
            author_id=author_id,
            amount=amount,
            client_id=organization_billing_info['client_id'],
        )

        payment_methods = app.billing_client.get_request_payment_methods(
            uid=author_id,
            request_id=invoice_request['RequestID'],
        )
        contract_id = self._extract_contract_id(
            payment_methods=payment_methods,
            person_id=organization_billing_info['person_id'],
        )
        if not contract_id:
            OrganizationBillingInfoModel(self._connection).remove_billing_info(
                org_id=org_id,
            )
            raise NoActivePerson()

        response = app.billing_client.create_pay_request(
            uid=author_id,
            request_id=invoice_request['RequestID'],
            person_id=organization_billing_info['person_id'],
            contract_id=contract_id,
        )
        return response['payment_url']

    def _extract_contract_id(self, payment_methods, person_id):
        for payment_method in payment_methods:
            if (payment_method['currency'] == 'RUB'
                    and payment_method['person_id'] == int(person_id)):
                return payment_method['contract_id']

    def _get_org_billing_data_for_payment(self, org_id):
        # нельзя внести денег на счет организации, которая не подключена к Биллингу
        organization_billing_info = OrganizationBillingInfoModel(self._connection).get(org_id=org_id)
        if not organization_billing_info:
            raise OrganizationIsWithoutContract()
        return organization_billing_info

    def _create_billing_request(self, author_id, amount, client_id, return_path=None):
        # делаем человека, который хочет платить - "представителем клиента"
        # другие не смогут платить даже если получат ссылку
        app.billing_client.create_client_user_association_if_needed(
            uid=author_id,
            user_uid=author_id,
            client_id=client_id,
        )
        # product_id пока что берем BILLING_PRODUCT_ID_RUB,
        # т.к. у нас нет нерезидентов
        app.billing_client.create_or_update_order(
            uid=author_id,
            client_id=client_id,
            product_id=app.config['BILLING_PRODUCT_ID_RUB'],
        )
        return app.billing_client.create_invoice_request(
            uid=author_id,
            client_id=client_id,
            quantity=amount,
            return_path=return_path,
        )

    def get_url_for_paying(self, org_id, author_id, amount, return_path=None):
        """
        Создает заказ в Биллинге на указанную сумму и возвращает пользователю ссылку на оплату,
        метод идемпотентный
        """
        organization_billing_info = self._get_org_billing_data_for_payment(org_id)

        response = self._create_billing_request(
            author_id=author_id,
            amount=amount,
            return_path=return_path,
            client_id=organization_billing_info['client_id'],
        )
        # добавляем в очередь задачу проверки баланса, чтобы не дожидаться крона
        try:
            from intranet.yandex_directory.src.yandex_directory.core.billing.tasks import CheckOrganizationBalanceTask
            CheckOrganizationBalanceTask(self._connection).delay(org_id=org_id, start_in=timedelta(minutes=3))
            CheckOrganizationBalanceTask(self._connection).delay(org_id=org_id, start_in=timedelta(minutes=5))
        except:
            log.trace().warning('Failed to create check balance tasks')
        return response['UserPath']

    def create_invoice(self, org_id, author_id, amount):
        organization_billing_info = self._get_org_billing_data_for_payment(org_id)

        invoice_request = self._create_billing_request(
            author_id=author_id,
            amount=amount,
            client_id=organization_billing_info['client_id'],
        )

        request_choices = app.billing_client.get_request_choices(
            uid=author_id,
            request_id=invoice_request['RequestID'],
        )
        paysys_id, contract_id = self._get_create_invoice_data(
            request_choices=request_choices,
            payment_method_code='bank',
            person_id=organization_billing_info['person_id'],
        )
        if not paysys_id:
            OrganizationBillingInfoModel(self._connection).remove_billing_info(
                org_id=org_id,
            )
            raise NoActivePerson()

        if not contract_id:
            OrganizationBillingInfoModel(self._connection).remove_billing_info(
                org_id=org_id,
            )
            raise NoActivePerson()

        return app.billing_client.create_invoice(
            uid=author_id,
            request_id=invoice_request['RequestID'],
            paysys_id=paysys_id,
            contract_id=contract_id,
            person_id=organization_billing_info['person_id'],
        )

    def _get_create_invoice_data(self, request_choices, payment_method_code, person_id):
        for person_data in request_choices['pcp_list']:
            if person_data['person']['id'] == person_id:
                for payment_data in person_data['paysyses']:
                    if payment_data['payment_method_code'] == payment_method_code:
                        return payment_data['id'], person_data['contract']['id']
        return None, None

    def get_pricing_info(self, org_id):
        """
        Возвращает информацию о ценах организации:

        {
            'total': 1000,
            'total_with_discount': 500,
            'currency': 'RUB',
            'services': {
                'connect': {
                    'per_user': 199,
                    'per_user_with_discount': 50,
                    'total': 1990,
                    'total_with_discount': 500,
                    'users_count': 10,
                },
            },
            'promocode': {
                'id': 'CONNECT_50',
                'expires': '2017-01-01',
            }  # or None
        }
        """
        from intranet.yandex_directory.src.yandex_directory.core.models.service import OrganizationServiceModel
        total_price = 0
        total_price_with_discount = 0
        total_users_count = 0
        services = {}

        promocode = OrganizationPromocodeModel(self._connection).get_active_for_organization(
            org_id=org_id,
            fields=['promocode_id', 'expires_at'],
        )

        for service in list(app.config['BILLING_PRODUCT_IDS_FOR_SERVICES'].keys()):
            services[service] = self.get_price_info_for_service(
                org_id=org_id,
                service_slug=service,
                promocode_id=promocode['promocode_id'] if promocode else None,
            )
            # проверяем, что трекер включен  и есть лицензии
            tracker_enabled_with_licenses = OrganizationServiceModel(self._connection).get_org_services_with_licenses(
                org_id=org_id,
                has_user_licenses=True,
                service_slug='tracker',
            )
            # TODO: выпилить и сделать нормальный расчет цены
            # костыль, чтобы учитывать цену за трекер, только если он включен
            if service != 'tracker' or tracker_enabled_with_licenses:
                total_price += services[service]['total']
                if services[service]['total_with_discount'] is not None:
                    total_price_with_discount += services[service]['total_with_discount']
                else:
                    total_price_with_discount += services[service]['total']
            total_users_count += services[service]['users_count']

        # отдаем None, если нет промокода или цена со скидкой равна цене без скидки
        # проверяем количество пользователей, чтобы не возвращать None и
        # отображать на фронте скидки, если в организации нет пользователей
        if not promocode or (total_users_count and total_price_with_discount == total_price):
            total_price_with_discount = None

        connect_upgrade = self.get_price_info_for_service(
            org_id=org_id,
            service_slug='connect_upgrade',
            promocode_id=promocode['promocode_id'] if promocode else None,
        )

        if promocode:
            with get_meta_connection() as meta_connection:
                promocode = PromocodeModel(meta_connection).get(promocode['promocode_id'])
            # если промокод внутренний, то не будем показывать его на фронте
            if promocode['promocode_type'] == promocode_type.internal:
                promocode = None
            else:
                promocode = format_promocode(self._connection, promocode, org_id)

        return {
            'currency': 'RUB',  # пока что у нас только рублёвые организации, пока нет нерезидентов
            'services': services,
            'total': total_price,
            'total_with_discount': total_price_with_discount,
            'promocode': promocode,
            'connect_upgrade': connect_upgrade,
        }

    def get_price_info_for_service(self, org_id, service_slug, promocode_id):
        from intranet.yandex_directory.src.yandex_directory.core.models.service import UserServiceLicenses
        """
        Получает информацию о стоимости сервиса для данной организации:
        {
            'per_user': 199,
            'total': 1990,
            'users_count': 10,
        }
        """

        if service_slug == 'connect_upgrade':  # фейковый сервис для информации о стоимости повышения тарифа
            users_count = self.get_users_count_for_billing(org_id)
            service_slug = 'connect'
        elif service_slug == 'connect':
            current_subscription_plan = self.get(org_id, fields=['subscription_plan'])['subscription_plan']
            if current_subscription_plan == subscription_plan.free:
                users_count = 0
            else:
                users_count = self.get_users_count_for_billing(org_id)
        elif service_slug in ('tracker', 'disk'):
            users_count = UserServiceLicenses(self._connection).count(
                filter_data={
                    'org_id': org_id,
                    'service_slug': service_slug,
                },
                distinct_field='user_id'
            )
        else:
            users_count = 0

        if service_slug == 'tracker' and app.config['NEW_TRACKER_PRICING']:
            from intranet.yandex_directory.src.yandex_directory.core.billing.utils import get_price_for_users

            total, total_with_discount = get_price_for_users(
                users_count=users_count,
                promocode_id=promocode_id,
            )

            # в новой схеме цену за пользователя не считаем
            per_user = 0
            per_user_with_discount = 0

            if users_count and total_with_discount == total:
                total_with_discount = None

        else:
            price_info = get_price_and_product_id_for_service(users_count, service_slug, promocode_id)

            total = price_info['price'] * users_count
            total_with_discount = None
            if price_info['price_with_discount'] is not None:
                total_with_discount = price_info['price_with_discount'] * users_count
            # проверяем количество пользователей, чтобы возвращать цену со скидкой, если их нет
            if users_count and total_with_discount == total:
                total_with_discount = None
            per_user = price_info['price']
            per_user_with_discount = price_info['price_with_discount']

        return {
            'total': total,
            'total_with_discount': total_with_discount,
            'per_user': per_user,
            'per_user_with_discount': per_user_with_discount,
            'users_count': users_count,
        }

    def get_users_count_for_billing(self, org_id):
        """
        Считает количество платных пользователей в организации
        """
        from intranet.yandex_directory.src.yandex_directory.core.models import UserModel

        return UserModel(self._connection).count(
            filter_data={
                'org_id': org_id,
                'is_robot': False,
                'is_dismissed': False,
            },
        )

    def activate_promocode(self, org_id, promocode_id, author_id, internal_activation=True):
        OrganizationPromocodeModel(self._connection).activate_for_organization(
            org_id=org_id,
            promocode_id=promocode_id,
            author_id=author_id,
            internal_activation=internal_activation,
        )

    def deactivate_promocode(self, org_id, author_id):
        OrganizationPromocodeModel(self._connection).deactivate_for_organization(
            org_id=org_id,
            author_id=author_id,
        )

    def get_tld_for_email(self, org_id):
        """
        TLD для организации ящиков в организациий
        Возвращает tld из списка поддерживаемых.
        Если tld на котором была заведена организация, пока не поддерживается, то  tld по умолчанию.
        (настройка DEFAULT_TLD)
        :param org_id: ид организации
        :return: tld
        :rtype: str
        """
        cache_key = '_tld_for_email_%s' % org_id
        if not getattr(g, cache_key, None):
            org_info = self.get(org_id, fields=['tld'])
            tld = org_info['tld']
            if tld not in app.config['PORTAL_TLDS']:
                tld = app.config['DEFAULT_TLD']

            setattr(g, cache_key, tld)

        return getattr(g, cache_key)

    def get_contract_print_form(self, org_id):
        organization_billing_info = OrganizationBillingInfoModel(self._connection).get(
            org_id=org_id,
            fields=['client_id', 'contract_type'],
        )
        if organization_billing_info and organization_billing_info['contract_type'] == 'contract':
            print_form = app.billing_client.get_contract_print_form(
                organization_billing_info['client_id']
            )
            print_form = base64.b64decode(print_form).decode('utf-8')
            return print_form

    def get_organization_type(self, org_id):
        data = self.filter(id=org_id).fields('organization_type').one()
        if data:
            return data['organization_type']

    def change_organization_type(
            self, org_id, new_org_type, author_id=None,
            client_id=None, partner_id=None, person_id=None,
    ):
        from intranet.yandex_directory.src.yandex_directory.core.actions import action_organization_type_change

        if new_org_type not in organization_type.all_types:
            raise ValueError('Unknown organization type: {}'.format(new_org_type))
        organization = self.get(id=org_id)
        current_type = organization['organization_type']
        if client_id:
            # иногда нужно сменить и client_id тоже
            if OrganizationBillingInfoModel(self._connection).get(org_id):
                OrganizationBillingInfoModel(self._connection).update_one(org_id, {'client_id': client_id})
            else:
                if not person_id:
                    persons_id = app.billing_client.get_client_persons(client_id)
                    if not persons_id or len(persons_id) > 1:
                        # такого сейчас быть не должно
                        raise TooManyPersonsFromBilling()
                    person_id = persons_id[0]['ID']

                self._complete_saving_billing_info(
                    org_id=org_id,
                    author_id=author_id,
                    client_id=client_id,
                    person_type='legal',
                    contract_type='offer',
                    person_id=person_id,
                )

        if current_type != new_org_type:
            current_type_settings = organization_type.get_type_settings(current_type)
            new_type_settings = organization_type.get_type_settings(new_org_type)

            if current_type_settings.get('promocode'):
                self.deactivate_promocode(org_id, author_id=None)

            if new_type_settings.get('promocode'):
                self.activate_promocode(
                    org_id=org_id,
                    promocode_id=new_type_settings['promocode'],
                    author_id=None,
                )
            if new_type_settings.get('billing_product'):
                # единоразово складываем в yt данные о потребленных продуктах в количестве, указанном в настройках
                # с защитой от многоразового складывания в один день
                if not OrganizationBillingConsumedInfoModel(self._connection).filter(
                        org_id=org_id,
                        service=new_type_settings['billing_product']['name'],
                        for_date=utcnow().date(),
                ).one():
                    OrganizationBillingConsumedInfoModel(self._connection).create(
                        org_id=org_id,
                        total_users_count=new_type_settings['billing_product']['count'],
                        service=new_type_settings['billing_product']['name'],
                        for_date=utcnow().date(),
                        organization_type=new_org_type,
                    )

            update_data = {'organization_type': new_org_type}
            if partner_id:
                update_data['partner_id'] = partner_id

            updated_organization = dict(
                self.update(
                    filter_data={'id': org_id},
                    update_data=update_data
                )
            )

            action_organization_type_change(
                self._connection,
                org_id=org_id,
                author_id=author_id,
                object_value=updated_organization,
                old_object=organization,
                content={'organization_type': new_org_type},
            )

    def remove_all_data_for_organization(self, org_id):
        """Удаляем организацию и все её данные"""

        from intranet.yandex_directory.src.yandex_directory.core.models import (
            ActionModel,
            DepartmentModel,
            DomainModel,
            EventModel,
            GroupModel,
            OrganizationServiceModel,
            ResourceModel,
            ResourceRelationModel,
            RobotServiceModel,
            UserDismissedModel,
            UserGroupMembership,
            UserMetaModel,
            UserModel,
        )

        with get_meta_connection(for_write=True) as meta_connection, \
                meta_connection.begin_nested():

            filter_data = {'org_id': org_id}
            kwargs_for_force_delete = {
                'filter_data': filter_data,
                'force': True,
            }
            UserMetaModel(meta_connection).delete(filter_data=filter_data)
            # для обхода ссылочной целостности
            # удалим руководителя организации
            self.update(
                update_data={'head_id': None},
                filter_data=dict(id=org_id)
            )
            # для обхода ссылочной целостности
            # без этого нельзя удалить пользователей т.к. они ссылаются в группы как авторы групп
            GroupModel(self._connection).update(
                filter_data={
                    'org_id': org_id,
                    'removed': Ignore
                }, update_data={
                    'author_id': None
                }
            )

            # удалим всех роботных пользователей в организации
            robots = RobotServiceModel(self._connection).find(filter_data)

            for robot in robots:
                try:
                    robot_user = UserModel(self._connection).get(robot['uid'], org_id=org_id)
                    if not robot_user:
                        continue
                    app.passport.account_delete(robot['uid'])
                except Exception:
                    log.trace().warning('Unable to delete robot')
                finally:
                    RobotServiceModel(self._connection).delete({'id': robot['id']})

            UserModel(self._connection).delete(**kwargs_for_force_delete)
            DepartmentModel(self._connection).delete(**kwargs_for_force_delete)
            GroupModel(self._connection).delete(generate_action=False, **kwargs_for_force_delete)
            OrganizationServiceModel(self._connection).delete(
                filter_data={
                    'org_id': org_id,
                    'enabled': Ignore,
                },
                force_remove_all=True,
            )
            models = [
                UserGroupMembership,
                ResourceRelationModel,
                ResourceModel,
                DomainModel,
                ActionModel,
                EventModel,
                UserDismissedModel,
            ]
            for model in models:
                model(self._connection).delete(filter_data=filter_data, force_remove_all=True)
                log.info('All items for model "%s" has been deleted' % model)

            self.delete(filter_data=dict(id=org_id))
            OrganizationMetaModel(meta_connection).delete(filter_data=dict(id=org_id))

    def update_user_count(self, org_id):
        from intranet.yandex_directory.src.yandex_directory.core.utils.tasks import UpdateOrganizationMembersCountTask
        from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask
        try:
            UpdateOrganizationMembersCountTask(self._connection).delay(org_id=org_id)
        except DuplicatedTask:
            pass

    def update_vip_reasons(self, org_id, vip):
        old_vip = self.filter(id=org_id).fields('vip').one()['vip']
        if set(old_vip) != set(vip):
            new_vip =  list(set(vip))
            with log.fields(org_id=org_id, new_reasons=new_vip, old_reasons=old_vip):
                log.debug('Update vip reasons')

                self.update_one(org_id, {'vip': new_vip})

                from intranet.yandex_directory.src.yandex_directory.core.actions import action_organization_vip_reason_change
                action_organization_vip_reason_change(
                    self._connection,
                    org_id=org_id,
                    author_id=None,
                    object_value={'id': org_id, 'vip': new_vip},
                    old_object={'id': org_id, 'vip': old_vip},
                )

    def is_blocked(self, org_id):
        return self.filter(id=org_id).scalar('is_blocked')[0]

    def block(self, org_id):
        from intranet.yandex_directory.src.yandex_directory.core.models.service import disable_licensed_services_by_org_blocked

        self.disable_paid_mode(org_id)
        with get_meta_connection() as meta_connection:
            disable_licensed_services_by_org_blocked(meta_connection, self._connection, org_id)

        if self.is_blocked(org_id):
            return

        self.update_one(org_id, {
            'is_blocked': True,
        })

        from intranet.yandex_directory.src.yandex_directory.core.actions import action_organization_block
        action_organization_block(
            self._connection,
            org_id=org_id,
            author_id=g.user.passport_uid,
            object_value={'id': org_id},
            old_object={'id': org_id},
        )

    def unblock(self, org_id):
        if not self.is_blocked(org_id):
            return

        self.update_one(org_id, {
            'is_blocked': False
        })

        from intranet.yandex_directory.src.yandex_directory.core.actions import action_organization_unblock
        action_organization_unblock(
            self._connection,
            org_id=org_id,
            author_id=g.user.passport_uid,
            object_value={'id': org_id},
            old_object={'id': org_id},
        )

    def has_owned_domains(self, org_id):
        has_master = True
        try:
            from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
                get_domains_from_db_or_domenator,
                DomainFilter,
            )
            with get_meta_connection() as meta_connection:
                get_domains_from_db_or_domenator(
                    meta_connection=meta_connection,
                    domain_filter=DomainFilter(org_id=org_id, master=True),
                    main_connection=self._connection,
                    one=True,
                    exc_on_empty_result=MasterDomainNotFound(org_id=org_id),
                )
        except MasterDomainNotFound:
            has_master = False
        return has_master

def get_price_and_product_id_for_service(users_count, service, promocode_id):
    with log.fields(service=service, users_count=users_count, promocode_id=promocode_id):
        if isinstance(service, six.integer_types) or service.isdigit():
            from intranet.yandex_directory.src.yandex_directory.core.models.service import ServiceModel
            with get_meta_connection() as meta_connection:
                service = ServiceModel(meta_connection).get(service)['slug']

        log.info('Getting price for service')
        product_id_map = deepcopy(app.config['BILLING_PRODUCT_IDS_FOR_SERVICES'][service])
        product_id = _get_product_id_for_service(users_count, product_id_map)
        product_id_with_discount = None

        if product_id != app.config['PRODUCT_ID_FREE'] and promocode_id:
            log.debug('Getting promocode')
            promocode = get_promocode_by_id(promocode_id=promocode_id)
            if promocode:
                log.info('Promocode has been found')
                for user_count, product in promocode.get('product_ids', {}).get(service, {}).items():
                    product_id_map[int(user_count)] = product
                product_id_with_discount = _get_product_id_for_service(users_count, product_id_map)

        all_prices = app.billing_client.get_products_price()
        # product_id_free используется в промокодах для бесплатных организаций
        all_prices[app.config['PRODUCT_ID_FREE']] = 0
        price = all_prices[product_id]

        if product_id_with_discount:
            log.info('We have discount with this promocode for service')
            per_user_with_discount = all_prices[product_id_with_discount]
        else:
            log.info('We have no discount or no promocode for service')
            per_user_with_discount = None

        # если цена с промокодом равна цене без него, не будем возвращать информацию о скидке
        if price == per_user_with_discount:
            per_user_with_discount = None

        # если промокод внутренний, то не будем показывать на фронте цену со скидкой
        if product_id_with_discount and promocode['promocode_type'] == promocode_type.internal:
            if per_user_with_discount is not None:
                price = per_user_with_discount
            per_user_with_discount = None

        return {
            'product_id': product_id_with_discount or product_id,
            'price': price,
            'price_with_discount': per_user_with_discount,
        }


if os.environ.get('ENVIRONMENT') == 'autotests':
    promocodes_cache = None
else:
    promocodes_cache = TTLCache(
        maxsize=1024 * 100,
        ttl=900,
    )
promocodes_cache_lock = RLock()


@cached(promocodes_cache, lock=promocodes_cache_lock)
def get_promocode_by_id(promocode_id):
    with get_meta_connection() as meta_connection:
        return PromocodeModel(meta_connection).get(promocode_id)


def _get_product_id_for_service(users_count, product_id_map):
    """
    Возвращает id продукта в Биллинге с учетом количества пользователей.

    Args:
        users_count (int) - количество пользователей
        product_id_map (dict) - словарь с информацией о том, какой продукт применять для какого количества пользователей:
            {
                10: 999,
                20: 888,
                30: 777,
            }
            Для users_count <= 10 вернется 999, <= 20 - 888, для всего что 30 и выше - 777.
    """
    user_c = None
    product_id = None
    for user_c in sorted(product_id_map):
        if users_count <= user_c:
            product_id = product_id_map[user_c]
            break
    else:
        if user_c:
            product_id = product_id_map.get(user_c)

    return product_id


class OrganizationMetaModel(BaseModel):
    db_alias = 'meta'
    table = 'organizations'
    all_fields = [
        'id',
        'label',
        'shard',
        'ready',
        'limits',
        'cloud_org_id',
    ]
    json_fields = [
        'limits',
    ]

    def create(self, label, shard, ready=True, cloud_org_id=None, id=None, limits=None):
        """
        Создаем организацию в мета базе
        Args:
            label: label организации
            shard: номер шарда базы организции
            ready: организация готова к работе (например недо мигрировала)
            (если False - для такой организации доступно только одно действие, подтверждение домена)

        Returns:

        """
        params = dict(
            label=label,
            shard=shard,
            ready=ready,
            limits=limits or {},
        )
        if id is not None:
            params['id'] = id
        if cloud_org_id is not None:
            params['cloud_org_id'] = cloud_org_id

        return self.insert_into_db(**params)

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts, joins, used_filters = [], [], []

        self.filter_by(filter_data, filter_parts, used_filters)\
            ('id', can_be_list=True)\
            ('label')\
            ('shard')\
            ('ready')\
            ('cloud_org_id', can_be_list=True)

        return distinct, filter_parts, joins, used_filters

    def get(self, id=None, cloud_org_id=None):
        if cloud_org_id:
            response = self.find({'cloud_org_id': cloud_org_id}, limit=1)
        else:
            response = self.find({'id': id}, limit=1)
        if response:
            return response[0]
        return None

    @classmethod
    def get_shard_for_new_organization(cls, organization_type=None):
        # todo: test me
        from intranet.yandex_directory.src.yandex_directory.core.models.excluded_shard import ExcludedShardModel

        # DIR-5335. Теперь для различных организаций будут ограничения по шардам
        shards = get_shards_by_org_type(organization_type=organization_type)
        with get_meta_connection() as meta:
            excluded_shards = ExcludedShardModel(meta).fields('shard').scalar()

        shards = set(shards) - set(excluded_shards)
        if len(shards) == 0:
            raise RuntimeError('All shards were excluded')

        return choice(tuple(shards))

        # код ниже очень медленный поэтому пока решили
        # возвращать рандомное число парой строк выше
        #coef = app.config['WEIGHT_COEF_FOR_SHARDS']
        #shards_with_weigth = get_shards_with_weight(shards, coef)
        # мы случайно выберем некоторый шард, с учетом весов
        #rand_float = random()
        #for val, shard_num in shards_with_weigth:
        #    if rand_float < val:
        #        return shard_num

    def get_id_for_new_organization(self):
        rows = self._connection.execute('SELECT NEXTVAL(\'organizations_id_seq\')').fetchall()
        return rows[0][0]

    def raise_organization_not_found_exception(self, org_id):
        if org_id < self.get_max_org_id():
            # организация была в базе, но ее откатили/удалили
            raise OrganizationDeleted()
        raise OrganizationUnknown()

    def get_max_org_id(self):
        query = 'SELECT MAX(id) FROM organizations'
        org_id = dict(
            self._connection.execute(
                query,
            ).fetchone()
        ).get('max')
        if not org_id:
            return 0
        return int(org_id)

    def orgs_from_same_environment(self, org_ids):
        """Возвращает только те org_id, которые принадлежат текущему окружению.
        """
        result = []

        if org_ids:
            orgs = list(self.filter(id=org_ids).fields('id', 'shard').all())
            get_shard = lambda org: org['shard']
            orgs.sort(key=get_shard)
            grouped = groupby(orgs, get_shard)

            for shard, items in grouped:
                shard_org_ids = only_attrs(items, 'id')
                with get_main_connection(shard=shard) as main_connection:
                    shard_orgs = OrganizationModel(main_connection).filter(
                        id=list(shard_org_ids),
                        environment=app.config['ENVIRONMENT'],
                    ).fields(
                        'id'
                    ).all()
                    result.extend(only_attrs(shard_orgs, 'id'))
        return result

    def get_orgs_by_shards(self, *org_ids):
        query = '''
            select
                id,
                shard
            from
                organizations
            where
                id in %(org_ids)s
        '''
        orgs_data = self._connection.execute(query, {'org_ids': org_ids})
        orgs_by_shard = collections.defaultdict(set)
        for org_data in orgs_data:
            orgs_by_shard[org_data['shard']].add(org_data['id'])

        return orgs_by_shard

    def get_users_orgs_by_shards(self, *uids):
        """
        Возвращает словарь всех организаций пользователей
        сгрупированный по шардам
        """
        query = '''
            select u.org_id, o.shard
            from users u
            inner join organizations o on u.org_id = o.id
            where u.is_dismissed = false
            and u.id in %(uids)s
            limit 500
        '''
        orgs_data = self._connection.execute(
            query, {'uids': tuple(uids), }
        )
        orgs_by_shard = collections.defaultdict(set)
        for org_data in orgs_data:
            orgs_by_shard[org_data['shard']].add(org_data['org_id'])
        return orgs_by_shard

    def update_one(self, org_id, update_data):
        return self.update(
            filter_data={
                'id': org_id,
            },
            update_data=update_data,
        )


class OrganizationRevisionCounterModel(BaseModel):
    db_alias = 'main'
    table = 'revision_counters'
    order_by = 'org_id'
    primary_key = 'org_id'

    all_fields = [
        'org_id',
        'revision',
    ]

    def increment_revisions_for_user(self, meta_connection, *uids):
        """
        Увеличивает счетчик ревизий во всех организациях пользователей
        """
        query = '''
            UPDATE revision_counters SET revision=revision_counters.revision + 1
            WHERE revision_counters.org_id in %(org_ids)s
        '''
        orgs_by_shard = OrganizationMetaModel(meta_connection).get_users_orgs_by_shards(*uids)

        for shard, org_ids in orgs_by_shard.items():
            with get_main_connection(shard=shard, for_write=True) as main_connection:
                main_connection.execute(
                    query,
                    {
                        'org_ids': tuple(org_ids),
                    }
                )

    def increment_revision(self, org_id):
        """
        Возвращает следующий номер ревизии для организации.
        Инкрементит значение в колонке revision.
        """
        query = '''
            UPDATE revision_counters SET revision=revision_counters.revision + 1
                WHERE revision_counters.org_id=%(org_id)s RETURNING *;
        '''
        shard = self._connection.engine.db_info['shard']

        # получаем новый коннект на запись чтобы не блокировать строчку со счетчиком
        # до окончания основной транзакции
        with get_main_connection(shard=shard, for_write=True) as main_connection:
            result = dict(
                main_connection.execute(
                    query,
                    {
                        'org_id': org_id
                    }
                ).fetchone()
            )
        return int(result['revision'])

    def create(self, org_id):
        """
        Создаёт счетчик в таблице для организации с ревизией 0
        """
        shard = self._connection.engine.db_info['shard']

        # получаем новый коннект на запись чтобы не блокировать строчку со счетчиком
        # до окончания основной транзакции
        with get_main_connection(shard=shard, for_write=True) as main_connection:
            return OrganizationRevisionCounterModel(main_connection).insert_into_db(
                org_id=org_id,
                revision=0,
            )

    def get(self, id):
        return self.find({self.primary_key: id}, one=True)

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('org_id', can_be_list=True)

        return distinct, filter_parts, joins, used_filters


class OrganizationBillingInfoModel(BaseModel):
    db_alias = 'main'
    table = 'organizations_billing_info'
    order_by = 'org_id'
    primary_key = 'org_id'
    simple_fields = set([
        'org_id',
        'client_id',
        'contract_type',
        'person_type',
        'is_contract_active',
        'last_mail_sent_at',
        'balance',
        'first_debt_act_date',
        'receipt_sum',
        'act_sum',
        'person_id',
    ])
    date_fields = ['last_mail_sent_at', 'first_debt_act_date']
    all_fields = simple_fields

    def get(self, org_id, fields=None):
        return self.find(
            {'org_id': org_id},
            fields=fields,
            one=True,
        )

    def create(self,
               org_id,
               client_id,
               contract_type,
               person_type,
               person_id,
               is_contract_active=False):
        if contract_type == 'offer':
            is_contract_active = True  # оферта сразу активна

        return self.insert_into_db(
            org_id=org_id,
            client_id=client_id,
            contract_type=contract_type,
            person_type=person_type,
            is_contract_active=is_contract_active,
            person_id=person_id,
        )

    def update_one(self, org_id, update_data):
        return self.update(
            filter_data={
                'org_id': org_id,
            },
            update_data=update_data,
        )

    def remove_billing_info(self, org_id):
        from intranet.yandex_directory.src.yandex_directory.core.actions import action_organization_billing_info_remove
        billing_info = self.get(org_id=org_id)
        action_organization_billing_info_remove(
            self._connection,
            org_id=org_id,
            author_id=g.user.passport_uid,
            object_value={},
            old_object=billing_info,
        )
        self.delete(filter_data={'org_id': org_id})

    def check_contract_status_in_billing(self, org_id, client_id=None):
        from intranet.yandex_directory.src.yandex_directory.core.models.service import disable_licensed_services_by_inactive_contracts

        if not client_id:
            organizations_billing_info = self.get(org_id)
        else:
            organizations_billing_info = {
                'org_id': org_id,
                'client_id': client_id,
            }

        with log.name_and_fields(
                'billing',
                org_id=org_id,
                client_id=organizations_billing_info['client_id']
        ):
            log.info('Checking contracts in Billing')
            contracts = app.billing_client.get_client_contracts(
                client_id=organizations_billing_info['client_id']
            )
            if contracts:
                for contract in contracts:
                    # активным может быть только один договор
                    if bool(contract['IS_ACTIVE']):
                        log.info('Organization has active contract in Billing')
                        self.update(
                            update_data={
                                'is_contract_active': True,
                            },
                            filter_data={
                                'org_id': org_id,
                            }
                        )
                        break
                else:
                    log.info('Organization has inactive contracts in Billing. Disabling paid mode and services')
                    OrganizationModel(self._connection).disable_paid_mode(org_id)
                    with get_meta_connection() as meta_connection:
                        disable_licensed_services_by_inactive_contracts(meta_connection, self._connection, org_id)
            else:
                log.info('Organization has no contracts')

    def check_balance_and_debt(self, org_id, force_update=False, retry_without_change=False, retries=5):
        """
        Проверяем баланс и дату начала задолженности организации, обновляем в базе, если поменялись,
        либо принудительно, если force_update=True.
        Если текущий баланс меньше нуля проверяем сколько дней прошло с начала задолженности
        и отправляем письма администраторам, если нужно.
        Если прошел 31 день с начала задолженности, выключаем платный режим и отключаем лицензионные сервисы.
        retry_without_change - если True будет сделано N ретраев в попытке получить обновленный баланс
        """
        from intranet.yandex_directory.src.yandex_directory.core.models.service import disable_licensed_services_by_debt

        billing_info = self.get(org_id)
        with log.name_and_fields(
                'billing',
                org_id=org_id,
                client_id=billing_info['client_id'],
                old_balance=billing_info['balance'],
                old_first_debt_act_date=billing_info['first_debt_act_date'],
                old_receipt_sum=billing_info['receipt_sum'],
                old_act_sum=billing_info['act_sum'],
        ):
            log.info('Checking balance in billing for organization')
            current_balance_info = app.billing_client.get_balance_info(billing_info['client_id'])
            current_balance, first_debt_act_date = current_balance_info['balance'], \
                                                   current_balance_info['first_debt_act_date']
            if retry_without_change and current_balance == billing_info['balance']:
                # баланс не всегда мгновенно обновляет баланс счета после оплаты
                # сделаем несколько ретраев с нарастающей задержкой
                for retry in range(retries):
                    sleep(0.09 * retry)
                    log.info('Rechecking balance in billing for organization')
                    current_balance_info = app.billing_client.get_balance_info(billing_info['client_id'])
                    current_balance, first_debt_act_date = current_balance_info['balance'], \
                                                           current_balance_info['first_debt_act_date']
                    if current_balance != billing_info['balance']:
                        break
                else:
                    log.error('Dont find balance change in billing for organization')

            update_data = {}

            with log.fields(
                    new_balance=current_balance,
                    new_first_debt_act_date=first_debt_act_date,
                    new_receipt_sum=current_balance_info['receipt_sum'],
                    new_act_sum=current_balance_info['act_sum']
            ):
                if force_update or current_balance != billing_info['balance'] or \
                        first_debt_act_date != billing_info['first_debt_act_date']:
                    log.info('Updating balance and first_debt_act_date for organization')
                    update_data = current_balance_info
                else:
                    log.info('Balance and first_debt_act_date is not changed')

            if first_debt_act_date:
                has_debt, days_since_debt = check_has_debt(
                    first_debt_act_date=first_debt_act_date,
                    balance=current_balance
                )
                if has_debt:
                    OrganizationModel(self._connection).disable_paid_mode(org_id)
                    # выключаем все лицензионные сервисы
                    with get_meta_connection() as meta_connection:
                        disable_licensed_services_by_debt(meta_connection, self._connection, org_id)
                    log.info('Disabling paid mode and licensed services due to organization debts')
                today = utcnow().date()
                # отправляем письма о задолженности только когда баланс отрицательный
                if current_balance < 0 and days_since_debt in SEND_DEBT_MAIL_DAYS \
                        and billing_info['last_mail_sent_at'] != today:
                    self._notify_about_debt(org_id, days_since_debt, first_debt_act_date)
                    update_data['last_mail_sent_at'] = today

            if update_data:
                self.update(
                    filter_data={'org_id': org_id},
                    update_data=update_data,
                )

    def get_filters_data(self, filter_data):
        distinct = False

        filter_parts = []
        joins = []
        used_filters = []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('org_id', can_be_list=True) \
            ('is_contract_active')

        if 'organization__subscription_plan' in filter_data:
            joins.append("""
                LEFT OUTER JOIN organizations ON (
                    organizations_billing_info.org_id = organizations.id
            )
            """)
            filter_parts.append(
                self.mogrify(
                    'organizations.subscription_plan =  %(subscription_plan)s',
                    {
                        'subscription_plan': filter_data['organization__subscription_plan'],
                    }
                )
            )
            used_filters.append('organization__subscription_plan')

        return distinct, filter_parts, joins, used_filters

    def _notify_about_debt(self, org_id, days_since_debt, act_date):
        """
        Отправляем письма о наличии задолженности администраторам.
        """
        from intranet.yandex_directory.src.yandex_directory.core.mailer.utils import send_email_to_admins

        with log.fields(
                org_id=org_id,
                days_since_debt=days_since_debt,
        ):
            log.info('Sending mail to admins about debt')
            with get_meta_connection() as meta_connection:
                send_email_to_admins(
                    meta_connection,
                    self._connection,
                    org_id,
                    app.config['SENDER_CAMPAIGN_SLUG']['CONNECT_DEBT_EMAILS'][days_since_debt],
                    lang='ru',
                    tld='ru',  # платными пока могут быть только русскоязычные организации
                    invoice_month=act_date.month,
                    downgrade_time_msk='23:59',
                )


class OrganizationBillingConsumedInfoModel(BaseModel):
    db_alias = 'main'
    table = 'organizations_billing_consumed_info'
    order_by = 'org_id'
    primary_key = 'org_id'
    simple_fields = set([
        'org_id',
        'total_users_count',
        'for_date',
        'saved_at',
        'service',
        'organization_billing_info',
        'promocode_id',
        'organization_type',
    ])
    date_fields = [
        'for_date',
    ]
    select_related_fields = {
        'organization_billing_info': 'OrganizationBillingInfoModel',
    }
    all_fields = simple_fields

    def create(self,
               org_id,
               total_users_count,
               service,
               for_date,
               organization_type):

        return self.insert_into_db(
            org_id=org_id,
            total_users_count=total_users_count,
            service=service,
            for_date=for_date,
            organization_type=organization_type,
        )

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts, joins, used_filters = [], [], []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('org_id', can_be_list=True) \
            ('service') \
            ('for_date')

        if 'for_date__gte' in filter_data:
            dt = filter_data.get('for_date__gte')
            if isinstance(dt, datetime.datetime):
                value = dt.isoformat()
            else:
                value = dt

            filter_parts.append(
                self.mogrify(
                    'for_date >= %(timestamp)s',
                    {
                        'timestamp': value
                    }
                )
            )
            used_filters.append('for_date__gte')

        if 'for_date__gt' in filter_data:
            dt = filter_data.get('for_date__gt')
            if isinstance(dt, datetime.datetime):
                value = dt.isoformat()
            else:
                value = dt

            filter_parts.append(
                self.mogrify(
                    'for_date > %(timestamp)s',
                    {
                        'timestamp': value
                    }
                )
            )
            used_filters.append('for_date__gt')

        if 'for_date__lte' in filter_data:
            dt = filter_data.get('for_date__lte')
            if isinstance(dt, datetime.datetime):
                value = dt.isoformat()
            else:
                value = dt

            filter_parts.append(
                self.mogrify(
                    'for_date <= %(timestamp)s',
                    {
                        'timestamp': value
                    }
                )
            )
            used_filters.append('for_date__lte')

        if 'for_date__lt' in filter_data:
            dt = filter_data.get('for_date__lt')
            if isinstance(dt, datetime.datetime):
                value = dt.isoformat()
            else:
                value = dt

            filter_parts.append(
                self.mogrify(
                    'for_date < %(timestamp)s',
                    {
                        'timestamp': value
                    }
                )
            )
            used_filters.append('for_date__lt')

        return distinct, filter_parts, joins, used_filters

    def get_select_related_data(self, select_related):
        if not select_related:
            return [self.default_all_projection], [], []

        select_related = select_related or []
        projections = set()
        joins = []
        processors = []

        if 'organization_billing_info' in select_related:
            projections.update([
                'organizations_billing_consumed_info.*',
                'organization_billing_info.client_id AS "organization_billing_info.client_id"',
                'organization_billing_info.org_id AS "organization_billing_info.org_id"',
            ])
            sql = """
                LEFT OUTER JOIN organizations_billing_info as organization_billing_info ON (
                    organizations_billing_consumed_info.org_id = organization_billing_info.org_id
                )
            """
            joins.append(sql)

        return projections, joins, processors

    def calculate(self, org_id=None, for_date=None, subscription_plan_changed_at=None, rewrite=False):
        """
        Считает количество пользователей во всех платных организациях и сохраняет в таблицу на указанный день.
        Если не указать for_date, подсчет выполнится на вчерашний день.

        При этом учитывается дата включения платного режима, если он был включен позже даты,
        на которую мы считаем потребленные услуги, в выборку эта организация не попадет.

        Args:
            rewrite (bool) - Если данные уже есть, метод всё пересчитает заново
            org_id (int) - id организации, если указано, подсчет идет только для этой организации
            for_date (datetime) - дата в UTC на которую нужно подсчитать, по умолчанию: вчера
        Returns:
            bool - признак того, были ли пересчитаны данные
        """
        if not for_date:
            yesterday = (utcnow() - datetime.timedelta(days=1)).date()
            for_date = yesterday

        for_date = ensure_date(for_date)

        # если передан subscription_plan_changed_at, то фильтруем по нему
        # если нет, то subscription_plan_changed_at == for_date
        if not subscription_plan_changed_at:
            subscription_plan_changed_at = for_date

        subscription_plan_changed_at = ensure_date(subscription_plan_changed_at)

        if rewrite:
            rewrite_query = '''
                ON CONFLICT (org_id, service, for_date) DO UPDATE SET total_users_count=excluded.total_users_count;
            '''
        else:
            rewrite_query = ''

        if org_id:
            org_id_filter = 'AND organizations.id=%(org_id)s'
            organizations_license_filter = 'AND organizations_license_consumed_info.org_id=%(org_id)s'
        else:
            org_id_filter = ''
            organizations_license_filter = ''

        # мы должны посчитать количество людей в организациях
        # на определенную дату for_date
        # при этом, надо учесть, что если организация в тот день была бесплатной,
        # то она не должна участвовать в подсчетах
        # Также, не нужно считать роботов и пользователей, которые в тот день уже были уволены
        #
        # Сравнивать нужно даты, чтобы не получилось, что платный режим включили в 12 дня,
        # а подсчет запустили для 0 часов этого же дня. Если сравнивать время, получится что мы
        # считаем потребленные услуги для организации, которая еще не была в платном режиме и отдадим 0
        query = '''
            INSERT INTO organizations_billing_consumed_info(org_id, total_users_count, service, for_date)
                SELECT
                    users.org_id,
                    count(users.id),
                    'connect',
                    %(for_date)s
                FROM users
                    LEFT JOIN
                        organizations AS organizations
                    ON users.org_id=organizations.id
                    LEFT JOIN
                        robot_services
                    ON robot_services.uid = users.id
                    LEFT JOIN
                        users_dismissed_info
                    ON users_dismissed_info.user_id = users.id
                WHERE
                    organizations.subscription_plan='{subscription_plan}'
                    AND (organizations.subscription_plan_changed_at at time zone 'UTC')::date <= %(subscription_plan_changed_at)s
                    {org_id_filter}
                    AND (users.created at time zone 'UTC')::date <= %(for_date)s
                    AND (
                        users.is_dismissed IS false
                        OR (users_dismissed_info.dismissed_date at time zone 'UTC')::date >= %(for_date)s
                    )
                    AND robot_services.uid IS NULL
                GROUP BY users.org_id

                UNION
                SELECT
                    organizations_license_consumed_info.org_id,
                    count(organizations_license_consumed_info.user_id),
                    services.slug,
                    %(for_date)s
                FROM
                    organizations_license_consumed_info
                    JOIN
                        services
                    ON organizations_license_consumed_info.service_id = services.id
                WHERE
                    organizations_license_consumed_info.for_date = %(for_date)s
                    {organizations_license_filter}
                GROUP BY organizations_license_consumed_info.org_id, services.slug
            {rewrite};

            UPDATE organizations_billing_consumed_info AS ci SET promocode_id = op.promocode_id
                FROM organization_promocodes AS op WHERE ci.org_id = op.org_id AND op.active=TRUE AND op.expires_at::date > (NOW() at time zone 'UTC')::date;

            UPDATE organizations_billing_consumed_info SET organization_type = org.organization_type
                FROM organizations AS org WHERE organizations_billing_consumed_info.org_id = org.id;
        '''
        query = query.format(
            rewrite=rewrite_query,
            subscription_plan=subscription_plan.paid,
            org_id_filter=org_id_filter,
            organizations_license_filter=organizations_license_filter,
        )
        query = self.mogrify(
            query,
            {
                'for_date': for_date,
                'org_id': org_id,
                'subscription_plan_changed_at': subscription_plan_changed_at,
            }
        )

        with log.name_and_fields('billing', org_id=org_id, for_date=for_date):
            log.info('Calculating organizations consumed data...')
            try:
                self._connection.execute(query)
            except IntegrityError:
                if rewrite:
                    # если rewrite==True, то IntegrityError происходить не должно
                    log.trace().error('Can not calculate consumed data')
                    raise
                return False
            else:
                log.info('Organizations consumed data has been calculated')
                return True

    def find_gaps_for_organization(self, from_date, to_date, org_id):
        """
        Для заданного org_id находит дыры в последовательности дат в organizations_billing_consumed_info.for_date
        (даты, для которых у организации нет подсчитанных потребленных продуктов)

        Нужно для того, чтобы при выключении платного режима, зная дату его начала найти все пропущенные
        при сохранении потребленных услуг даты.
        """
        timezone_error = "Parameters from_date and to_date must have timezone."
        try:
            from_tz = from_date.tzinfo
            to_tz = to_date.tzinfo
            if not from_tz or not to_tz:
                raise ValueError(timezone_error)
        except AttributeError:
            raise ValueError(timezone_error)

        query = '''
            SELECT dates::date FROM generate_series((%(from_date)s AT TIME ZONE 'UTC')::date,
            (%(to_date)s  AT TIME ZONE 'UTC')::date, interval '1 day') AS dates
            WHERE dates NOT IN (SELECT for_date FROM organizations_billing_consumed_info WHERE org_id=%(org_id)s)
        '''
        dates = [
            d[0] for d in self._connection.execute(
                self.mogrify(
                    query,
                    {
                        'org_id': org_id,
                        'from_date': from_date,
                        'to_date': to_date,
                    },
                )
            ).fetchall()
        ]
        return dates


class promocode_type:
    """
    Типы промокодов
    """
    public = 'public'
    internal = 'internal'


class PromocodeModel(BaseModel):
    db_alias = 'meta'
    table = 'promocodes'
    order_by = 'id'
    primary_key = 'id'

    json_fields = {
        'description',
        'product_ids',
    }

    all_fields = {
        'id',
        'activate_before',
        'expires_at',
        'description',
        'product_ids',
        'activation_limit',
        'promocode_type',
        'valid_for',
    }

    def create(self,
               id,
               activate_before,
               expires_at,
               description,
               product_ids,
               activation_limit=None,
               promocode_type=promocode_type.public,
               valid_for=None):
        """
        Args:
            id (str) - id промо кода в виде строки
            activate_before (datetime) - дата до которой необходимо активировать промокод
            expires_at (datetime) дата ДО которой действует промокод
            valid_for (int) (если указан, то expires_at игнорируется) - сколько дней будет действовать промокод после активации
            description (dict) - описание промокода на русском и английском языках в виде:
                {
                    'ru': 'Какой-то промокод',
                    'en': 'Some promo code',
                }
            product_ids - (dict) словарь, ключи которого - слаги продуктов, а значения - словарь
                                с количеством людей и id этих продуктов в Биллинге:
                {
                    'tracker': {
                        1: 98765, - id для категории 1-100 пользователей
                        100: 43210, - id для категории 101-250 пользователей
                        250: 5555, - id для категории 250+ пользователей
                    }
                }
                Этот словарь будет смерджен со словарем app.config['BILLING_PRODUCT_IDS_FOR_SERVICES']
                и данные промокода будут перезаписывать стандартные настройки
        """

        self._check_product_ids_for_valid_keys(product_ids)

        return self.insert_into_db(
            id=id,
            activate_before=activate_before,
            expires_at=expires_at,
            product_ids=product_ids,
            description=description,
            activation_limit=activation_limit,
            promocode_type=promocode_type,
            valid_for=valid_for,
        )

    def _check_product_ids_for_valid_keys(self, product_ids):
        """
        Проверяет формат словаря с id продуктов для скидок

        Слаги сервисов (ключи на первом уровне словаря) должны присутствовать в app.config['BILLING_PRODUCT_IDS_FOR_SERVICES']
        Все ключи и значения во вложенных словарях должны быть типа int или 'free'.
        """
        default_product_ids = app.config['BILLING_PRODUCT_IDS_FOR_SERVICES']
        diff_services = set(product_ids.keys()) - set(default_product_ids.keys())
        if diff_services:
            # если в промокод пытаются добавить слаг сервиса которого нет в настройках приложения,
            # вероятно это ошибка
            with log.fields(diff_services=diff_services):
                log.warning('Strange services in product_ids')
            raise ValueError('Strange services in product_ids: %s' % diff_services)

        for prices in list(default_product_ids.values()):
            if not all([isinstance(x, int) for x in list(prices.keys())]):
                raise ValueError('Price dict keys must contains only numbers')

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('id', can_be_list=True)

        return distinct, filter_parts, joins, used_filters

    def update_one(self, promocode_id, update_data):
        return self.update(
            filter_data={
                'id': promocode_id,
            },
            update_data=update_data,
        )


class OrganizationPromocodeModel(BaseModel):
    db_alias = 'main'
    table = 'organization_promocodes'
    order_by = 'org_id'
    primary_key = 'org_id'
    simple_fields = set([
        'org_id',
        'promocode_id',
        'expires_at',
        'activated_at',
        'active',
    ])
    all_fields = simple_fields

    def get(self, org_id, promocode_id):
        return self.find(
            {
                'org_id': org_id,
                'promocode_id': promocode_id,
            },
            one=True,
        )

    def get_active_for_organization(self, org_id, fields=None):
        """Возвращает активный промокод для организации, если он есть"""
        return self.find(
            {
                'org_id': org_id,
                'active': True,
                'expires_at__gt': utcnow(),
            },
            fields=fields,
            one=True,
        )

    def activate_for_organization(self, org_id, promocode_id, author_id, internal_activation=True):
        """
        Добавляет промокод для организации. При этом мы помечаем все существующие коды как выключенные.
        Текущий уникальный индекс в базе гарантирует нам, что у одной организации будет только один активный промокод.
        """
        from intranet.yandex_directory.src.yandex_directory.core.actions import action_organization_promocode_activate

        with get_meta_connection() as meta_connection:
            promocode = PromocodeModel(meta_connection).get(promocode_id)

        # если активный промокод организации совпадает с тем, который мы добавляем, то не делаем ничего
        active_promocode = self.get_active_for_organization(org_id)
        if active_promocode and active_promocode['promocode_id'] == promocode_id:
            return promocode

        _check_can_activate_promocode(promocode, internal_activation)

        # сначала выключим активный промокод, если он есть и отправим соответствующие события
        self.deactivate_for_organization(
            org_id=org_id,
            author_id=author_id,
        )

        # затем добавим вновь действующий
        query = '''
        INSERT INTO {0} (org_id, promocode_id, expires_at, activated_at, active) VALUES
            (%(org_id)s, %(promocode_id)s, %(expires_at)s, %(activated_at)s, %(active)s)
            ON CONFLICT (org_id, promocode_id) DO UPDATE SET active=TRUE, activated_at=NOW()
        RETURNING *;
        '''.format(self.table)
        new_org_promocode = dict(
            self._connection.execute(
                self.mogrify(
                    query,
                    {
                        'org_id': org_id,
                        'promocode_id': promocode['id'],
                        'expires_at': get_promocode_expires_at(promocode),
                        'activated_at': utcnow(),
                        'active': True,
                    },
                ),
            ).fetchone()
        )

        if promocode['activation_limit']:
            with get_meta_connection(for_write=True) as meta_connection:
                try:
                    PromocodeModel(meta_connection).update_one(
                        promocode_id,
                        {'activation_limit': promocode['activation_limit'] - 1}
                    )
                except IntegrityError:
                    raise PromocodeInvalidException()
        action_organization_promocode_activate(
            self._connection,
            org_id=org_id,
            author_id=author_id,
            object_value=None,
            content={
                'promocode_id': promocode_id,
            },
        )
        return new_org_promocode

    def deactivate_for_organization(self, org_id, author_id, promocode_id=None):
        """Выключает промокоды для организации"""
        from intranet.yandex_directory.src.yandex_directory.core.actions import action_organization_promocode_deactivate

        if not promocode_id:
            active_promocode = self.get_active_for_organization(
                org_id=org_id,
                fields=['promocode_id'],
            )
            if active_promocode:
                promocode_id = active_promocode['promocode_id']

        if promocode_id:
            self.update(
                update_data={
                    'active': False,
                },
                filter_data={
                    'promocode_id': promocode_id,
                    'org_id': org_id,
                },
            )

            action_organization_promocode_deactivate(
                self._connection,
                org_id=org_id,
                author_id=author_id,
                object_value=None,
                content={
                    'promocode_id': promocode_id,
                },
            )

    def deactivate_expired_promocodes(self, org_id=None):
        """
        Выключает просроченные активные промокоды. Делаем не одним апдейтом чтобы сохранить все события

        В основном нужно для кроновой команды выключения промокодов по времени истечения.
        """
        filter_data = {
            'expires_at__lt': utcnow(),
            'active': True,
        }
        if org_id is not None:
            filter_data['org_id'] = org_id

        expired_active_promocodes = self.find(
            filter_data=filter_data,
            fields=['org_id', 'promocode_id'],
        )

        for promocode in expired_active_promocodes:
            # берем новый коннект чтобы выключить код транзакционно в одной организации
            shard = self._connection.engine.db_info['shard']
            with get_main_connection(for_write=True, shard=shard) as main_connection:
                OrganizationPromocodeModel(main_connection).deactivate_for_organization(
                    org_id=promocode['org_id'],
                    author_id=None,
                    promocode_id=promocode['promocode_id'],
                )

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('id', can_be_list=True) \
            ('org_id') \
            ('promocode_id') \
            ('active')

        if 'expires_at__gt' in filter_data:
            date = filter_data.get('expires_at__gt')
            if not isinstance(date, datetime.datetime):
                raise ValueError('expires_at__gt must be datetime.datetime not {}'.format(type(date)))

            filter_parts.append(
                self.mogrify(
                    'expires_at > %(date)s',
                    {
                        'date': date,
                    }
                )
            )
            used_filters.append('expires_at__gt')

        if 'expires_at__lt' in filter_data:
            date = filter_data.get('expires_at__lt')
            if not isinstance(date, datetime.datetime):
                raise ValueError('expires_at__lt must be datetime.datetime not {}'.format(type(date)))

            filter_parts.append(
                self.mogrify(
                    'expires_at < %(date)s',
                    {
                        'date': date,
                    }
                )
            )
            used_filters.append('expires_at__lt')

        return distinct, filter_parts, joins, used_filters


def _check_can_activate_promocode(promocode, internal_activation=True):
    if not promocode:
        raise PromocodeInvalidException()

    now = utcnow().date()

    if now > promocode['activate_before']:
        raise PromocodeExpiredException()

    if now > promocode['expires_at'] and not promocode['valid_for']:
        raise PromocodeExpiredException()

    if promocode['activation_limit'] == 0:
        raise PromocodeInvalidException()

    # нельзя активировать внутренние прокоды через портальную ручку
    if promocode['promocode_type'] == promocode_type.internal and not internal_activation:
        raise PromocodeInvalidException()


class OrganizationLicenseConsumedInfoModel(BaseModel):
    db_alias = 'main'
    table = 'organizations_license_consumed_info'
    order_by = 'org_id'
    primary_key = 'org_id'
    simple_fields = set([
        'org_id',
        'user_id',
        'service_id',
        'for_date',
    ])
    date_fields = ['for_date']
    select_related_fields = {}
    all_fields = simple_fields

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts, joins, used_filters = [], [], []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('org_id', can_be_list=True) \
            ('service_id', can_be_list=True) \
            ('for_date')

        if 'begin_date' in filter_data:
            if filter_data['begin_date']:
                filter_parts.append(
                    self.mogrify(
                        'for_date::date >= %(begin_date)s::date',
                        {
                            'begin_date': filter_data.get('begin_date')
                        }
                    )
                )
            used_filters.append('begin_date')

        if 'end_date' in filter_data:
            if filter_data['end_date']:
                filter_parts.append(
                    self.mogrify(
                        'for_date::date <= %(end_date)s::date',
                        {
                            'end_date': filter_data.get('end_date')
                        }
                    )
                )
            used_filters.append('end_date')

        return distinct, filter_parts, joins, used_filters

    def save_user_service_licenses(self, org_id=None, service_id=None, for_date=None):
        """
        В таблице user_service_licenses храним текущие лицензии пользователей.
        Метод копирует ее содержимое, чтобы сохранять информацию об использованных лицензиях на текущую дату
        и затем отправлять ее в YT.
        Args:
            org_id (int) - id организации, если указано, копируются только лицензии для этой организации
            service_id (int) - id сервиса, если указано, копируются только лицензии для этого сервиса
            for_date (datetime) - дата за которую сохраняются данные, по умолчанию - сегодня
        """
        for_date = ensure_date(for_date or utcnow())
        where_filter = 'WHERE organization_services.enabled = True and ' \
                       '(organization_services.trial_expires < %(for_date)s OR organization_services.trial_expires is NULL) '
        if org_id:
            where_filter += ' AND user_service_licenses.org_id=%(org_id)s'
            if service_id:
                where_filter += ' AND user_service_licenses.service_id=%(service_id)s'
        elif service_id:
            where_filter += ' AND user_service_licenses.service_id=%(service_id)s'

        query = '''
            INSERT INTO organizations_license_consumed_info(org_id, user_id, service_id, for_date)
                SELECT
                    user_service_licenses.org_id,
                    user_service_licenses.user_id,
                    user_service_licenses.service_id,
                    %(for_date)s
                FROM user_service_licenses
                LEFT JOIN
                    organization_services
                    ON organization_services.org_id=user_service_licenses.org_id
                        AND organization_services.service_id = user_service_licenses.service_id
                {filter}
            ON CONFLICT (org_id, user_id, service_id, for_date) DO NOTHING;
        '''

        query = query.format(
            filter=where_filter,
        )
        query = self.mogrify(
            query,
            {
                'org_id': org_id,
                'service_id': service_id,
                'for_date': for_date,
            }
        )

        with log.name_and_fields('billing', org_id=org_id, for_date=for_date):
            log.info('Saving user service licenses consumed data...')
            self._connection.execute(query)


class OrganizationsAnalyticsInfoModel(BaseAnalyticsModel):
    db_alias = 'main'
    table = 'organizations_analytics_info'
    order_by = 'id'

    simple_fields = set([
        'id',
        'registration_date',
        'source',
        'subscription_plan',
        'language',
        'tld',
        'country',
        'balance',
        'first_debt_act_date',
        'services',
        'organization_type',
        'for_date',
        'admins',
        'deputy_admins',
        'name',
        'vip',
        'feature_ids',
    ])

    all_fields = simple_fields

    def save(self, org_id=None):
        # сохраняем информацию об организациях на текущую дату
        from intranet.yandex_directory.src.yandex_directory.core.models import UserMetaModel

        today = ensure_date(utcnow())
        with log.name_and_fields('analytics', org_id=org_id, for_date=today.isoformat()), \
             get_meta_connection() as meta_connection:
            log.info('Saving organization analytics data...')

            admins_table = 'temp_admins'
            admins = UserMetaModel(meta_connection).get_outer_deputy_admins(org_id=org_id)
            by_org_id = defaultdict(list)
            for item in admins:
                by_org_id[item['org_id']].append(item['id'])

            grouped = [
                {'org_id': key, 'admin_ids': admin_ids}
                for key, admin_ids in by_org_id.items()
            ]

            self._connection.execute('CREATE TEMP TABLE {} (org_id BIGINT, admin_ids bigint[])'.format(admins_table))
            batch_insert(self._connection, admins_table, grouped)


            if org_id:
                org_id_filter = 'WHERE org.id=%(org_id)s'
            else:
                org_id_filter = ''

            query = '''
                   INSERT INTO organizations_analytics_info(id, registration_date, source, subscription_plan, language,
                                                                tld, country, balance, first_debt_act_date, services,
                                                                feature_ids, admins, deputy_admins, organization_type, name, vip)
                           SELECT
                               org.id,
                               (org.created at time zone 'UTC')::date,
                               org.source,
                               org.subscription_plan,
                               org.language,
                               org.tld,
                               org.country,
                               billing_info.balance::VARCHAR(255),
                               billing_info.first_debt_act_date,
                               ARRAY(
                                   SELECT services.slug FROM services
                                   JOIN organization_services
                                   ON organization_services.service_id = services.id
                                   RIGHT JOIN organizations
                                   ON organization_services.org_id = organizations.id
                                   WHERE organization_services.enabled=TRUE
                                   AND organizations.id=org.id
                               ),
                               array[]::int[],
                               ARRAY(
                                   SELECT u.id FROM users as u
                                   WHERE u.role='admin'
                                   AND u.org_id=org.id
                                   UNION SELECT org.admin_uid
                               ),
                               ARRAY(
                                   SELECT u.id FROM users as u
                                   WHERE u.role='deputy_admin'
                                   AND u.org_id=org.id
                               ) || COALESCE({admins_table}.admin_ids, '{{}}'),
                               org.organization_type,
                               org.name_plain,
                               org.vip
                           FROM organizations as org
                               LEFT JOIN
                                   organizations_billing_info as billing_info
                               ON billing_info.org_id = org.id
                               LEFT JOIN
                                   {admins_table}
                               ON {admins_table}.org_id = org.id
                           {org_id_filter}
                       ON CONFLICT (id, for_date) DO UPDATE
                           SET subscription_plan = excluded.subscription_plan,
                               balance = excluded.balance,
                               first_debt_act_date = excluded.first_debt_act_date,
                               services = excluded.services;
            '''
            query = query.format(
                org_id_filter=org_id_filter,
                admins_table=admins_table,
            )

            query = self.mogrify(
                query,
                {
                    'org_id': org_id,
                }
            )

            self._connection.execute(query)

            log.info('Users analytics data has been saved')
            return True


class PartnersMetaModel(BaseModel):
    db_alias = 'meta'
    table = 'partners'
    all_fields = [
        'id',
        'name',
    ]

    def create(self, name):
        return self.insert_into_db(
            name=name,
        )

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('id', can_be_list=True) \
            ('name')

        return distinct, filter_parts, joins, used_filters
