# -*- coding: utf-8 -*-
import re
from intranet.yandex_directory.src import settings
from intranet.yandex_directory.src.yandex_directory.common.models.base import (
    BaseModel,
)

from intranet.yandex_directory.src.yandex_directory.core.models.service import ServiceModel
from intranet.yandex_directory.src.yandex_directory.core.utils import only_attrs
from intranet.yandex_directory.src.yandex_directory.core.dependencies import Service, Setting
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask


class PresetModel(BaseModel):
    db_alias = 'meta'
    table = 'presets'
    order_by = 'name'
    primary_key = 'name'
    json_fields = [
        'settings'
    ]
    all_fields = [
        'name',
        'service_slugs',
        'settings'
    ]

    def create(self, name, service_slugs, settings):
        """
        Args:
            name: строка латинские буквы, цифры и -
            service_slugs: список слагов сервисов
            settings: словарь с настройками
        """
        if not re.match('[a-zA-Z\d\-]+$', name):
            raise RuntimeError('Wrong name format, use only latin letters, numbers and -')

        all_services = only_attrs(ServiceModel(self._connection).find(fields=['slug']), 'slug')
        wrong_services = set(service_slugs) - set(all_services)
        if wrong_services:
            raise RuntimeError('Services {} do not exist'.format(', '.join(wrong_services)))

        data = {
            'name': name,
            'service_slugs': service_slugs,
            'settings': settings,
        }
        return self.insert_into_db(**data)

    def get(self, name, fields=None):
        return self.find(
            filter_data={'name': name},
            fields=fields,
            one=True,
        )

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts, joins, used_filters = [], [], []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('name', can_be_list=True) \
            ('service_slugs', array=True) \
            ('settings')

        return distinct, filter_parts, joins, used_filters


def apply_preset(meta_connection,
                 main_connection,
                 org_id,
                 preset_name):
    from intranet.yandex_directory.src.yandex_directory.core.cloud.tasks import SyncCloudOrgTask
    preset = PresetModel(meta_connection).get(preset_name)
    if not preset:
        raise RuntimeError('Preset "{0}" not found'.format(preset_name))

    services = list(map(Service, preset['service_slugs']))
    services_settings = [
        Setting(name, value)
        for name, value in list(preset['settings'].items())
    ]

    for item in services + services_settings:
        item.enable(meta_connection, main_connection, org_id)

    if preset_name in settings.CLOUD_PRESETS:
        # Если создается облачная организация - запустим синк пользователей
        # Ожидаем, что облаку хватит 15 секунд на то, чтобы выдать нам права листинга членов организации
        try:
            SyncCloudOrgTask(main_connection).delay(org_id=org_id, start_in=15)
        except DuplicatedTask:
            pass
