# -*- coding: utf-8 -*-
from bson.objectid import ObjectId
from sqlalchemy.exc import IntegrityError

from intranet.yandex_directory.src.yandex_directory.common.models.base import (
    BaseModel,
    set_to_none_if_no_id,
)


RESOURCE_OBJECT_TYPES = [
    'user',
    'group',
    'department',
]


class relation_name:
    """Это как бы такой enum для тех отношений, которые
    используются в самой Директории.
    """
    # связь для включения сущностей в группу
    include = 'include'
    # связь для установки админа группы
    admin = 'admin'
    # связь для выдачи лицензий
    member = 'member'


class ResourceModel(BaseModel):
    db_alias = 'main'
    table = 'resources'

    all_fields = [
        'id',
        'org_id',
        'service',
        'external_id',
        # поля не из базы
        'relations',
    ]
    prefetch_related_fields = {
        'relations': 'ResourceRelationModel',
    }

    def create(self, org_id, service, id=None, relations=None, external_id=None):
        """
        При создании записи ресурса мы должны заполнить поля id и external_id, т.к.
        через API пользователи на самом деле работают не с id ресурсов, а с external_id.
        external_id уникален в рамках (external_id, org_id, service).
        Сам же id используется для внутренних связей в директории и является уникальным на всю таблицу.
        Нужно подумать о смене id со строкового в числовой тип для ускорения.
        """
        if not relations:
            relations = []

        query = """
        INSERT INTO resources (id, org_id, service, external_id)
        VALUES (%(id)s, %(org_id)s, %(service)s, %(external_id)s)
        RETURNING *
        """

        conn = self._connection
        resource_id = id or str(ObjectId())
        external_id = external_id or resource_id
        resource = dict(
            conn.execute(
                query, {
                    'id': resource_id,
                    'org_id': org_id,
                    'service': service,
                    'external_id': external_id,
                }
            ).fetchone()
        )

        created_relations = []
        for relation in relations:
            user_id = relation.get('user_id')
            group_id = relation.get('group_id')
            department_id = relation.get('department_id')
            
            if not any((user_id, group_id, department_id)):
                raise RuntimeError('Resource relation should have a link to an object')
            
            created_relations.append(
                ResourceRelationModel(connection=conn).create(
                    org_id=org_id,
                    resource_id=resource['id'],
                    name=relation['name'],
                    user_id=user_id,
                    group_id=group_id,
                    department_id=department_id,
                )
            )
        resource['relations'] = created_relations

        return resource

    def get_relations(self, resource_id, org_id, **other_filter_data):
        filter_data = {
            'resource_id': resource_id,
            'org_id': org_id
        }
        filter_data.update(other_filter_data)
        return ResourceRelationModel(self._connection).find(filter_data=filter_data)

    def update_relations(self, resource_id, org_id, relations):
        current_relations = self.get_relations(
            resource_id=resource_id,
            org_id=org_id
        )

        relation_model = ResourceRelationModel(self._connection)

        for r in current_relations:
            relation_model.delete({'id': r['id'], 'org_id': org_id})

        for relation in relations:
            relation_model.create(
                org_id=org_id,
                resource_id=resource_id,
                name=relation['name'],
                user_id=relation.get('user_id'),
                group_id=relation.get('group_id'),
                department_id=relation.get('department_id'),
            )

    def add_relations(self, resource_id, org_id, relations):
        connection = self._connection
        relation_model = ResourceRelationModel(connection)

        for relation in relations:
            try:
                # если не начать тут вложенную транзакцию, то в результате возможной
                # IntegrityError ломается основная транзакция
                user_id=relation.get('user_id')
                group_id=relation.get('group_id')
                department_id=relation.get('department_id')

                if not any((user_id, group_id, department_id)):
                    raise RuntimeError('Resource relation should have a link to an object')
                
                with connection.begin_nested():
                    relation_model.create(
                        org_id=org_id,
                        resource_id=resource_id,
                        name=relation['name'],
                        user_id=user_id,
                        group_id=group_id,
                        department_id=department_id,
                    )
            except IntegrityError:
                # если такая связь уже была, то просто игнорируем ошибку
                pass

    def delete_relations(self, resource_id, org_id, relations):
        """Удалить заданные отношения.

        relations должно содержать список словарей, каждый из которых описывает
        отношение, которое необходимо удалить. Удаляться они могут либо по
        имени, либо по id сущности, которая этим отношением связана с ресурсом.
        Например:

        relations = [
            {'name': 'member'}, # или
            {'user_id': 123},
            {'department_id': 567},
            {'group_id': 897}
        ]
        """
        relation_model = ResourceRelationModel(self._connection)

        for relation in relations:
            filter_data = {
                'org_id': org_id,
                'resource_id': resource_id,
            }
            for i in ['name', 'user_id', 'group_id', 'department_id']:
                if relation.get(i):
                    filter_data[i] = relation.get(i)
            relation_model.delete(filter_data)

    def prefetch_related(self, items, prefetch_related):
        if not prefetch_related or not items:
            return

        if 'relations' in prefetch_related:
            resource_relations = {}
            nested_fields = prefetch_related['relations']
            find_kwargs = {
                'filter_data': {
                    'resource_id': [i['id'] for i in items]
                },
                'fields': {
                    'resource_id': True,
                },
            }
            if isinstance(nested_fields, dict):
                find_kwargs['fields'].update(nested_fields)

            relation_model = ResourceRelationModel(self._connection)
            for group in relation_model.find(**find_kwargs):
                resource_relations.setdefault(group['resource_id'], []).append(group)

            for item in items:
                item['relations'] = resource_relations.get(item['id'], [])

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        if 'service' in filter_data:
            filter_parts.append(
                self.mogrify(
                    'resources.service = %(service)s',
                    {
                        'service': filter_data.get('service')
                    }
                )
            )
            used_filters.append('service')

        if 'external_id' in filter_data:
            operator = '='
            external_id = filter_data.get('external_id')
            if isinstance(external_id, list):
                external_id = tuple(external_id)
                operator = 'IN'
            filter_parts.append(
                self.mogrify(
                    'resources.external_id {operator} %(external_id)s'.format(operator=operator),
                    {
                        'external_id': external_id,
                    }
                )
            )
            used_filters.append('external_id')

        if 'org_id' in filter_data:
            filter_parts.append(
                self.mogrify(
                    'resources.org_id = %(org_id)s',
                    {
                        'org_id': filter_data.get('org_id')
                    }
                )
            )
            used_filters.append('org_id')

        if 'id' in filter_data:
            operator = '='
            id = filter_data.get('id')
            if isinstance(id, list):
                id = tuple(id)
                operator = 'IN'
            filter_parts.append(
                self.mogrify(
                    'resources.id {operator} %(id)s'.format(operator=operator),
                    {
                        'id': id
                    }
                )
            )
            used_filters.append('id')

        if 'service__in' in filter_data:
            filter_parts.append(
                self.mogrify(
                    'resources.service IN %(service__in)s',
                    {
                        'service__in': tuple(filter_data.get('service__in')),
                    }
                )
            )
            used_filters.append('service__in')

        if (
            'relation_name' in filter_data or
            'user_id' in filter_data or
            'department_id' in filter_data or
            'group_id' in filter_data
        ):
            joins.append("""
            LEFT OUTER JOIN resource_relations ON (
                resources.org_id = resource_relations.org_id AND
                resources.id = resource_relations.resource_id
            )
            """)

            # так как ресурс с сущностью может быть связан разными типами
            # связей, то надо сделать DISTINCT, чтобы не возвращать дубликаты
            distinct = True

            # отдается ресурс по заданной группы
            # не отдаются ресурсы для родительских групп
            if 'group_id' in filter_data:
                operator = '='
                group_id = filter_data.get('group_id')
                if isinstance(group_id, list):
                    group_id = tuple(group_id)
                    operator = 'IN'
                filter_parts.append(
                    self.mogrify(
                        'resource_relations.group_id {operator} %(group_id)s'.format(operator=operator),
                        {
                            'group_id': group_id,
                        }
                    )
                )
                used_filters.append('group_id')

            if 'relation_name' in filter_data:
                operator = '='
                relation_name = filter_data.get('relation_name')
                if hasattr(relation_name, '__iter__') and not isinstance(relation_name, (str, bytes)):
                    relation_name = tuple(relation_name)
                    operator = 'IN'
                filter_parts.append(
                    self.mogrify(
                        'resource_relations.name {operator} %(relation_name)s'.format(operator=operator),
                        {
                            'relation_name': relation_name,
                        }
                    )
                )
                used_filters.append('relation_name')

            if 'user_id' in filter_data:
                joins.append("""
                LEFT OUTER JOIN departments ON (
                    resource_relations.org_id = departments.org_id AND
                    resource_relations.department_id = departments.id
                )
                """)
                joins.append("""
                LEFT OUTER JOIN user_group_membership as users_in_group ON (
                    resource_relations.org_id = users_in_group.org_id AND
                    resource_relations.group_id = users_in_group.group_id
                )
                """)
                joins.append("""
                LEFT OUTER JOIN users ON (
                    resources.org_id = users.org_id
                )
                LEFT OUTER JOIN departments as user_department ON (
                    users.org_id = user_department.org_id AND
                    users.department_id = user_department.id
                )
                """)

                filter_parts.append(
                    self.mogrify(
                        """
                        users.id = %(user_id)s AND
                        (
                            (
                                resource_relations.user_id = %(user_id)s
                            )
                                OR
                            (
                                departments.id IS NOT NULL AND
                                departments.path @> user_department.path
                            )
                                OR
                            (
                                users_in_group.user_id = %(user_id)s
                            )
                        )
                        """,
                        {
                            'user_id': filter_data.get('user_id'),
                        }
                    )
                )

                used_filters.append('user_id')

            if 'department_id' in filter_data:
                if 'org_id' not in filter_data:
                    raise RuntimeError('org_id is required for resource filter')
                joins.append("""
                LEFT OUTER JOIN departments ON (
                    resource_relations.org_id = departments.org_id AND
                    resource_relations.department_id = departments.id
                )
                """)

                filter_parts.append(
                    self.mogrify(
                        """
                        departments.id IS NOT NULL AND
                        departments.path @> (
                            SELECT departments.path
                            FROM departments
                            WHERE departments.id = %(department_id)s
                            AND departments.org_id = %(org_id)s 
                        )

                        """,
                        {
                            'department_id': filter_data.get('department_id'),
                            'org_id': filter_data.get('org_id')
                        }
                    )
                )

                used_filters.append('department_id')

        return distinct, filter_parts, joins, used_filters

    def get(self,
            id=None,
            external_id=None,
            org_id=None,
            service=None,
            fields=None):

        if (id and external_id) or not (id or external_id):
            raise ValueError('You must specify only one parameter: id or external_id')

        filter_data = {}
        if id:
            filter_data['id'] = id
        elif external_id:
            filter_data['external_id'] = external_id

        if service is not None:
            filter_data['service'] = service
        if org_id:
            filter_data['org_id'] = org_id

        return self.find(
            filter_data=filter_data,
            limit=1,
            fields=fields,
            one=True,
        )


class ResourceRelationModel(BaseModel):
    db_alias = 'main'
    table = 'resource_relations'

    all_fields = [
        'id',
        'org_id',
        'resource_id',
        'name',
        'user_id',
        'department_id',
        'group_id',
        # not in db
        'user',
        'group',
        'department',
    ]
    select_related_fields = {
        'user': 'UserModel',
        'group': 'GroupModel',
        'department': 'DepartmentModel',
    }

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        joins = []
        filter_parts = []
        used_filters = []

        # TODO: надо бы сплющить этот одинаковый код в один маленький кусочек

        if 'resource_id' in filter_data:
            resource_id = filter_data.get('resource_id')
            operator = '='
            if hasattr(resource_id, '__iter__') and not isinstance(resource_id, (str, bytes)):
                resource_id = tuple(resource_id)
                operator = 'IN'
            filter_parts.append(
                self.mogrify(
                    'resource_relations.resource_id {operator} %(resource_id)s'.format(
                        operator=operator
                    ),
                    {
                        'resource_id': resource_id
                    }
                )
            )
            used_filters.append('resource_id')

        if 'org_id' in filter_data:
            filter_parts.append(
                self.mogrify(
                    'resource_relations.org_id = %(org_id)s',
                    {
                        'org_id': filter_data.get('org_id')
                    }
                )
            )
            used_filters.append('org_id')

        if 'resource_id__in' in filter_data:
            filter_parts.append(
                self.mogrify(
                    'resource_relations.resource_id IN %(resource_id__in)s',
                    {
                        'resource_id__in': tuple(filter_data.get('resource_id__in')),
                    }
                )
            )
            used_filters.append('resource_id__in')

        if 'name__in' in filter_data:
            filter_parts.append(
                self.mogrify(
                    'resource_relations.name IN %(name__in)s',
                    {
                        'name__in': tuple(filter_data.get('name__in')),
                    }
                )
            )
            used_filters.append('name__in')

        if 'id' in filter_data:
            id = filter_data.get('id')
            operator = '='
            if hasattr(id, '__iter__') and not isinstance(id, (str, bytes)):
                id = tuple(id)
                operator = 'IN'

            filter_parts.append(
                self.mogrify(
                    'resource_relations.id {operator} %(id)s'.format(operator=operator),
                    {
                        'id': id
                    }
                )
            )
            used_filters.append('id')

        if 'name' in filter_data:
            condition = self.mogrify(
                'resource_relations.name = %(name)s',
                {'name': filter_data.get('name')}
            )
            filter_parts.append(condition)
            used_filters.append('name')

        if 'user_id' in filter_data:
            condition = self.mogrify(
                'resource_relations.user_id = %(user_id)s',
                {'user_id': filter_data.get('user_id')}
            )
            filter_parts.append(condition)
            used_filters.append('user_id')

        if 'user_id__isnull' in filter_data:
            user_id__isnull = 'NULL' if bool(filter_data.get('user_id__isnull')) else 'NOT NULL'
            filter_parts.append(
                self.mogrify(
                    'resource_relations.user_id IS {}'.format(user_id__isnull)
                )
            )
            used_filters.append('user_id__isnull')

        if 'id__in' in filter_data:
            filter_parts.append(
                self.mogrify(
                    'resource_relations.id IN %(id__in)s',
                    {
                        'id__in': tuple(filter_data.get('id__in')),
                    }
                )
            )
            used_filters.append('id__in')

        if 'department_id' in filter_data:
            condition = self.mogrify(
                'resource_relations.department_id = %(department_id)s',
                {'department_id': filter_data.get('department_id')}
            )
            filter_parts.append(condition)
            used_filters.append('department_id')

        if 'group_id' in filter_data:
            condition = self.mogrify(
                'resource_relations.group_id = %(group_id)s',
                {'group_id': filter_data.get('group_id')}
            )
            filter_parts.append(condition)
            used_filters.append('group_id')

        return distinct, filter_parts, joins, used_filters

    def get_select_related_data(self, select_related):
        if not select_related:
            return [self.default_all_projection], [], []

        select_related = select_related or []
        projections = []
        joins = []
        processors = []

        if 'user' in select_related:
            projections += [
                'resource_relations.org_id',
                'resource_relations.user_id',
                'users.id AS "user.id"',
                'users.nickname AS "user.nickname"',
                'users.gender AS "user.gender"',
                'users.department_id AS "user.department_id"',
                'users.name AS "user.name"',
                'users.position_plain AS "user.position"',
            ]
            joins.append("""
            LEFT OUTER JOIN users ON (
                resource_relations.org_id = users.org_id AND
                resource_relations.user_id = users.id
            )
            """)
            processors.append(
                set_to_none_if_no_id('user')
            )

        if 'department' in select_related:
            projections += [
                'resource_relations.org_id',
                'resource_relations.department_id',
                'departments.id AS "department.id"',
                'departments.name AS "department.name"',
                'departments.parent_id AS "department.parent_id"',
                'departments.members_count AS "department.members_count"',
            ]
            joins.append("""
            LEFT OUTER JOIN departments ON (
                resource_relations.org_id = departments.org_id AND
                resource_relations.department_id = departments.id
            )
            """)
            processors.append(
                set_to_none_if_no_id('department')
            )

        if 'group' in select_related:
            projections += [
                'resource_relations.org_id',
                'resource_relations.group_id',
                'groups.id AS "group.id"',
                'groups.name AS "group.name"',
                'groups.type AS "group.type"',
                'groups.members_count AS "group.members_count"',
            ]
            joins.append("""
            LEFT OUTER JOIN groups ON (
                resource_relations.org_id = groups.org_id AND
                resource_relations.group_id = groups.id
            )
            """)
            processors.append(
                set_to_none_if_no_id('group')
            )

        return projections, joins, processors

    def create(self,
               org_id,
               resource_id,
               name,
               user_id=None,
               department_id=None,
               group_id=None):
        # todo: test me
        query = """
        INSERT INTO resource_relations (org_id, resource_id, name, user_id, department_id, group_id)
        VALUES (%(org_id)s, %(resource_id)s, %(name)s, %(user_id)s, %(department_id)s, %(group_id)s)
        ON CONFLICT DO NOTHING RETURNING id
        """

        data = {
            'org_id': org_id,
            'resource_id': resource_id,
            'name': name,
            'user_id': user_id,
            'department_id': department_id,
            'group_id': group_id,
        }
        result = self._connection.execute(query, data).fetchone()
        if result:
            data['id'] = result['id']
        return data

    def get(self,
            id,
            org_id,
            fields=None):

        filter_data = {
            'id': id,
            'org_id': org_id,
        }

        return self.find(
            filter_data=filter_data,
            limit=1,
            fields=fields,
            one=True,
        )
