# -*- coding: utf-8 -*-
import datetime
from collections import defaultdict
from copy import deepcopy

from dateutil.relativedelta import relativedelta

from intranet.yandex_directory.src import settings
from intranet.yandex_directory.src.yandex_directory.common.cache import cached_in_memory_with_ttl, hashkey
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_meta_connection,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    AuthorizationError,
    ServiceNotLicensed,
    ServiceNotFound,
    ServiceCanNotBeDisabledError,
)
from intranet.yandex_directory.src.yandex_directory.common.models.base import (
    BaseModel,
    PseudoModel,
    set_to_none_if_no_id,
    BaseAnalyticsModel,
    Values,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    Ignore,
    utcnow,
    ensure_date,
)

from intranet.yandex_directory.src.yandex_directory.core.exceptions import (
    OrganizationHasDebt,
    ValidationQueryParametersError,
)
from intranet.yandex_directory.src.yandex_directory.core.models.organization import (
    OrganizationModel,
    OrganizationMetaModel,
    OrganizationLicenseConsumedInfoModel,
    check_has_debt, organization_type,
)
from intranet.yandex_directory.src.yandex_directory.core.models.license import TrackerLicenseLogModel
from intranet.yandex_directory.src.yandex_directory.core.models.resource import ResourceModel
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    only_ids,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.services import (
    get_services,
)
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

# slug сервиса Трекер
# нужна для всяких костылей для трекера
TRACKER_SERVICE_SLUG = 'tracker'
# нужна для всяких костылей для wiki
WIKI_SERVICE_SLUG = 'wiki'
# нужна для всяких костылей для maillist (сервис управления рассылками)
MAILLIST_SERVICE_SLUG = 'maillist'
# нужна для всяких костылей для внутренней админки
INTERNAL_ADMIN_SERVICE_SLUG = 'connect-control-panel'


class reason:
    """
    Причины отключения сервиса
    """
    disabled_by_user = 'disabled_by_user'  # отключен пользователем
    trial_expired = 'trial_expired'  # отключен после окончания тестового периода
    no_payment = 'no_payment'  # отключен за неуплату
    inactive_contracts = 'inactive_contracts'  # отключен по причине отсуствия активных договоров
    organization_blocked = 'organization_blocked'  # отключаем по причине блокировки организации
    no_activity = 'no_activity'  # отключаем по причине отсутствия активности


class trial_status:
    """
    Статусы триальности сервиса
    """
    inapplicable = 'inapplicable'  # триал не применим для бесплатных сервисов
    in_progress = 'in_progress'  # сервис в состоянии триала
    expired = 'expired'  # триал закончился или его не было


class DiskMetadata(PseudoModel):
    all_fields = [
        'has_paid_space',
    ]
    prefetch_related_fields = {
        'has_paid_space': None
    }


class ResponsibleCannotBeChanged(NotImplementedError):
    """
    DIR-6321
    """
    pass


class ServiceModel(BaseModel):
    db_alias = 'meta'
    table = 'services'
    json_fields = (
        'data_by_tld',
        'data_by_language',
        'robot_name',
        'redirect_tld',
    )
    deleted_fields = [
        'out_of_the_box',
        'priority',
        'data_by_language',
        'show_in_header',
        'in_new_tab',
        'available_for_external_admin',
    ]
    simple_fields = [
        'id',
        'slug',
        'name',
        'client_id',
        'tvm_client_id',
        'scopes',
        'robot_required',
        'ready_default',
        'robot_name',
        'robot_uid',
        'internal',
        'paid_by_license',
        'trial_period_months',
        'redirect_tld',
        # Временно держим одновременно и это поле и tvm_client_id.
        # Когда задача DIR-3679 выкатится в прод, поле tvm_client_id можно будет убрать.
        'tvm_client_ids',
        'tvm2_client_ids',
        'team_tvm2_client_ids',
        # not in db
        'disk',  # это нужно для валидации полей в API
    ]
    all_fields = simple_fields + [
        'available_for_external_admin',
        'data_by_language',
        'data_by_tld',
        'in_new_tab',
        'priority',
        'show_in_header',
    ]
    prefetch_related_fields = {
        'disk': DiskMetadata,
    }
    select_related_fields = {
        'available_for_external_admin': None,
        'data_by_language': None,
        'data_by_tld': None,
        'in_new_tab': None,
        'priority': None,
        'show_in_header': None,
    }

    def create(self,
               slug,
               name,
               client_id=None,
               tvm_client_id=None,
               tvm_client_ids=None,
               tvm2_client_ids=None,
               team_tvm2_client_ids=None,
               scopes=[],
               robot_required=False,
               robot_name=None,
               robot_uid=None,
               internal=False,
               ready_default=True,
               id=None,
               paid_by_license=False,
               trial_period_months=0,
               redirect_tld={},
               ):
        """
        Создаем новый сервис. Должен быть задан хотя бы один из
        параметров client_id или tvm_client_id.

        :param client_id: client_id OAuth-приложения
        :param tvm_client_id: client_id TVM сервиса
        :param slug: Короткое имя сервиса
        :param name: Человеческое название
        :param robot_required: требуется ли сервису роботный аккаунт?
        :param robot_name: Кастомное имя робота
        :param robot_name: UID который надо выдать роботу
        :param internal: внутренний сервис?
        :param ready_default: готов к работе сразу?
        :param id: задаём id в таблице на шарде.
        :param paid_by_license: сервис оплачивается по лицензиям?
        :param trial_period_months: длительность триального периода в месяцах.
        :param redirect_tld: урлы для редиректа при прихождении по инвайту
        """
        if tvm_client_id:
            tvm_client_ids = [tvm_client_id]
        else:
            tvm_client_ids = tvm_client_ids or []

        tvm2_client_ids = tvm2_client_ids or []
        team_tvm2_client_ids = team_tvm2_client_ids or []

        if robot_name:
            # Поле robot_name должно иметь такой вид:
            # {'ru': basesting, 'en': basestring}
            if not (isinstance(robot_name, dict) and
                    len(robot_name) == 2 and
                    'ru' in robot_name and
                    'en' in robot_name and
                    isinstance(robot_name['ru'], str) and
                    isinstance(robot_name['en'], str)
            ):
                raise RuntimeError('Custom robot name has a wrong scheme')
        data = {
            'client_id': client_id,
            'tvm_client_id': tvm_client_id,
            'tvm_client_ids': tvm_client_ids,
            'tvm2_client_ids': tvm2_client_ids,
            'team_tvm2_client_ids': team_tvm2_client_ids,
            'slug': slug,
            'name': name,
            'robot_required': robot_required,
            'scopes': scopes,
            'robot_name': robot_name,
            'robot_uid': robot_uid,
            'internal': internal,
            'ready_default': ready_default,
            'paid_by_license': paid_by_license,
            'trial_period_months': trial_period_months,
            'redirect_tld': redirect_tld,
        }
        if id:
            data['id'] = id

        result = self.insert_into_db(**data)
        # Так как поля всё ещё в базе, а мы их выпиливаем, то надо
        # убрать их из результатов.
        for field in self.deleted_fields:
            result.pop(field, None)
        return result

    def update(self, *args, **kwargs):
        result = super(ServiceModel, self).update(*args, **kwargs)
        # Так как поля всё ещё в базе, а мы их выпиливаем, то надо
        # убрать их из результатов.
        for field in self.deleted_fields:
            result.pop(field, None)
        return result

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts, joins, used_filters = [], [], []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('id', can_be_list=True) \
            ('tvm_client_id') \
            ('tvm_client_ids', array=True, cast='BIGINT[]') \
            ('tvm2_client_ids', array=True, cast='BIGINT[]') \
            ('team_tvm2_client_ids', array=True, cast='BIGINT[]') \
            ('client_id') \
            ('ready_default') \
            ('robot_required') \
            ('slug', can_be_list=True) \
            ('robot_name') \
            ('internal') \
            ('paid_by_license') \
            ('robot_uid')

        flag_contain_select_related_fields = False
        if 'available_for_external_admin' in filter_data:
            if not isinstance(filter_data['available_for_external_admin'], bool):
                try:
                    filter_data['available_for_external_admin'] = int(filter_data['available_for_external_admin'])
                except ValueError:
                    raise ValidationQueryParametersError('available_for_external_admin')

            if not filter_data['available_for_external_admin']:
                filter_data['available_for_external_admin'] = None
            else:
                filter_data['available_for_external_admin'] = True
            filter_parts.append(
                self.mogrify(
                    'service_link.available_for_external_admin is %(available_for_external_admin)s',
                    {
                        'available_for_external_admin': filter_data.get('available_for_external_admin')
                    }
                )
            )
            used_filters.append('available_for_external_admin')
            flag_contain_select_related_fields = True

        if 'show_in_header' in filter_data:
            filter_parts.append(
                self.mogrify(
                    'service_link.show_in_header = %(show_in_header)s',
                    {
                        'show_in_header': filter_data.get('show_in_header')
                    }
                )
            )
            used_filters.append('show_in_header')
            flag_contain_select_related_fields = True

        if flag_contain_select_related_fields:
            joins.append(
                "LEFT OUTER JOIN services_links as service_link ON (service_link.slug = services.slug)"
            )
        return distinct, filter_parts, joins, used_filters

    def get(self, id):
        return self.find(
            filter_data={'id': id},
            one=True,
        )

    def get_by_slug(self, slug, fields=None):
        return self.find(
            filter_data={'slug': slug},
            fields=fields,
            one=True,
        )

    def delete_one(self, id):
        self.delete(
            filter_data={'id': id}
        )

    def get_licensed_service_by_slug(self, slug):
        service = self.get_by_slug(slug)
        if not service:
            log.warning('Service not found')
            raise ServiceNotFound()

        if not service['paid_by_license']:
            log.warning('Service not licensed')
            raise ServiceNotLicensed()
        return service

    def get_select_related_data(self, select_related):
        if not select_related:
            return [self.default_all_projection], [], []

        select_related = select_related or []
        projections = set()
        joins = []
        processors = []
        list_select_related = [
            'available_for_external_admin',
            'data_by_language',
            'data_by_tld',
            'in_new_tab',
            'priority',
            'show_in_header',
        ]
        for select_related_field in list_select_related:
            if select_related_field in select_related:
                projections.update([
                    'services_links.{0} AS "{0}"'.format(select_related_field),
                    ])

        if set(select_related).intersection(set(list_select_related)):
            joins.append(
                "LEFT OUTER JOIN services_links ON (services_links.slug = services.slug)"
            )

        return projections, joins, processors

    def prefetch_related(self, items, prefetch_related):
        default_value_fields = {
            'available_for_external_admin': False,
            'data_by_language': {},
            'data_by_tld': {},
            'in_new_tab': False,
            'priority': 1,
            'show_in_header': False,
        }

        for item in items:
            for field in default_value_fields:
                if item.get(field, '') is None:
                    item[field] = default_value_fields[field]


class OrganizationServiceModel(BaseModel):
    db_alias = 'main'
    table = 'organization_services'
    all_fields = [
        'id',
        'service_id',
        'org_id',
        'responsible_id',
        'ready',
        'ready_at',
        'enabled',
        'trial_expires',
        'resource_id',
        'disable_reason',
        'enabled_at',
        'disabled_at',
        'last_mail_sent_at',
        'trial_status',
        'user_limit',
        'expires_at',
        'last_mail_sent_for_disable',
        'mail_count',
        # не из бд
        'internal',
        'responsible',
    ]
    date_fields = [
        'trial_expires',
        'last_mail_sent_at',
        'expires_at',
    ]
    select_related_fields = {
        'internal': None,
        'responsible': 'UserModel',
    }
    prefetch_related_fields = {
    }
    field_dependencies = {}

    # Дополнительные поля мы подтягиваем из модели Сервиса
    fields_from_service_model = {
        key
        for key in ServiceModel.simple_fields
        if key not in ('id', 'org_id', 'internal')
    }
    for key in fields_from_service_model:
        all_fields.append(key)
        prefetch_related_fields[key] = None
        field_dependencies[key] = ['service_id']

    def create(self, org_id, service_id, ready=False, responsible_id=None):
        """

        :param responsible_id: Ответственный за сервис
        """
        data = {}
        with get_meta_connection() as meta_connection:
            service = ServiceModel(meta_connection).get(service_id)
        if not service:
            raise ServiceNotFound()
        if service['paid_by_license']:
            resource = ResourceModel(self._connection).create(
                org_id=org_id,
                service=app.config['DIRECTORY_SERVICE_NAME'],
            )
            data['resource_id'] = resource['id']
            data['trial_status'] = trial_status.in_progress if service['trial_period_months'] else trial_status.expired

            # для трекера и для cloud организаций выключаем триал
            if service['slug'] == 'tracker':
                organization = OrganizationModel(self._connection).get(org_id, fields=['organization_type'])
                if organization['organization_type'] in organization_type.cloud_types:
                    data['trial_status'] = trial_status.expired
                    data['trial_expires'] = (utcnow().now() - relativedelta(days=1)).date()

        if ready:
            ready_at = utcnow()
            if service['paid_by_license'] and service['trial_period_months']:
                data['trial_expires'] = (ready_at + relativedelta(months=service['trial_period_months'])).date()
        else:
            ready_at = None

        data.update({
            'org_id': org_id,
            'service_id': service_id,
            'responsible_id': responsible_id,
            'ready': ready,
            'ready_at': ready_at,
        })
        return self.insert_into_db(**data)

    def get_select_related_data(self, select_related):
        if not select_related:
            return [self.default_all_projection], [], []

        select_related = select_related or []
        projections = set()
        joins = []
        processors = []

        if 'internal' in select_related:
            projections.update([
                'organization_services.service_id',
                'service.id as "service.id"',
                'service.internal as "service.internal"',
            ])
            joins.append("""
            LEFT OUTER JOIN services as service ON (
                organization_services.service_id = service.id
            )
            """)

        if 'responsible' in select_related:
            projections.update([
                'users.id as "responsible.id"',
                'users.name as "responsible.name"',
            ])
            joins.append("""
            LEFT OUTER JOIN users ON (
                organization_services.responsible_id = users.id
            )
            """)

        return projections, joins, processors

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_data_disabled = deepcopy(filter_data)

        if 'enabled' not in filter_data_disabled:
            filter_data_disabled['enabled'] = True

        filter_parts, joins, used_filters = [], [], []

        if filter_data.get('enabled', None) is Ignore:  # убираем из фильтра
            del filter_data_disabled['enabled']

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('id') \
            ('service_id', can_be_list=True) \
            ('org_id', can_be_list=True) \
            ('trial_expires', can_be_list=True) \
            ('ready') \
            ('trial_status', can_be_list=True) \
            ('responsible_id', can_be_list=True) \
            ('resource_id')

        if 'enabled' in filter_data_disabled:
            enabled = filter_data_disabled.get('enabled')
            filter_parts.append(
                self.mogrify(
                    'organization_services.enabled = %(enabled)s',
                    {
                        'enabled': enabled
                    }
                )
            )
            used_filters.append('enabled')
        if 'enabled' in filter_data:
            used_filters.append('enabled')

        if 'internal' in filter_data:
            joins.append("""
            LEFT OUTER JOIN services ON (
                organization_services.service_id = services.id
            )
            """)
            filter_parts.append(
                self.mogrify(
                    'services.internal = %(internal)s',
                    {
                        'internal': filter_data.get('internal')
                    }
                )
            )
            used_filters.append('internal')

        if 'service_slug' in filter_data:
            joins.append("""
                        LEFT OUTER JOIN services ON (
                            organization_services.service_id = services.id
                        )
                        """)
            filter_parts.append(
                self.mogrify(
                    'services.slug = %(service_slug)s',
                    {
                        'service_slug': filter_data.get('service_slug')
                    }
                )
            )
            used_filters.append('service_slug')

        if 'has_user_licenses' in filter_data:
            # принимает True/False, возвращает сервисы, у которых есть(нет) выданные лицензии
            distinct = True
            joins.append("""
                        LEFT OUTER JOIN user_service_licenses ON (
                            organization_services.org_id = user_service_licenses.org_id AND
                            organization_services.service_id = user_service_licenses.service_id
                        )
                        """)
            has_user_licenses = 'NOT NULL' if filter_data['has_user_licenses'] else 'NULL'
            filter_parts.append(
                self.mogrify(
                    'user_service_licenses.service_id IS ' + has_user_licenses
                )
            )
            used_filters.append('has_user_licenses')

        return distinct, filter_parts, joins, used_filters

    def delete_one(self, _id):
        self.delete(
            filter_data={'id': _id}
        )

    def find_organizations_without_robot(self, service_slug, service_id, nickname=None):
        """
        Возвращает id организаций, у которых почему-то нет робота при включенном сервисе
        """
        if not nickname:
            from intranet.yandex_directory.src.yandex_directory.core.utils.robots import get_robot_nickname

            nickname = get_robot_nickname(service_slug=service_slug)

        query = '''
        SELECT organizations.id
        FROM organizations
        JOIN organization_services ON organization_services.org_id = organizations.id
        LEFT JOIN (
          SELECT org_id FROM users WHERE nickname = %(nickname)s AND NOT is_dismissed
        ) AS has_robot ON has_robot.org_id = organizations.id
        WHERE organizations.environment = %(environment)s
          AND organization_services.service_id = %(service_id)s
          AND organization_services.enabled = TRUE
          AND has_robot.org_id IS NULL
        '''

        organizations = self._connection.execute(
            query,
            {
                'service_id': service_id,
                'nickname': nickname,
                'environment': app.config['ENVIRONMENT'],
            }
        ).fetchall()
        return [o[0] for o in organizations]

    def prefetch_related(self, items, prefetch_related_fields):

        if set(self.fields_from_service_model).intersection(set(prefetch_related_fields)):
            # Если запрошено хотя бы одно поле с информацией про сервис,
            # то надо подтянуть эту информацию из метабазы

            with get_meta_connection() as meta_connection:
                all_services = get_services(meta_connection, fields=['*'])

            def get_service(service_id):
                return all_services.get(service_id).copy()

            for obj in items:
                service_id = obj['service_id']
                service = get_service(service_id)

                for field in self.fields_from_service_model:
                    if field in service:
                        obj[field] = service[field]

    def update_one(self, id, update_data):
        return self.update(
            filter_data={
                'id': id,
            },
            update_data=update_data,
        )

    def get_licensed_service_resource_id(self, org_id, service_slug):
        with log.fields(service=service_slug, org_id=org_id):
            with get_meta_connection() as meta_connection:
                service = ServiceModel(meta_connection).get_licensed_service_by_slug(service_slug)
                org_service = self.find(
                    filter_data={
                        'org_id': org_id,
                        'service_id': service['id'],
                    },
                    fields=[
                        'resource_id',
                    ],
                    one=True,
                )
                if not org_service:
                    log.warning('Service not enabled in organization (get_licensed_service_resource_id)')
                    raise AuthorizationError(
                        'Service is not enabled',
                        'service_is_not_enabled'
                    )
                return org_service['resource_id']

    def get_by_org_service(self, org_id, service_id, fields):
        org_service = self \
            .filter(org_id=org_id, service_id=service_id, enabled=Ignore) \
            .fields(*fields) \
            .one()

        if not org_service:
            log.warning('Service not enabled in organization (get_by_org_service)')
            raise AuthorizationError(
                'Service is not enabled',
                'service_is_not_enabled'
            )

        return org_service

    def get_by_slug(self, org_id, service_slug, fields=None):
        with log.fields(service=service_slug, org_id=org_id):
            with get_meta_connection() as meta_connection:
                service = ServiceModel(meta_connection) \
                    .get_by_slug(service_slug, fields=['id'])
                if not service:
                    log.warning('Service "%s" not found', service_slug)
                    raise ServiceNotFound()
                return self.get_by_org_service(org_id, service['id'], fields)

    def get_org_services_with_licenses(self,
                                       org_id,
                                       trial_expired=False,
                                       service_ids=None,
                                       only_id=False,
                                       has_user_licenses=True,
                                       service_slug=None):
        """
        Получаем подключенные к организации сервисы с лицензиями и отдаем в виде:
        {
            service1_resource_id: service1,
            service2_resource_id: service2,
            ...
        }
        если указан org_id отдаем лицензионные сервисы только для указанной организации, иначе для всех
        если trial_expired = True, то отдаем сервисы, триальный период у которых закончился
        service_ids ищем лицензионные сервисы только серди указанных id
        only_id=True отдаем только id сервисов, если весь объект не требуется, избавляет от запроса в мета базу
        has_user_licenses=False отдаем только сервисы, у которых нет лицензии
        service_slug проверяем только указанные слаг
        {
            service1_resource_id: service1_id,
            service2_resource_id: service2_id,
            ...
        }
        """
        filter_data = {
            'org_id': org_id,
            'enabled': True,
            'resource_id__isnull': False,
        }
        if trial_expired:
            filter_data['trial_status'] = trial_status.expired
        if not has_user_licenses:
            filter_data['has_user_licenses'] = False
        if service_ids:
            filter_data['service_id'] = service_ids
        if service_slug:
            filter_data['service_slug'] = service_slug
        org_services = self.find(
            filter_data=filter_data,
            fields=['service_id', 'resource_id'],
        )
        if org_services:
            if only_id:
                return {item['resource_id']: item['service_id'] for item in org_services}
            org_services = {item['service_id']: item['resource_id'] for item in org_services}
            with get_meta_connection() as meta_connection:
                services_with_licenses = ServiceModel(meta_connection).find(
                    filter_data={'id__in': list(org_services.keys())}
                )
            return {org_services[service['id']]: service for service in services_with_licenses}
        return {}

    def send_mail_about_trial_expiration_if_needed(self, org_id):
        """
        Проверяем у организации все платные подключенные сервисы,
        если этот сервис есть в конфиге и для него есть соотв. письмо из рассылятора
        для уведомления о заканчивающемся триале,
        то отправляем письмо и сохраняем дату отправки
        """
        with log.name_and_fields(
                'check-trial',
                org_id=org_id,
        ):
            log.info('Check the need to send mails about the end of trial for services in organization')
            mail_config = app.config['MAIL_IDS_BEFORE_TRIAL_END']
            for service_slug in mail_config:
                with get_meta_connection() as meta_connection:
                    service_id = ServiceModel(meta_connection).get_by_slug(service_slug)['id']

                config_days = mail_config[service_slug]
                today = utcnow().date()
                # выберем все даты, для которых нужно отправлять письмо по данному сервису
                expire_dates = [today + datetime.timedelta(rest_days) for rest_days in config_days]

                with log.fields(service_slug=service_slug, expire_dates=expire_dates):
                    organization_service = self.find(
                        filter_data={
                            'org_id': org_id,
                            'service_id': service_id,
                            'trial_expires': expire_dates,
                            'enabled': True,
                        },
                        one=True,
                        fields=['last_mail_sent_at', 'id', 'trial_expires']
                    )
                    # и если у сервиса истекает триал в один из таких дней и мы не отправляли никакого письма,
                    # отправим его
                    if organization_service and organization_service['last_mail_sent_at'] != today:
                        with log.fields(service_info=organization_service):
                            log.info('We will notify about the end of the trial period')
                            self.update(
                                filter_data={'id': organization_service['id']},
                                update_data={'last_mail_sent_at': today}
                            )
                            self._notify_about_trial_expiration(
                                org_id=org_id,
                                service_slug=service_slug,
                                rest_days=(organization_service['trial_expires'] - today).days,
                            )

    def _notify_about_trial_expiration(self, org_id, service_slug, rest_days):
        """
        Отправляет письмо за определенное количество дней до окончания триальной версии сервиса
        """
        from intranet.yandex_directory.src.yandex_directory.core.mailer.utils import send_email_to_admins

        with log.fields(
                org_id=org_id,
                rest_days=rest_days,
        ):
            log.info('Sending mail to admins about ended trial')
            with get_meta_connection() as meta_connection:
                org_name = OrganizationModel(self._connection).get(org_id)['name']
                send_email_to_admins(
                    meta_connection,
                    self._connection,
                    org_id,
                    app.config['MAIL_IDS_BEFORE_TRIAL_END'][service_slug][rest_days],
                    lang='ru',
                    tld='ru',  # платными пока могут быть только русскоязычные организации
                    organization_name=org_name,
                )

    def is_service_enabled(self, org_id, service_slug):
        return self.filter(org_id=org_id, service_slug=service_slug, enabled=True).count() > 0

    def change_responsible(self, service_id, org_id, responsible_id):
        """
        Установить ответственного за сервис.

        :param service_id:
        :param org_id:
        :param responsible_id:
        :return:
        """
        current_state = self.filter(org_id=org_id, service_id=service_id).one()
        current_responsible_id = current_state['responsible_id']
        if current_responsible_id is not None:
            # такое не предусмотрено бизнес логикой
            raise ResponsibleCannotBeChanged()
        self.update_one(current_state['id'], dict(
            responsible_id=responsible_id
        ))
        return True


def on_service_enable(main_connection, org_id, service_slug, service_id):
    if service_slug == TRACKER_SERVICE_SLUG:
        update_license_cache_task(main_connection, org_id=org_id, service_id=service_id)

        future_paid_date = TrackerBillingStatusModel(main_connection).filter(
            org_id=org_id,
            payment_status=False,
            payment_date__gte=utcnow().date(),
        ).one()
        if not future_paid_date:
            TrackerBillingStatusModel(main_connection).create(
            org_id=org_id,
            payment_date=utcnow() + relativedelta(months=1),
        )



def on_service_disable(main_connection, org_id, service_slug):
    # Раньше тут была обработка отключения листов рассылки.
    # А теперь эта функция оставлена на будущее, если понадобится ещё для какого-то сервис
    # писать код по его отключению.
    pass


def is_service_enabled(main_connection, org_id, service_slug):
    """Хелпер для более короткой записи для проверки того, включен ли сервис."""
    return OrganizationServiceModel(main_connection).is_service_enabled(org_id, service_slug)


def enable_service(meta_connection,
                   main_connection,
                   org_id,
                   service_slug,
                   author_id=None,
                   drop_licenses=False,
                   responsible_id=None):
    """Вспомогательная функция для активации сервиса.

    Если uid автора не задан, то берётся uid админа
    организации из поля admin_id объекта организации.

    responsible_id может быть None.
    """

    # Чтобы избежать циклического импорта, придётся импортировать тут.
    from intranet.yandex_directory.src.yandex_directory.core.actions import action_service_enable
    from intranet.yandex_directory.src.yandex_directory.core.dependencies import dependencies, Service
    from intranet.yandex_directory.src.yandex_directory.core.utils.tasks import ChangeIsEnabledRobotStatusTask

    # Если у сервиса есть зависимости, то пройдёмся по ним
    deps = dependencies.get(Service(service_slug), [])
    for dependency in deps:
        dependency.enable(
            meta_connection,
            main_connection,
            org_id,
            author_id=author_id,
        )

    service = ServiceModel(meta_connection).get_by_slug(service_slug)
    if not service:
        raise ServiceNotFound()

    if author_id is None:
        organization = OrganizationModel(main_connection).get(org_id)
        author_id = organization['admin_uid']

    model = OrganizationServiceModel(main_connection)
    data = {
        'org_id': org_id,
        'service_id': service['id'],
        'enabled': Ignore,
    }
    # если сервис не подключен
    organization_service = model.find(filter_data=data, one=True)
    if not organization_service or not organization_service['enabled']:
        if not organization_service:
            del data['enabled']
            data['responsible_id'] = responsible_id

            model.create(
                ready=service['ready_default'],
                **data
            )
        else:
            # если повторно включаем платный сервис,
            # нужно проверить есть ли у организации биллинговая информация и нет ли задолженности
            if organization_service['trial_status'] == trial_status.expired:
                billing_info = OrganizationModel(main_connection).get(
                    org_id,
                    fields=[
                        'billing_info.first_debt_act_date',
                        'billing_info.balance',
                    ],
                )['billing_info']
                if billing_info and check_has_debt(
                        first_debt_act_date=billing_info['first_debt_act_date'],
                        balance=billing_info['balance']
                )[0]:
                    raise OrganizationHasDebt()
            # активируем сервисных роботов
            if service['robot_required']:
                ChangeIsEnabledRobotStatusTask(main_connection).delay(
                    service_slug=service_slug,
                    org_id=org_id,
                    is_enabled=True,
                )
            if organization_service['resource_id'] and drop_licenses:
                # удаляем существующие лицензии
                drop_all_service_licenses(
                    main_connection,
                    org_id,
                    service_slug,
                    service['id'],
                    organization_service['resource_id'],
                    author_id,
                )
            ready = service['ready_default']
            ready_at = utcnow() if ready else None
            model.update(
                update_data={
                    'enabled': True,
                    'ready': ready,
                    'ready_at': ready_at,
                    'disable_reason': None,
                    'disabled_at': None,
                },
                filter_data=data
            )

        # Так как сервис из базы мы получили до того, как связали с организацией,
        # то там всё ещё нет поля про ответственного.
        service['responsible_id'] = responsible_id
        action_service_enable(
            main_connection,
            org_id=org_id,
            author_id=author_id,
            object_value=service,
        )

        on_service_enable(main_connection, org_id, service_slug, service['id'])

    return service


def disable_licensed_services_by_trial(meta_connection, main_connection, org_id):
    # выключаем все лицензионные сервисы, у которых закончился триальный период и нет лицензий
    return _disable_licensed_services_by_reason(
        meta_connection,
        main_connection,
        org_id,
        reason.trial_expired,
    )


def disable_licensed_services_by_inactive_contracts(meta_connection, main_connection, org_id):
    # выключаем все лицензионные сервисы, у которых закончился триальный период,
    # если у организации нет активных договоров
    return _disable_licensed_services_by_reason(
        meta_connection,
        main_connection,
        org_id,
        reason.inactive_contracts,
    )


def disable_licensed_services_by_debt(meta_connection, main_connection, org_id):
    # выключаем все лицензионные сервисы, у которых закончился триальный период, за долги
    return _disable_licensed_services_by_reason(
        meta_connection,
        main_connection,
        org_id,
        reason.no_payment,
    )


def disable_licensed_services_by_org_blocked(meta_connection, main_connection, org_id):
    # выключаем все лицензионные сервисы, если организация блокируется
    return _disable_licensed_services_by_reason(
        meta_connection,
        main_connection,
        org_id,
        reason.organization_blocked,
    )


def _disable_licensed_services_by_reason(meta_connection,
                                         main_connection,
                                         org_id,
                                         disable_reason,
                                         author_id=None):
    # Для партнерских организаций не отключаем сервисы по триалу
    if disable_reason == reason.trial_expired and \
            OrganizationModel(main_connection).is_partner_organization(org_id):
        return []

    services_to_disable = list(OrganizationServiceModel(main_connection).get_org_services_with_licenses(
        org_id,
        trial_expired=disable_reason != reason.organization_blocked,
        has_user_licenses=disable_reason != reason.trial_expired,
    ).values())

    for service in services_to_disable:
        # диск не отключаем, а отбираем лицензии
        if service['slug'] == 'disk':
            _drop_service_licenses(main_connection, author_id, org_id, service)

        if service['slug'] == 'tracker':
            # для трекера ничего не делаем
            continue

        if service['slug'] in settings.ALWAYS_ENABLED_SERVICES:
            continue

        disable_service(
            meta_connection,
            main_connection,
            org_id,
            service['slug'],
            disable_reason,
            service=service,
            author_id=author_id,
        )

    return services_to_disable


def _drop_service_licenses(main_connection, author_id, org_id, service):
    org_service = OrganizationServiceModel(main_connection).find(filter_data={
        'org_id': org_id,
        'service_id': service['id'],
    }, one=True)
    drop_all_service_licenses(
        main_connection,
        org_id,
        service['slug'],
        service['id'],
        org_service['resource_id'],
        author_id,
    )

def disable_service(meta_connection,
                    main_connection,
                    org_id,
                    service_slug,
                    disable_reason,
                    author_id=None,
                    service=None):
    """Вспомогательная функция для деактивации сервиса.

    Если uid автора не задан, то берётся uid админа
    организации из поля admin_id объекта организации.
    """

    # Чтобы избежать циклического импорта, придётся импортировать тут.
    from intranet.yandex_directory.src.yandex_directory.core.actions import action_service_disable
    from intranet.yandex_directory.src.yandex_directory.core.utils.tasks import ChangeIsEnabledRobotStatusTask

    if service_slug in settings.ALWAYS_ENABLED_SERVICES:
        raise ServiceCanNotBeDisabledError()

    service = service or ServiceModel(meta_connection).get_by_slug(service_slug)
    if not service:
        raise ServiceNotFound()

    if author_id is None:
        organization = OrganizationModel(main_connection).get(org_id)
        author_id = organization['admin_uid']

    model = OrganizationServiceModel(main_connection)
    data = {
        'org_id': org_id,
        'service_id': service['id'],
    }
    # если сервис подключен, то отключаем
    organization_service = model.find(filter_data=data, one=True)
    if organization_service:
        update_data = {
            'enabled': False,
            'disable_reason': disable_reason,
            'disabled_at': utcnow(),
        }
        if disable_reason == reason.trial_expired:
            update_data['trial_status'] = trial_status.expired
        # блокируем сервисных роботов
        if service['robot_required']:
            ChangeIsEnabledRobotStatusTask(main_connection).delay(
                service_slug=service_slug,
                org_id=org_id,
                is_enabled=False,
            )
        model.update(update_data=update_data, filter_data=data)
        action_service_disable(
            main_connection,
            org_id=org_id,
            author_id=author_id,
            object_value=service
        )

    on_service_disable(main_connection, org_id, service_slug)

    return service


def drop_all_service_licenses(main_connection, org_id, slug, service_id, resource_id, author_id):
    if slug == 'disk':
        from intranet.yandex_directory.src.yandex_directory.connect_services.partner_disk.tasks import DeleteSpacePartnerDiskTask

        current_lic = ResourceModel(main_connection).get_relations(resource_id, org_id, name='member')
        uids = [lic['user_id'] for lic in current_lic]
        for uid in uids:
            DeleteSpacePartnerDiskTask(main_connection).delay(
                org_id=org_id,
                uid=uid,
                resource_id=resource_id,
                author_id=author_id,
            )

    ResourceModel(main_connection).delete_relations(
        resource_id,
        org_id,
        [{'name': 'member'}]
    )
    update_license_cache_task(main_connection, org_id=org_id, service_id=service_id)


def enable_service_for_all_organizations(service_slug, meta_connection, main_connection):
    """ Эта фукнция нам пригодится для того, чтобы включить какой-нибудь внутренний
        сервис для всех организаций.
    """

    # Выберем сервисы, которые должны быть подключены во всех организациях.
    service = ServiceModel(meta_connection).find(
        {'slug': service_slug},
        fields=['id'],
        one=True,
    )

    if not service:
        raise RuntimeError('Service not found')

    # Найдём организации в которых этот сервис не подключен
    orgs = OrganizationModel(main_connection).find(
        {'service_id__not_enabled': service['id']},
        fields=('id', 'admin_uid'),
    )
    ids = only_ids(orgs)

    ready_orgs = OrganizationMetaModel(meta_connection).find(
        {'id': ids, 'ready': True},
        fields=['id'],
    )
    ready_orgs = set(only_ids(ready_orgs))

    # Уберем из списка те организации, которые пока не готовы к работе
    orgs = (org for org in orgs if org['id'] in ready_orgs)

    # Пройдёмся по каждой и включим для неё сервис.
    for org in orgs:
        org_id = org['id']

        with log.fields(org_id=org_id,
                        service=service):
            try:
                enable_service(
                    meta_connection,
                    main_connection,
                    org_id,
                    service_slug,
                    author_id=org['admin_uid'],
                    responsible_id=None,
                )
            except Exception:
                log.trace().error('Unable to enable service for organization')


class UserServiceLicenses(BaseModel):
    """ Эта табличка служит для представления всех
    сотрудников, у которых есть лицензии на сервис, независимо от того,
    выданы они непосредственно, или через членство
    в других группах или отделах.
    """
    db_alias = 'main'
    table = 'user_service_licenses'
    order_by = 'user_service_licenses.org_id, user_service_licenses.user_id'
    primary_key = 'user_id'
    all_fields = [
        'org_id',
        'user_id',
        'service_id',
        'via_department_id',
        'via_group_id',
        # not from db
        'organization_service',
    ]
    select_related_fields = {
        'organization_service': 'OrganizationServiceModel',
    }

    def create(self, user_id, org_id, service_id):
        data = {
            'user_id': user_id,
            'org_id': org_id,
            'service_id': service_id,
        }
        return self.insert_into_db(**data)

    def get_select_related_data(self, select_related):
        select_related = select_related or []
        projections = set()
        joins = []
        processors = []

        if 'organization_service' in select_related:
            projections.update([
                'user_service_licenses.group_id',
                'user_service_licenses.service_id',
                'organization_services.service_id as "organization_service.service_id"',
                'organization_services.org_id as "organization_service.org_id"',
                'organization_services.enabled as "organization_service.enabled"',
                'organization_services.trial_expires as "organization_service.trial_expires"',
            ])
            joins.append("""
            LEFT OUTER JOIN organization_services ON (
                organization_services.service_id = user_service_licenses.service_id and
                organization_services.org_id = user_service_licenses.org_id
            )
            """)
        return projections, joins, processors

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        # фильтр по user_id и org_id/resource_id позволяет найти
        # сервисы, к которым у пользователя есть доступ
        self.filter_by(filter_data, filter_parts, used_filters) \
            ('user_id', can_be_list=True) \
            ('org_id') \
            ('service_id', can_be_list=True)

        if 'service_slug' in filter_data:
            joins.append("""
                        LEFT OUTER JOIN services ON (
                            user_service_licenses.service_id = services.id
                        )
                        """)
            filter_parts.append(
                self.mogrify(
                    'services.slug = %(service_slug)s',
                    {
                        'service_slug': filter_data.get('service_slug')
                    }
                )
            )
            used_filters.append('service_slug')

        return distinct, filter_parts, joins, used_filters

    def get_users_with_service_licenses(self, org_id, service_ids, user_ids):
        licenses = self.find(
            filter_data={
                'org_id': org_id,
                'service_id': service_ids,
                'user_id': user_ids,
            },
            fields=['user_id', 'service_id'],
            distinct=True,
        )
        service_users = defaultdict(list)
        for lic in licenses:
            service_users[lic['service_id']].append(lic['user_id'])
        return service_users

    def update_licenses_cache(self, org_id, service_id):
        """
        Разворачивает список лицензий в плоский список пользователей и создает записи в таблице одним запросом
        """
        resource_id = OrganizationServiceModel(self._connection).find(
            filter_data={
                'org_id': org_id,
                'service_id': service_id,
                'enabled': Ignore,
            },
            fields=['resource_id'],
            one=True,
            for_update=True,
        )['resource_id']

        old_users = self.filter(
            org_id=org_id,
            service_id=service_id,
        ).scalar('user_id')

        # удаляем старые записи
        self.delete(
            filter_data={
                'org_id': org_id,
                'service_id': service_id,
            }
        )

        query = '''
            WITH query_users AS (
                  SELECT DISTINCT users.id AS uid,
                                  resource_relations.department_id as department_id,
                                  resource_relations.group_id as group_id
                   FROM users
                        LEFT OUTER JOIN resource_relations ON (
                          users.org_id = resource_relations.org_id
                        )
                        LEFT OUTER JOIN departments ON (
                            users.org_id = departments.org_id AND
                            users.department_id = departments.id
                        )
                        LEFT OUTER JOIN user_group_membership ON (
                            users.org_id = user_group_membership.org_id AND
                            users.id = user_group_membership.user_id AND
                            resource_relations.group_id = user_group_membership.group_id
                        )
                        WHERE (users.org_id=%(org_id)s) AND (users.is_dismissed=false)
                            AND (
                            resource_relations.resource_id=%(resource)s AND
                                (
                                    (
                                        resource_relations.user_id = users.id
                                    )
                                        OR
                                    (
                                        resource_relations.department_id IS NOT NULL AND
                                        departments.path ~ CAST(concat('*.', resource_relations.department_id, '.*') AS lquery)
                                    )
                                        OR
                                    (
                                        resource_relations.group_id IS NOT NULL AND
                                        resource_relations.group_id = user_group_membership.group_id
                                    )
                                )
                        )
            )
            INSERT INTO user_service_licenses (user_id, org_id, service_id, via_department_id, via_group_id)
            SELECT uid, %(org_id)s, %(service_id)s, department_id, group_id
            FROM query_users
        '''

        # получаем плоский список пользователей и добавляем новые записи о лицензиях
        self._connection.execute(
            query,
            {
                'resource': resource_id,
                'org_id': org_id,
                'service_id': service_id,
            }
        )
        new_users = self.filter(
            org_id=org_id,
            service_id=service_id,
        ).scalar('user_id')

        self._save_license_log(org_id, service_id, old_users, new_users)
        # добавляем новые лицензии в таблицу потребленных лицензий
        OrganizationLicenseConsumedInfoModel(self._connection).save_user_service_licenses(org_id, service_id)

    def _save_license_log(self, org_id, service_id, old_users, new_users):
        old_users = set(old_users)
        new_users = set(new_users)
        added = new_users - old_users
        deleted = old_users - new_users
        with get_meta_connection(for_write=True) as meta_connection:
            service_slug = ServiceModel(meta_connection).get(service_id)['slug']
            if service_slug == 'tracker':
                TrackerLicenseLogModel(meta_connection).save_log(org_id=org_id, added=added, deleted=deleted)


def update_license_cache_task(main_connection, org_id, service_id=None, sync=False):
    from intranet.yandex_directory.src.yandex_directory.core.utils.tasks import UpdateLicenseCache
    from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask

    try:
        attr = 'do' if sync else 'delay'
        getattr(UpdateLicenseCache(main_connection), attr)(org_id=org_id, service_id=service_id)
    except DuplicatedTask as err:
        log.error('DuplicatedTask: UpdateLicenseCache already in queue with id {}'.format(err.existing_task_id))


def update_service_licenses(main_connection, org_id, service_slug, author_id, sync=False):
    with get_meta_connection() as meta_connection:
        service = ServiceModel(meta_connection).get_by_slug(service_slug)

    update_license_cache_task(main_connection, org_id=org_id, service_id=service['id'], sync=sync)
    from intranet.yandex_directory.src.yandex_directory.core.actions import action_service_license_change
    action_service_license_change(
        main_connection,
        org_id=org_id,
        author_id=author_id,
        object_value=service,
        content={},
    )


class RequestedUserServiceLicenses(BaseModel):
    """
    Эта табличка служит для хранения заявок сотрудников на получение лицензий.
    """
    db_alias = 'main'
    table = 'requested_user_service_licenses'
    order_by = 'requested_user_service_licenses.org_id'
    all_fields = [
        'org_id',
        'user_id',
        'department_id',
        'group_id',
        'service_slug',
        'author_id',
        'created_at',
        'comment',
        'external_id',
        # not in db
        'author',
        'user',
        'department',
        'group',
    ]
    select_related_fields = {
        'author': 'UserModel',
        'user': 'UserModel',
        'group': 'GroupModel',
        'department': 'DepartmentModel',
    }

    def create(self, user_id, department_id, group_id, org_id, service_slug, author_id, comment,
               external_resource_id=None):
        data = {
            'user_id': user_id,
            'department_id': department_id,
            'group_id': group_id,
            'org_id': org_id,
            'service_slug': service_slug,
            'author_id': author_id,
            'comment': comment,
            'external_id': external_resource_id,
        }
        return self.insert_into_db(**data)

    def create_requests(self, lic_requests, comment, org_id, service_slug, author_id):
        data = []
        for req in lic_requests:
            data.append({
                'org_id': org_id,
                'comment': comment,
                'author_id': author_id,
                'service_slug': service_slug,
                'user_id': req.get('user_id'),
                'department_id': req.get('department_id'),
                'group_id': req.get('group_id'),
            })
        self.bulk_create(data, strategy=Values(on_conflict=Values.do_nothing))

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('user_id', can_be_list=True) \
            ('department_id', can_be_list=True) \
            ('group_id', can_be_list=True) \
            ('org_id') \
            ('service_slug', can_be_list=True) \
            ('external_id', can_be_list=True)

        return distinct, filter_parts, joins, used_filters

    def get_select_related_data(self, select_related):
        if not select_related:
            return [self.default_all_projection], [], []

        select_related = select_related or []
        projections = []
        joins = []
        processors = []

        if 'user' in select_related:
            projections += [
                'requested_user_service_licenses.org_id',
                'requested_user_service_licenses.user_id',
                'users.id AS "user.id"',
                'users.nickname AS "user.nickname"',
                'users.gender AS "user.gender"',
                'users.department_id AS "user.department_id"',
                'users.name AS "user.name"',
                'users.position_plain AS "user.position"',
            ]
            joins.append("""
            LEFT OUTER JOIN users ON (
                requested_user_service_licenses.org_id = users.org_id AND
                requested_user_service_licenses.user_id = users.id
            )
            """)
            processors.append(
                set_to_none_if_no_id('user')
            )

        if 'department' in select_related:
            projections += [
                'requested_user_service_licenses.org_id',
                'requested_user_service_licenses.department_id',
                'departments.id AS "department.id"',
                'departments.name AS "department.name"',
                'departments.parent_id AS "department.parent_id"',
                'departments.members_count AS "department.members_count"',
            ]
            joins.append("""
            LEFT OUTER JOIN departments ON (
                requested_user_service_licenses.org_id = departments.org_id AND
                requested_user_service_licenses.department_id = departments.id
            )
            """)
            processors.append(
                set_to_none_if_no_id('department')
            )

        if 'group' in select_related:
            projections += [
                'requested_user_service_licenses.org_id',
                'requested_user_service_licenses.group_id',
                'groups.id AS "group.id"',
                'groups.name AS "group.name"',
                'groups.type AS "group.type"',
                'groups.members_count AS "group.members_count"',
            ]
            joins.append("""
            LEFT OUTER JOIN groups ON (
                requested_user_service_licenses.org_id = groups.org_id AND
                requested_user_service_licenses.group_id = groups.id
            )
            """)
            processors.append(
                set_to_none_if_no_id('group')
            )

        if 'author' in select_related:
            projections += [
                'requested_user_service_licenses.org_id',
                'requested_user_service_licenses.user_id',
                'author.id AS "author.id"',
                'author.nickname AS "author.nickname"',
                'author.gender AS "author.gender"',
                'author.department_id AS "author.department_id"',
                'author.name AS "author.name"',
                'author.position_plain AS "author.position"',
            ]
            joins.append("""
            JOIN users as author ON (
                requested_user_service_licenses.org_id = author.org_id AND
                requested_user_service_licenses.author_id = author.id
            )
            """)
            processors.append(
                set_to_none_if_no_id('author')
            )

        return projections, joins, processors


class OrganizationServicesAnalyticsInfoModel(BaseAnalyticsModel):
    db_alias = 'main'
    table = 'organizations_services_analytics_info'
    primary_key = 'org_id'
    order_by = 'org_id'

    simple_fields = set([
        'slug',
        'org_id',
        'trial_expires',
        'ready_at',
        'enabled',
        'disable_reason',
        'enabled_at',
        'disabled_at',
        'for_date',
    ])

    all_fields = simple_fields

    def save(self, org_id=None):
        # сохраняем текущую информацию о платных сервисах
        if org_id:
            org_id_filter = 'AND organization_services.org_id=%(org_id)s'
        else:
            org_id_filter = ''

        query = '''
                   INSERT INTO organizations_services_analytics_info(
                                    slug,
                                    org_id,
                                    trial_expires,
                                    ready_at,
                                    enabled,
                                    disable_reason,
                                    enabled_at,
                                    disabled_at
                               )
                       SELECT
                           services.slug,
                           organization_services.org_id,
                           organization_services.trial_expires,
                           (organization_services.ready_at AT TIME ZONE 'UTC')::DATE,
                           organization_services.enabled,
                           organization_services.disable_reason,
                           (organization_services.enabled_at AT TIME ZONE 'UTC')::DATE,
                           (organization_services.disabled_at AT TIME ZONE 'UTC')::DATE
                       FROM organization_services
                           JOIN
                               services
                           ON services.id = organization_services.service_id
                       WHERE
                           services.paid_by_license = True
                           AND organization_services.ready = True
                           {org_id_filter}
                   ON CONFLICT (slug, org_id, for_date) DO UPDATE
                       SET trial_expires = excluded.trial_expires,
                           ready_at = excluded.ready_at,
                           enabled = excluded.enabled,
                           disable_reason = excluded.disable_reason,
                           enabled_at = excluded.enabled_at,
                           disabled_at = excluded.disabled_at;
        '''

        query = query.format(
            org_id_filter=org_id_filter,
        )
        query = self.mogrify(
            query,
            {
                'org_id': org_id,
            }
        )

        with log.name_and_fields('analytics', org_id=org_id, for_date=utcnow().isoformat()):
            log.info('Saving organization services analytics data...')
            self._connection.execute(query)
            log.info('Organization services analytics data has been saved')
            return True


def notify_about_trail_ended(main_connection):
    """
    Если триал закончился вчера и сервис при этом включен создадим про это событие
    """
    from intranet.yandex_directory.src.yandex_directory.core.actions import action_service_trial_end
    yesterday_date = utcnow().date() - datetime.timedelta(days=1)

    trial_expired_services = OrganizationServiceModel(main_connection). \
        fields('service_id', 'org_id'). \
        filter(trial_expires=yesterday_date)

    for org_service in trial_expired_services:
        service = ServiceModel(main_connection).filter(id=org_service['service_id']).one()
        with main_connection.begin_nested():
            action_service_trial_end(
                main_connection,
                org_id=org_service['org_id'],
                author_id=None,
                object_value=service,
            )


def set_trial_status_to_expired(main_connection):
    """
    Меняем 'trial_status' на expired, если кончился триал
    """

    services_to_change = OrganizationServiceModel(main_connection) \
        .fields('service_id', 'org_id') \
        .filter(
            # это поле в базе хранит только дату
            trial_expires__lt=utcnow().date(),
            trial_status=trial_status.in_progress,
            enabled=Ignore,
        )

    for org_service in services_to_change:
        OrganizationServiceModel(main_connection) \
            .filter(
                org_id=org_service['org_id'],
                service_id=org_service['service_id'],
                enabled=Ignore,
            ) \
            .update(
                trial_status=trial_status.expired
            )


def get_service_field_by_tld(service, tld, field, data_field='data_by_tld'):
    assert field in ('url', 'icon')
    data_by_tld = service[data_field]
    if data_by_tld:
        fallback_tlds = ['ru', 'com'] if tld in app.config['RKUB_TLDS'] else ['com']
        variants = [tld] + fallback_tlds
        for tld in variants:
            tld_data = data_by_tld.get(tld, {})
            value = tld_data.get(field)
            if value:
                return value


@cached_in_memory_with_ttl(600, key=lambda *args, **kwargs: hashkey(*args[1:], **kwargs))
def get_service_url(meta_connection, slug, tld='ru'):
    service = ServiceModel(meta_connection).filter(slug=slug).fields('redirect_tld').one()
    if service:
        return get_service_field_by_tld(service, tld, 'url', data_field='redirect_tld')


class UserServiceLicenseExternalData(BaseModel):
    """ Таблица для хранения информации, которую получаем от других сервисов, когда выдаем им лицензии.
    Например, для хранения внешних id лицензии пользователей.
    """
    db_alias = 'main'
    table = 'user_service_licenses_external_data'
    order_by = 'user_service_licenses_external_data.org_id, user_service_licenses_external_data.user_id'
    primary_key = 'user_id'
    all_fields = [
        'org_id',
        'user_id',
        'service_id',
        'external_data',
    ]
    json_fields = [
        'external_data'
    ]

    def create(self, user_id, org_id, external_data, service_id):
        data = {
            'user_id': user_id,
            'org_id': org_id,
            'service_id': service_id,
            'external_data': external_data,
        }
        return self.insert_into_db(**data)

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        # фильтр по user_id и org_id/resource_id позволяет найти
        # сервисы, к которым у пользователя есть доступ
        self.filter_by(filter_data, filter_parts, used_filters) \
            ('user_id', can_be_list=True) \
            ('org_id') \
            ('service_id', can_be_list=True)

        return distinct, filter_parts, joins, used_filters


class ServicesLinksModel(BaseModel):
    """
        Таблица для хранения информации о сервисах
    """
    db_alias = 'meta'
    table = 'services_links'
    order_by = '-priority'

    json_fields = (
        'actions',
        'data_by_tld',
        'data_by_language',
        'options',
    )
    all_fields = [
        'id',
        'slug',
        'data_by_tld',
        'data_by_language',
        'actions',
        'options',
        'features',
        'available_for_external_admin',
        'show_in_header',
        'priority',
        'in_new_tab',
        'can_be_enabled',
        'can_be_disabled',
        'settings_url',
        'always_enabled',
        'is_configurable',
    ]


    def create(
            self,
            slug,
            data_by_tld={},
            data_by_language={},
            actions=[],
            options={},
            features=[],
            available_for_external_admin=False,
            show_in_header=False,
            priority=1,
            in_new_tab=False,
            can_be_enabled=False,
            can_be_disabled=False,
            settings_url='',
            always_enabled=False,
            is_configurable=False,
    ):
        data = {
            'slug': slug,
            'data_by_tld': data_by_tld,
            'data_by_language': data_by_language,
            'actions': actions,
            'options': options,
            'features': features,
            'available_for_external_admin': available_for_external_admin,
            'show_in_header': show_in_header,
            'priority': priority,
            'in_new_tab': in_new_tab,
            'can_be_enabled': can_be_enabled,
            'can_be_disabled': can_be_disabled,
            'settings_url': settings_url,
            'always_enabled': always_enabled,
            'is_configurable': is_configurable,
        }
        result = self.insert_into_db(**data)
        return result

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('id') \
            ('slug') \
            ('show_in_header') \
            ('can_be_enabled') \
            ('can_be_disabled') \
            ('available_for_external_admin')
        return distinct, filter_parts, joins, used_filters

    def get_by_slug(self, slug, fields=None):
        return self.find(
            filter_data={'slug': slug},
            fields=fields,
            one=True,
        )


class TrackerBillingStatusModel(BaseModel):
    db_alias = 'main'
    table = 'tracker_billing_status'

    all_fields = [
        'id',
        'org_id',
        'payment_date',
        'payment_status',
        'client_id',
    ]

    def create(self, org_id, payment_date, client_id=None):
        data = {
            'org_id': org_id,
            'payment_date': ensure_date(payment_date),
            'payment_status': False,
            'client_id': client_id,
        }
        return self.insert_into_db(**data)

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('org_id') \
            ('payment_status') \
            ('payment_date')

        if 'id__in' in filter_data:
            ids = filter_data.get('id__in')

            filter_parts.append(
                self.mogrify(
                    'id IN %(ids)s',
                    {
                        'ids': tuple(ids)
                    }
                )
            )
            used_filters.append('id__in')

        if 'payment_date__lte' in filter_data:
            dt = filter_data.get('payment_date__lte')
            if isinstance(dt, datetime.datetime):
                value = dt.isoformat()
            else:
                value = dt

            filter_parts.append(
                self.mogrify(
                    'payment_date <= %(timestamp)s',
                    {
                        'timestamp': value
                    }
                )
            )
            used_filters.append('payment_date__lte')

        if 'payment_date__gt' in filter_data:
            dt = filter_data.get('payment_date__gt')
            if isinstance(dt, datetime.datetime):
                value = dt.isoformat()
            else:
                value = dt

            filter_parts.append(
                self.mogrify(
                    'payment_date > %(timestamp)s',
                    {
                        'timestamp': value
                    }
                )
            )
            used_filters.append('payment_date__gt')

        if 'payment_date__gte' in filter_data:
            dt = filter_data.get('payment_date__gte')
            if isinstance(dt, datetime.datetime):
                value = dt.isoformat()
            else:
                value = dt

            filter_parts.append(
                self.mogrify(
                    'payment_date >= %(timestamp)s',
                    {
                        'timestamp': value
                    }
                )
            )
            used_filters.append('payment_date__gte')

        return distinct, filter_parts, joins, used_filters


class TrackerBillingErrorModel(BaseModel):
    db_alias = 'main'
    table = 'tracker_billing_errors'

    all_fields = [
        'id',
        'org_id',
        'payment_date',
        'error',
        'created_at',
    ]

    def create(self, org_id, payment_date, error):
        data = {
            'org_id': org_id,
            'payment_date': ensure_date(payment_date),
            'error': error,
        }
        return self.insert_into_db(**data)

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('id') \
            ('org_id') \
            ('payment_date') \
            ('error') \
            ('created_at')

        return distinct, filter_parts, joins, used_filters
