# -*- coding: utf-8 -*-
import json
import uuid
from intranet.yandex_directory.src import settings
from collections import defaultdict
from copy import deepcopy
from datetime import datetime

from flask import g
import pytz

from intranet.yandex_directory.src.blackbox_client import (
    TIMEZONE_ATTRIBUTE,
    LANGUAGE_ATTRIBUTE,
    IS_AVAILABLE_ATTRIBUTE,
)
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_meta_connection,
    get_main_connection,
    get_shard_numbers,
)

from intranet.yandex_directory.src.yandex_directory.common.models.base import (
    BaseModel,
    set_to_none_if_no_id,
    BaseAnalyticsModel,
    Values,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    Ignore,
    get_user_domain_from_blackbox,
    prepare_for_tsquery,
    to_lowercase,
    get_localhost_ip_address,
    utcnow,
    make_simple_strings,
    get_user_data_from_blackbox_by_uid,
    get_user_data_from_blackbox_by_uids,
)
from intranet.yandex_directory.src.yandex_directory.core.db import queries
from intranet.yandex_directory.src.yandex_directory.core.models.robot_service import RobotServiceModel
from intranet.yandex_directory.src.yandex_directory.core.models.service import OrganizationServiceModel, ServiceModel, UserServiceLicenses
from intranet.yandex_directory.src.yandex_directory.core.models.organization import OrganizationModel
from intranet.yandex_directory.src.yandex_directory.core import exceptions
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    except_fields,
    is_outer_uid,
    is_yandex_team_uid,
    only_ids,
    objects_map_by_id,
    only_attrs,
    int_or_none, is_need_blackbox_info, is_cloud_uid, is_domain_uid, RANGE_PDD, RANGE_PASSPORT,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.users import responsible
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.core.utils.users.dismiss import dismiss_user
from functools import reduce

from intranet.yandex_directory.src.yandex_directory.passport.utils import convert_gender_from_passport_format, convert_gender_to_passport_format

CONTACT_TYPES = [
    'email',
    'phone_extension',
    'phone',
    'site',
    'icq',
    'twitter',
    'facebook',
    'skype',
    'telegram',
    'staff',
]

GENDER_TYPES = [
    'male',
    'female'
]


class UserRoles:
    user = 'user'
    admin = 'admin'
    deputy_admin = 'deputy_admin'
    outer_admin = 'outer_admin'

    admin_roles = ['admin', 'outer_admin']

    def get_group_type(self, role):
        from intranet.yandex_directory.src.yandex_directory.core.models.group import (
            GROUP_TYPE_ORGANIZATION_ADMIN,
            GROUP_TYPE_ORGANIZATION_DEPUTY_ADMIN,
        )

        role_groups = {
            self.admin: GROUP_TYPE_ORGANIZATION_ADMIN,
            self.deputy_admin: GROUP_TYPE_ORGANIZATION_DEPUTY_ADMIN,
        }
        return role_groups.get(role)


class FieldsMustComeTogether(RuntimeError):
    """
    Поля переданные в фильтрацию должны идти строго вместе.
    """

    def __init__(self, fields, *args, **kwargs):
        self.fields = fields
        super(FieldsMustComeTogether, self).__init__(*args, **kwargs)


class UserModel(BaseModel):
    db_alias = 'main'
    table = 'users'
    json_fields = [
        'aliases',
        'name',
        'position',
        'contacts',
        'about',
    ]
    date_fields = ['birthday']
    all_fields = [
        'id',
        'org_id',
        'login',
        'email',
        'department_id',
        'name',
        'gender',
        'position',
        'about',
        'birthday',
        'contacts',
        'aliases',
        'nickname',
        'is_dismissed',
        'is_sso',
        'external_id',
        'user_type',
        'created',
        'updated_at',
        'first_name',
        'last_name',
        'middle_name',
        'position_plain',
        'recovery_email',
        'role',
        'cloud_uid',
        # не из базы
        'department',
        'departments',
        'groups',
        'services',
        'is_admin',
        'is_robot',
        'is_outstaff',
        'service_slug',
        'is_enabled',
        'timezone',
        'language',
        'karma',
        'avatar_id',
        'tracker_licenses',
    ]
    select_related_fields = {
        'department': 'DepartmentModel',
        'tracker_licenses': 'UserServiceLicenses',
    }
    prefetch_related_fields = {
        'departments': 'DepartmentModel',
        'groups': 'GroupModel',
        'services': 'ServiceModel',
        'is_admin': None,
        'is_robot': None,
        'is_outstaff': None,
        'service_slug': None,
        'is_enabled': None,
        'timezone': None,
        'language': None,
        'karma': None,
        'avatar_id': None,

        # Получаем из blackbox
        'name': None,
        'first_name': None,
        'last_name': None,
        'gender': None,
        'birthday': None,
    }
    field_dependencies = {
        'departments': ['department.path', 'org_id'],
        'services': ['org_id'],
        'groups': ['org_id'],
        'is_admin': ['org_id'],
        'is_robot': ['org_id', 'user_type'],
        'is_outstaff': ['org_id', 'department.path'],
        'service_slug': ['org_id'],
        'is_enabled': ['org_id'],
        'avatar_id': ['org_id'],
        'tracker_licenses': ['id', 'org_id'],
        'first_name': ['org_id', 'email', 'name', 'nickname'],
        'last_name': ['org_id', 'email', 'name', 'nickname'],
        'name': ['org_id', 'email', 'name', 'nickname'],
        'gender': ['org_id', 'email', 'gender'],
        'birthday': ['org_id', 'email', 'birthday'],
        'timezone': ['org_id', 'email'],
        'language': ['org_id', 'email'],
        'karma': ['org_id', 'email'],
    }
    simple_fields = {
        'id',
        'org_id',
        'login',
        'email',
        'department_id',
        'name',
        'gender',
        'position',
        'about',
        'birthday',
        'contacts',
        'aliases',
        'nickname',
        'is_dismissed',
        'external_id',
        'user_type',
        'created',
        'updated_at',
        'first_name',
        'last_name',
        'middle_name',
        'position_plain',
        'recovery_email',
        'role',
        'cloud_uid',
        'is_sso',
    }
    blackbox_fields = {
        'is_enabled',
        'avatar_id',
        'name',
        'gender',
        'birthday',
        'first_name',
        'last_name',
        'timezone',
        'language',
        'karma',
    }

    def make_admin_of_organization(self, org_id, user_id):
        from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel
        # todo: test me
        # todo: make thread safe
        m_groups = GroupModel(self._connection)

        group = m_groups.get_or_create_admin_group(org_id)

        m_groups.add_member(
            org_id=org_id,
            group_id=group['id'],
            member={'type': 'user', 'id': user_id}
        )
        self.filter(org_id=org_id, id=user_id).update(role=UserRoles.admin)
        # установим в паспорте опцию, что пользователь теперь админ
        if not is_cloud_uid(user_id):
            app.passport.set_admin_option(user_id)

    def make_deputy_admin_of_organization(self, org_id, user_id):
        from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel
        m_groups = GroupModel(self._connection)

        group = m_groups.get_or_create_deputy_admin_group(org_id)

        m_groups.add_member(
            org_id=org_id,
            group_id=group['id'],
            member={'type': 'user', 'id': user_id}
        )
        self.filter(org_id=org_id, id=user_id).update(role=UserRoles.deputy_admin)

    def revoke_admin_permissions(self, org_id, user_id):
        from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel
        group_model = GroupModel(self._connection)
        admin_group = group_model.get_or_create_admin_group(org_id)
        group_model.remove_member(org_id, admin_group['id'], {'type': 'user', 'id': user_id})
        self.update(
            update_data={'role': UserRoles.user},
            filter_data={'org_id': org_id, 'id': user_id}
        )
        # снимим в паспорте опцию, что пользователь админ
        if not UserModel(self._connection).is_user_admin(user_id=user_id, exclude={org_id}):
            app.passport.reset_admin_option(user_id)

    def revoke_deputy_admin_permissions(self, org_id, user_id):
        from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel
        group_model = GroupModel(self._connection)
        deputy_admin_group = group_model.get_or_create_deputy_admin_group(org_id)
        group_model.remove_member(org_id, deputy_admin_group['id'], {'type': 'user', 'id': user_id})
        self.update(
            update_data={'role': UserRoles.user},
            filter_data={'org_id': org_id, 'id': user_id}
        )

    def is_user_owner(self, user_id, exclude=None):
        exclude = exclude or set()
        for shard in get_shard_numbers():
            with get_main_connection(shard=shard) as shard_connection:
                shard_org_ids = (
                    OrganizationModel(shard_connection)
                    .filter(
                        admin_uid=user_id,
                    )
                    .fields('id')
                    .scalar()
                )
                for org_id in shard_org_ids:
                    if org_id not in exclude:
                        return True
        return False

    def is_user_admin(self, user_id, exclude=None):
        exclude = exclude or set()
        for shard in get_shard_numbers():
            with get_main_connection(shard=shard) as shard_connection:
                shard_org_ids = (
                    UserModel(shard_connection)
                    .filter(
                        id=user_id,
                        role=UserRoles.admin,
                        is_dismissed=False,
                        is_robot=False
                    )
                    .fields('org_id')
                    .scalar()
                )
                for org_id in shard_org_ids:
                    if org_id not in exclude:
                        return True
        return False


    def get_organization_admins(self, org_id):
        """
        Получаем внутренних админов организации
        """
        admins = self.find(
            filter_data={
                'role': UserRoles.admin,
                'org_id': org_id,
            }
        )
        return admins

    def explain_fields(self, fields):
        projections, select_related, prefetch_related = super(UserModel, self).explain_fields(fields)

        for field in ['name', 'gender', 'birthday', 'first_name', 'last_name']:
            if field in fields:
                projections.append('%s.%s' % (self.table, field))
        return projections, select_related, prefetch_related

    def get_select_related_data(self, select_related):
        if not select_related:
            return [self.default_all_projection], [], []

        select_related = select_related or []
        projections = []
        joins = []
        processors = []

        if 'department' in select_related:

            # Используем поля, указанные для вложенного объекта
            # department.
            # Раньше мы так в select_related не делали и надо
            # будет придумать какое-то общее решение,
            # когда появится необходимость сделать так в другом месте.
            department_fields = select_related['department']
            for name in department_fields:

                projections.append(
                    'department.{0} AS "department.{0}"'.format(name)
                )

            projections += [
                'department.id AS "department.id"',
                'department.org_id AS "department.org_id"',
                'department_parent.id AS "department.parent.id"',
                'department_parent.name AS "department.parent.name"',
                'department_parent.parent_id AS "department.parent.parent_id"',
            ]
            joins.append("""
            LEFT OUTER JOIN departments as department ON (
                users.department_id = department.id AND users.org_id = department.org_id
            )
            LEFT OUTER JOIN departments as department_parent ON (
                department.parent_id = department_parent.id AND department.org_id = department_parent.org_id
            )
            """)

            processors.append(set_to_none_if_no_id('department'))

        if 'tracker_licenses' in select_related:
            projections += [
                'coalesce(tracker_licenses.items, \'[]\'::json) as "tracker_licenses"',
            ]
            joins.append("""
            LEFT JOIN LATERAL (
                select
                    user_service_licenses.user_id,
                    json_agg(user_service_licenses
                        order by user_service_licenses.via_group_id nulls first,
                        user_service_licenses.via_department_id nulls first
                    ) as items,
                    1 as exists
                from user_service_licenses
                where user_service_licenses.org_id = users.org_id
                and user_service_licenses.user_id = users.id
                and user_service_licenses.service_id = (select id from services where slug='tracker')
                group by user_service_licenses.user_id
            ) tracker_licenses on true
            """)

        return projections, joins, processors

    def prefetch_related(self, items, prefetch_related):
        from intranet.yandex_directory.src.yandex_directory.core.models.group import UserGroupMembership
        from intranet.yandex_directory.src.yandex_directory.core.models.department import is_outstaff

        if not prefetch_related or not items:
            return

        org_id = set(item['org_id'] for item in items)
        if len(org_id) > 1 and any(x not in ['name', 'birthday', 'gender', 'first_name', 'last_name'] for x in prefetch_related):
            raise RuntimeError('All groups should belong to one organization')
        org_id = org_id.pop()

        # TODO: rename
        # по факту, это поле должно называться all_groups, так как будет
        # содержать список всех групп, даже тех, в которые пользователь
        # входит опосредованно
        if 'groups' in prefetch_related:
            group_fields = prefetch_related['groups']
            user_groups = defaultdict(list)
            for group in UserGroupMembership(self._connection).find(
                filter_data={
                    'org_id': org_id,
                    'user_id': [i['id'] for i in items],
                },
                fields={
                    'user_id': True,
                    'groups': group_fields,
                },
            ):
                user_groups[group['user_id']].append(group['groups'])

            for user in items:
                user['groups'] = user_groups.get(user['id'], [])

        if 'departments' in prefetch_related:
            from intranet.yandex_directory.src.yandex_directory.core.models.department import DepartmentModel

            department_fields = prefetch_related['departments']

            item_parent_ids = [
                list(map(int, user['department']['path'].split('.')))
                if user.get('department')
                else [] # Роботы не состоят ни в одном отделе,
                        # поэтому у них цепочка отделов пуста
                for user in items
            ]
            all_parent_ids = reduce(lambda x, y: x+y, item_parent_ids)
            department_by_id = {}

            if all_parent_ids:  # todo: test me
                department_model = DepartmentModel(self._connection)
                departments = department_model.find(
                    filter_data={
                        'id': all_parent_ids,
                        'org_id': org_id,
                    },
                    fields=department_fields,
                )
                for department in departments:
                    department_by_id[department['id']] = department

            for i, user in enumerate(items):
                user['departments'] = [
                    department_by_id[id]
                    for id in item_parent_ids[i]
                ]

        if 'is_robot' in prefetch_related:
            for user in items:
                user['is_robot'] = user['user_type'] != 'user'

        if 'service_slug' in prefetch_related:
            org_ids = set(only_attrs(items, 'org_id'))
            robots = RobotServiceModel(self._connection) \
                     .filter(org_id=org_ids, uid=set(only_ids(items))) \
                     .fields('org_id', 'uid', 'slug')

            robot_services = {(r['org_id'], r['uid']): r['slug'] for r in robots}

            for user in items:
                if 'service_slug' in prefetch_related:
                    key = (user['org_id'], user['id'])
                    user['service_slug'] = robot_services.get(key, None)

        if 'is_admin' in prefetch_related:
            uids = only_ids(items)

            with get_meta_connection() as meta_connection:
                outer_admins = UserMetaModel(meta_connection).get_outer_admins(org_id, uids, fields=['id'])
            outer_admins = set(only_ids(outer_admins))

            internal_admins = UserGroupMembership(self._connection).find(
                filter_data={
                    'org_id': org_id,
                    'user_id': uids,
                    'group__type': 'organization_admin',
                },
                fields=['user_id'],
            )
            internal_admins = set(only_attrs(internal_admins, 'user_id'))

            for user in items:
                user['is_admin'] = user['id'] in outer_admins or user['id'] in internal_admins

        blackbox_prefetch_related = [
            elem for elem in prefetch_related
            if elem in self.blackbox_fields
        ]
        if blackbox_prefetch_related:
            need_all_blackbox_users = False
            if any(x in ['is_enabled', 'avatar_id', 'timezone', 'language', 'karma'] for x in blackbox_prefetch_related):
                need_all_blackbox_users = True

            if getattr(g, 'user', None) and g.user.ip:
                user_ip = g.user.ip
            else:
                user_ip = get_localhost_ip_address()

            if need_all_blackbox_users:
                uids = [user['id'] for user in items]
            else:
                uids = [user['id'] for user in items if is_need_blackbox_info(user['id'], user['email'])]

            if uids:
                userinfo = get_user_data_from_blackbox_by_uids(
                    ip=user_ip,
                    uids=uids,
                    attributes=[IS_AVAILABLE_ATTRIBUTE, TIMEZONE_ATTRIBUTE],
                )
                for user in items:
                    bb_user = userinfo.get(str(user['id']), {})
                    if 'is_enabled' in blackbox_prefetch_related:
                        user['is_enabled'] = bb_user.get('is_available', True)
                    if 'avatar_id' in blackbox_prefetch_related:
                        user['avatar_id'] = bb_user.get('avatar_id')
                    if 'timezone' in blackbox_prefetch_related:
                        user['timezone'] = bb_user.get('attributes', {}).get(TIMEZONE_ATTRIBUTE)
                    if 'language' in blackbox_prefetch_related:
                        user['language'] = bb_user.get('language')
                    if 'karma' in blackbox_prefetch_related:
                        user['karma'] = bb_user.get('karma')

                    if is_need_blackbox_info(user['id'], user.get('email', '')):
                        if 'name' in blackbox_prefetch_related:
                            user['name'] = {
                                'first': bb_user.get('first_name') or user['nickname'],
                                'last': bb_user.get('last_name') or user['nickname']
                            }
                        if 'first_name' in blackbox_prefetch_related:
                            user['first_name'] = bb_user.get('first_name') or user['nickname']
                        if 'last_name' in blackbox_prefetch_related:
                            user['last_name'] = bb_user.get('last_name') or user['nickname']
                        if 'gender' in blackbox_prefetch_related:
                            user['gender'] = convert_gender_from_passport_format(bb_user.get('sex', '0'))
                        if 'birthday' in blackbox_prefetch_related:
                            user['birthday'] = bb_user.get('birth_date')

        if 'services' in prefetch_related:
            # получаем все сервисы, к которым у пользователя есть доступ
            # сервис доступен пользователю, если он бесплатный, либо лицензионный и находится в триальном периоде,
            # либо у пользователя есть лицензии
            service_fields = prefetch_related['services']
            user_services = defaultdict(list)
            org_services = OrganizationServiceModel(self._connection).find(
                filter_data={
                    'org_id': org_id,
                    'enabled': True
                },
                fields=['service_id'],
            )
            if org_services:
                org_service_ids = only_attrs(org_services, 'service_id')
                with get_meta_connection() as meta_connection:
                    services = ServiceModel(meta_connection).find(
                        filter_data={'id': org_service_ids},
                        fields=service_fields,
                    )
                # получаем id лицензионных сервисов, у которых закончился триальный период
                licensed_services = list(OrganizationServiceModel(self._connection).get_org_services_with_licenses(
                    org_id,
                    trial_expired=True,
                    only_id=True,
                ).values())
                if licensed_services:
                    # ищем пользователей, у которых есть лицензии
                    service_users = UserServiceLicenses(self._connection).get_users_with_service_licenses(
                        org_id,
                        licensed_services,
                        only_ids(items)
                    )
                common_services = []    # сервисы, доступные всем
                for service in services:
                    if service['id'] in licensed_services:
                        for user_id in service_users[service['id']]:
                            user_services[user_id].append(service)
                    else:
                        common_services.append(service)
                for user in items:
                    user['services'] = user_services.get(user['id'], []) + common_services
            else:
                for user in items:
                    user['services'] = []

        if 'is_outstaff' in prefetch_related:
            for user in items:
                department = user['department']
                # Иногда в тестах бывает, что пользователь без департамента
                # поэтому тут мы обрабатываем такую ситуацию, и считаем
                # его внештатником
                if department:
                    path = department['path']
                    user['is_outstaff'] = is_outstaff(path)
                else:
                    user['is_outstaff'] = False

    def get_suggest_filters(self, suggest):
        condition = '(make_user_search_ts_vector2(users.first_name, users.last_name, users.nickname) @@ %(text)s::tsquery)'
        text = prepare_for_tsquery(suggest)
        return self.mogrify(
            condition,
            {'text': text}
        )

    def get_filters_data(self, filter_data):
        from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel, relation_name

        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_data_dismissed = deepcopy(filter_data)
        if 'is_dismissed' not in filter_data_dismissed:
            filter_data_dismissed['is_dismissed'] = False

        filter_parts = []
        joins = []
        used_filters = []
        if filter_data.get('is_dismissed', False) is Ignore: # убираем из фильтра
            del filter_data_dismissed['is_dismissed']

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('id', can_be_list=True) \
            ('external_id', can_be_list=True) \
            ('email') \
            ('org_id') \
            ('department_id', can_be_list=True) \
            ('nickname', can_be_list=True) \
            ('user_type') \
            ('created', cast='timestamptz') \
            ('created__gt', cast='timestamptz') \
            ('created__lt', cast='timestamptz') \
            ('updated_at', cast='timestamptz') \
            ('updated_at__gt', cast='timestamptz') \
            ('updated_at__lt', cast='timestamptz') \
            ('role', can_be_list=True) \
            ('cloud_uid') \
            ('is_sso')

        if 'is_pdd' in filter_data:
            if 'id__gt' in filter_data or 'id__lt' in filter_data:
                raise RuntimeError('Filter "is_pdd" cannot be used at the same time as filter "id"')
            if filter_data['is_pdd']:
                filter_data['id__gt'] = RANGE_PDD[0]
                filter_data['id__lt'] = RANGE_PDD[1]
            else:
                filter_data['id__lt'] = RANGE_PASSPORT[1]
            used_filters.append('is_pdd')

        if 'id__lt' in filter_data and 'id__lt' not in used_filters:
            filter_parts.append(
                self.mogrify(
                    'users.id < %(id)s',
                    {
                        'id': filter_data.get('id__lt')
                    }
                )
            )
            used_filters.append('id__lt')

        if 'id__gt' in filter_data and 'id__gt' not in used_filters:
            filter_parts.append(
                self.mogrify(
                    'users.id > %(id)s',
                    {
                        'id': filter_data.get('id__gt')
                    }
                )
            )
            used_filters.append('id__gt')

        if 'is_dismissed' in filter_data_dismissed:
            is_dismissed = filter_data_dismissed.get('is_dismissed')
            if hasattr(is_dismissed, '__iter__') and not isinstance(is_dismissed, (str, bytes)):
                is_dismissed = tuple(is_dismissed)
                operator = 'IN'
            else:
                operator = '='
            filter_parts.append(
                self.mogrify(
                    'users.is_dismissed {operator} %(is_dismissed)s'.format(
                        operator=operator
                    ),
                    {
                        'is_dismissed': is_dismissed
                    }
                )
            )
            used_filters.append('is_dismissed')

        # is_dismissed - не учитываем при фильтрации внутри filter_data
        # (только внутри filter_data_dismissed),
        # но добавляем в used_filters, чтоб проходила проверка
        # get_filters_and_check
        if 'is_dismissed' in filter_data:
            used_filters.append('is_dismissed')

        if 'is_robot' in filter_data:
            if not isinstance(filter_data['is_robot'], bool):
                try:
                    filter_data['is_robot'] = int(filter_data['is_robot'])
                except ValueError:
                    raise exceptions.ValidationQueryParametersError('is_robot')

            if filter_data['is_robot']:
                filter_parts.append('users.user_type != \'user\'')
            else:
                filter_parts.append('users.user_type = \'user\'')

            used_filters.append('is_robot')

        if 'suggest' in filter_data:
            mogrify_condition = self.get_suggest_filters(
                filter_data.get('suggest')
            )
            filter_parts.append(mogrify_condition)
            used_filters.append('suggest')

        if 'recursive_department_id' in filter_data:
            department_id = filter_data.get('recursive_department_id')
            joins.append("""
                LEFT OUTER JOIN departments ON (
                    users.org_id = departments.org_id AND
                    users.department_id = departments.id
                )
                """)
            if isinstance(department_id, (list, tuple, set)):
                department_id = tuple(department_id)
            else:
                department_id = [department_id]

            condition = self.mogrify(
                """
                departments.id IS NOT NULL AND
                departments.path ~
                CAST(concat('*.', %(department_id)s, '.*') AS lquery)
                """,
                {
                    'department_id': department_id[0],
                }
            )
            for i in range(1, len(department_id)):
                    condition += self.mogrify(
                        """
                        OR departments.path ~
                        CAST(concat('*.', %(department_id)s, '.*') AS lquery)
                        """,
                        {
                            'department_id': department_id[i],
                        }
                    )
            filter_parts.append(condition)
            used_filters.append('recursive_department_id')

        group_model = GroupModel(self._connection)

        if 'group_id' in filter_data:
            # этот фильтр выводит пользователей, которые
            # входят в группу без учёта вложенности

            # добавляем фильтр по resource_relation_name
            # для вывода только тех кто входит в группу
            filter_data['resource_relation_name'] = relation_name.include
            used_filters.append('group_id')

            joins.append("""
                LEFT OUTER JOIN resource_relations ON (
                    users.org_id = resource_relations.org_id AND
                    users.id = resource_relations.user_id
            )
            """)
            groups = group_model.find(
                filter_data=dict(
                    org_id=filter_data['org_id'],
                    id=filter_data['group_id'],
                )
            )
            if not groups:
                resources = ('-1',)
            else:
                resources = tuple([group['resource_id'] for group in groups])
            filter_parts.append(self.mogrify(
                'resource_relations.resource_id IN %(resources)s',
                {
                    'resources': resources
                }
            ))

        if 'group_id__notin' in filter_data:
            used_filters.append('group_id__notin')
            groups = group_model.find(
                filter_data=dict(
                    org_id=filter_data['org_id'],
                    id=filter_data['group_id__notin'],
                )
            )
            if not groups:
                resources = ('-1',)
            else:
                resources = tuple([group['resource_id'] for group in groups])
            filter_parts.append(self.mogrify(
                """
                NOT EXISTS (
                    SELECT 1
                    FROM resource_relations rr
                    WHERE rr.org_id = users.org_id
                      AND rr.user_id = users.id
                      AND rr.resource_id IN %(resources)s
                      AND rr.name = '{relation_name}'
                )
                """.format(relation_name=relation_name.include),
                {
                    'resources': resources
                }
            ))

        if 'recursive_group_id' in filter_data:
            # этот фильтр выводит пользователей, которые
            # входят в группу c учётом вложенности

            used_filters.append('recursive_group_id')
            groups = group_model.find(
                filter_data=dict(
                    org_id=filter_data['org_id'],
                    id=filter_data['recursive_group_id']
                )
            )
            if not groups:
                resource_ids = ('-1',)
            else:
                resource_ids = tuple([group['resource_id']
                                      for group in groups])
            # просто делегируем фильтрацию механизму, который отдает
            # людей по ресурсу
            filter_data['resource'] = resource_ids
            filter_data['resource_service'] = Ignore

            # добавляем фильтр по resource_relation_name
            # для вывода только тех кто входит в группу
            filter_data['resource_relation_name'] = relation_name.include

        if 'group_id__in' in filter_data:
            joins.append("""
                       LEFT OUTER JOIN user_group_membership ON (
                           users.org_id = user_group_membership.org_id AND users.id = user_group_membership.user_id
                       )
                       """)
            filter_parts.append(
                self.mogrify(
                    'user_group_membership.group_id IN %(group_ids)s',
                    {
                        'group_ids': tuple(filter_data.get('group_id__in'))
                    }
                )
            )
            used_filters.append('group_id__in')


        fields_together = ('resource_service', 'resource')

        if not self._check_fields_come_together(fields_together, filter_data):
            raise FieldsMustComeTogether(fields_together)

        if 'resource_service' in filter_data:
            service_slug = filter_data['resource_service']
            if service_slug is not Ignore:
                filter_parts.append(self.mogrify(
                    'resources.service = %(resources)s',
                    {
                        'resources': service_slug
                    }
                ))

            used_filters.append('resource_service')

        if 'resource' in filter_data or 'resource_relation_name' in filter_data:
            if 'resource_relations' not in ' '.join(joins):
                joins.append("""
                LEFT OUTER JOIN resource_relations ON (
                    users.org_id = resource_relations.org_id
                )
                """)

            joins.append("""
            LEFT OUTER JOIN departments ON (
                users.org_id = departments.org_id AND
                users.department_id = departments.id
            )
            """)
            joins.append("""
            LEFT OUTER JOIN user_group_membership ON (
                users.org_id = user_group_membership.org_id AND
                users.id = user_group_membership.user_id AND
                resource_relations.group_id = user_group_membership.group_id
            )
            """)
            joins.append("""
            LEFT OUTER JOIN resources ON (
                resource_relations.resource_id = resources.id
            )
            """)

            if 'resource' in filter_data:
                # так как пользователь может быть связан с ресурсом
                # более чем одним способом, то фильтр по ресурсу
                # может выдать более одной записи. Поэтому тут нужно
                # сделать DISTINCT
                distinct = True

                resources = filter_data.get('resource')
                if hasattr(resources, '__iter__') and not isinstance(resources, (str, bytes)):
                    resources = tuple(resources)
                    operator = 'IN'
                else:
                    operator = '='

                service_slug = filter_data['resource_service']
                if service_slug is Ignore:
                    log.debug('Ignored service slug when filtering by resource')
                filter_by_resources = self.mogrify(
                    'resources.external_id {operator} %(resources)s'.format(operator=operator),
                    {
                        'resources': resources,
                    }
                )
                used_filters.append('resource')
            else:
                filter_by_resources = ''

            if 'resource_relation_name' in filter_data:
                names = filter_data.get('resource_relation_name')
                if hasattr(names, '__iter__') and not isinstance(names, (str, bytes)):
                    names = tuple(names)
                    operator = 'IN'
                else:
                    operator = '='
                filter_by_relation_names = self.mogrify(
                    'resource_relations.name {operator} %(names)s'.format(operator=operator),
                    {
                        'names': names
                    }
                )
                used_filters.append('resource_relation_name')
            else:
                filter_by_relation_names = ''

            top_filters = ' AND '.join(
                i for i in
                [
                    filter_by_resources,
                    filter_by_relation_names,
                ] if i
            )

            filter_parts.append(
                self.mogrify(
                    """
                    {top_filters} AND
                    (
                        (
                            resource_relations.user_id = users.id
                        )
                            OR
                        (
                            resource_relations.department_id IS NOT NULL AND
                            departments.path ~ CAST(concat('*.', resource_relations.department_id, '.*') AS lquery)
                        )
                            OR
                        (
                            resource_relations.group_id IS NOT NULL AND
                            resource_relations.group_id = user_group_membership.group_id
                        )
                    )
                    """.format(top_filters=top_filters),
                )
            )

        if 'service' in filter_data:
            distinct = True
            processed_filter = filter_data['service'].split('.')
            service_slug = processed_filter[0]
            license_issued = 'license_issued' in processed_filter

            joins.append("""
                        LEFT OUTER JOIN organization_services ON (
                            users.org_id = organization_services.org_id
                        )
                        """)
            joins.append("""
                        LEFT OUTER JOIN user_service_licenses ON (
                            users.org_id = user_service_licenses.org_id AND
                            users.id = user_service_licenses.user_id AND
                            organization_services.service_id = user_service_licenses.service_id
                        )
                        """)
            joins.append("""
                        LEFT OUTER JOIN services ON (
                            services.id = organization_services.service_id
                        )
                        """)
            query = """
                    services.slug = %(service_slug)s AND
                    organization_services.enabled=True AND
                    ({common_services_filter} user_service_licenses.service_id IS NOT NULL)
                    """

            if license_issued:
                # отдаем только пользователей, у которых есть выданные лицензии на сервис
                common_services_filter = ''
            else:
                # отдаем всех пользователей, у которых есть доступ к сервису
                # если сервис бесплатный или в триальном периоде
                common_services_filter = """
                                 organization_services.trial_status != 'expired' OR
                                 """
            filter_parts.append(
                self.mogrify(
                    query.format(common_services_filter=common_services_filter),
                    {
                        'service_slug': service_slug
                    }
                )
            )
            used_filters.append('service')

        if 'alias' in filter_data:
            filter_parts.append(
                self.mogrify(
                    'users.aliases::jsonb @> %(alias)s',
                    {
                        'alias': json.dumps(filter_data.get('alias'))
                    }
                )
            )
            used_filters.append('alias')

        if str(filter_data.get('tracker_licenses', 0)) == '1':
            joins.append("""
            INNER JOIN LATERAL (
                select distinct on (user_id) *
                from user_service_licenses
                where user_service_licenses.org_id = users.org_id
                and user_service_licenses.user_id = users.id
                and user_service_licenses.service_id = (select id from services where slug='tracker')
            ) usl_tracker_filter on true
            """)
            used_filters.append('tracker_licenses')

        return distinct, filter_parts, joins, used_filters

    def get(self, user_id=None, org_id=None, fields=None, is_dismissed=False, is_cloud=False):
        filter_data = {
            'is_dismissed': is_dismissed,
        }
        if is_cloud:
            filter_data['cloud_uid'] = user_id
        else:
            filter_data['id'] = user_id
        if org_id:
            filter_data['org_id'] = org_id

        return self.find(
            filter_data=filter_data,
            fields=fields,
            one=True,
        )

    def get_by_external_id(self, org_id, id):
        return self.find(
            filter_data={
                'org_id': org_id,
                'external_id': id,
            },
            one=True,
        )

    def create(self,
               id,
               nickname,
               name,
               email,
               gender,
               org_id,
               aliases=None,
               department_id=None,
               groups=None,
               position=None,
               about=None,
               birthday=None,
               contacts=None,
               external_id=None,
               user_type='user',
               calc_disk_usage=True,
               update_groups_leafs_cache=True,
               timezone=None,
               language=None,
               cloud_uid=None,
               is_dismissed=False,
               is_sso=False
               ):
        from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel

        if not aliases:
            aliases = []

        nickname = to_lowercase(nickname)
        email = to_lowercase(email)
        name_plain = make_simple_strings(name)

        connection = self._connection
        user = dict(
            connection.execute(
                queries.CREATE_USER['query'],
                self.prepare_dict_for_db({
                    'id': id,
                    'name': name,
                    'nickname': nickname,
                    'aliases': aliases,
                    'email': email,
                    'gender': gender,
                    'org_id': org_id,
                    'department_id': department_id,
                    'position': position,
                    'about': about,
                    'birthday': birthday,
                    'contacts': contacts,
                    'external_id': external_id,
                    'user_type': user_type,
                    'first_name': name_plain['first'],
                    'last_name': name_plain.get('last', ''),
                    'middle_name': name_plain.get('middle', ''),
                    'position_plain': make_simple_strings(position) or None,
                    'cloud_uid': cloud_uid,
                    'is_dismissed': is_dismissed,
                    'is_sso': is_sso,
                })
            ).fetchone()
        )

        # groups
        if groups is not None:
            GroupModel(connection=connection).set_user_groups(
                org_id=user['org_id'],
                user_id=user['id'],
                groups=groups,
                update_leafs_cache=update_groups_leafs_cache,
            )
        user['groups'] = groups or []

        # Пересчитаем лимит дискового пространства
        # Импорт здесь нужен, чтобы избежать циклической зависимости
        if calc_disk_usage:
            from intranet.yandex_directory.src.yandex_directory.core.models.disk_usage import DiskUsageModel
            DiskUsageModel(connection).update_organization_limits(org_id)

        if timezone:
            UserModel(connection).change_timezone(id, timezone)
        if language:
            UserModel(connection).change_language(id, language)

        return user

    def update(self, update_data, filter_data=None):
        if 'updated_at' not in update_data:
            update_data = update_data.copy()
            update_data['updated_at'] = utcnow()
        return super(UserModel, self).update(update_data, filter_data=filter_data)

    def update_one(self, update_data, filter_data=None, group_types=('generic',)):
        """
        Если в update_data присутствует поле groups, то по умолчанию заменяются
        лишь группы с типом generic. Чтобы обновить группы всех типов, нужно
        передать аргумент group_types=None. Так же, можно передавать список типов
        групп, которые нужно заменить.
        """
        from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel
        groups = update_data.get('groups')
        update_data = except_fields(update_data, 'groups')
        if 'name' in update_data:
            name_plain = make_simple_strings(update_data['name'])
            update_data['first_name'] = name_plain['first']
            update_data['last_name'] = name_plain.get('last', '')
            update_data['middle_name'] = name_plain.get('middle', '')
        if 'position' in update_data:
            update_data['position_plain'] = make_simple_strings(update_data['position'])

        if update_data:
            self.update(update_data, filter_data)

        if groups is not None:
            GroupModel(self._connection).set_user_groups(
                org_id=filter_data['org_id'],
                user_id=filter_data['id'],
                groups=groups,
                group_types=group_types,
            )

    def is_responsible(self, uid):
        """
        Вернуть True если пользователь ответственный за любой сервис на одном из шардов.

        :rtype: bool
        """
        return responsible.is_responsible(uid)

    def dismiss(self,
                org_id,
                user_id,
                author_id,
                old_user=None,
                # эти аргументы нужны для того, чтобы можно было
                # пропустить походы во внешние сервисы при генерации документации
                # про события
                skip_disk=False,
                skip_passport=False):
        with get_meta_connection(for_write=True) as meta_connection:
            return dismiss_user(
                meta_connection, self._connection,
                org_id=org_id,
                user_id=user_id,
                author_id=author_id,
                old_user=old_user,
                skip_disk=skip_disk,
                skip_passport=skip_passport
            )

    def change_password(self, org_id, author_id, user_id, new_password, force_next_login_password_change=False):
        """
            Смена пароля пользователя
            Возвращает boolean-статус успешности смены пароля и список ошибок при наличии

            Args:
                org_id (int): id организации пользователя
                author_id (int): uid автора изменения
                user_id (int): uid пользователя, которому меняем пароль
                new_password (str): новый пароль
                force_next_login_password_change (boolean): требовать принудительной смены пароля при следующем логине
        """
        changed = app.passport.change_password(
            uid=user_id,
            new_password=new_password,
            force_next_login_password_change=force_next_login_password_change,
        )
        if changed:
            from intranet.yandex_directory.src.yandex_directory.core.actions import action_security_user_password_changed

            self.sync_with_passport(user_id, org_id)
            action_security_user_password_changed(
                self._connection,
                org_id=org_id,
                author_id=author_id,
                object_value=user_id,
            )
        return changed
        # ToDo: я здесь поудалял возврат про ошибки. Надо проверить, не сломал ли чего.

    @staticmethod
    def is_enabled(uid, user_ip):
        """
        Проверяет не заблокирован ли пользователь в паспорте.
        Если атрибут is_available (1009) в паспорте установлен в 1 - пользователь не заблокирован
        """
        if is_yandex_team_uid(uid):
            return True

        userinfo = get_user_data_from_blackbox_by_uid(
            uid, ip=user_ip,
            attributes=[IS_AVAILABLE_ATTRIBUTE],
            dbfields=[],
        )
        return userinfo['is_available']

    def change_is_enabled_status(self, org_id, author_id, user_id, is_enabled):
        if is_outer_uid(user_id):
            with log.fields(uid=user_id):
                log.error('Trying to block outer user, action denied')
            return False

        if self.is_responsible(user_id):
            raise exceptions.UnableToBlockServiceResponsible()

        from intranet.yandex_directory.src.yandex_directory.core.actions import (
            action_security_user_unblocked,
            action_security_user_blocked,
        )
        if is_enabled:
            ok_to_call_action = app.passport.unblock_user(user_id)
            action_method = action_security_user_unblocked

            self.sync_with_passport(user_id, org_id)
        else:
            ok_to_call_action = app.passport.block_user(user_id)
            action_method = action_security_user_blocked

        if ok_to_call_action:
            action_method(
                self._connection,
                org_id=org_id,
                author_id=author_id,
                object_value=user_id
            )
        return ok_to_call_action

    def is_admin(self, org_id, user_id):
        return bool(self.filter(org_id=org_id, id=user_id, role=UserRoles.admin).one())

    def is_deputy_admin(self, org_id, user_id):
        return bool(self.filter(org_id=org_id, id=user_id, role=UserRoles.deputy_admin).one())

    def is_last_admin(self, org_id, user_id):
        outer_admins = self.filter(
            role=UserRoles.admin, org_id=org_id
        ).fields('id').scalar()

        this_user_is_outer_admin = False
        outer_admin_cnt = 0
        for outer_user_id in outer_admins:
            if is_outer_uid(outer_user_id):
                if outer_user_id == user_id:
                    this_user_is_outer_admin = True
                outer_admin_cnt += 1

        return this_user_is_outer_admin and outer_admin_cnt == 1

    def delete(self, filter_data=None, force=False, force_remove_all=False):
        # если не переданы данные с фильтром и org_id, то ничего
        # не удаляем
        if not filter_data or not filter_data.get('org_id'):
            return

        deleted_data = deepcopy(filter_data)
        if force:
            deleted_data['is_dismissed'] = Ignore

        super(UserModel, self).delete(deleted_data, force_remove_all=force_remove_all)

    def user_license_suggest(self, org_id, service_id, text, limit=None):
        # метод для получения пользователей с лицензиями и родительским контейнером,
        # через который ему выдана лицензия, если такой есть
        # используется в фильтре по пользователям с лицензиями

        query = """
                SELECT * FROM users
                JOIN user_service_licenses ON (
                    users.org_id = user_service_licenses.org_id AND
                    users.id = user_service_licenses.user_id
                )
                WHERE
                user_service_licenses.service_id = %(service_id)s AND
                user_service_licenses.org_id=%(org_id)s AND
                {suggest_filter} {limit_filter}
                """
        suggest_filter = self.get_suggest_filters(text)
        limit_filter = 'LIMIT %(limit)s' if limit else ''
        query = self.mogrify(
            query.format(suggest_filter=suggest_filter, limit_filter=limit_filter),
            {
                'org_id': org_id,
                'service_id': service_id,
                'limit': limit,
            }
        )

        result = self._connection.execute(query).fetchall()
        return list(map(dict, result))

    def change_user_role(self, org_id, user_id, new_role, admin_uid, old_user):
        from intranet.yandex_directory.src.yandex_directory.core.actions import (
            action_security_user_grant_organization_admin,
            action_security_user_revoke_organization_admin,
            action_security_user_grant_deputy_admin,
            action_security_user_revoke_deputy_admin,
        )
        func_map = {
            UserRoles.admin: {
                'add': (self.make_admin_of_organization, action_security_user_grant_organization_admin),
                'remove': (self.revoke_admin_permissions, action_security_user_revoke_organization_admin),
            },
            UserRoles.deputy_admin: {
                'add': (self.make_deputy_admin_of_organization, action_security_user_grant_deputy_admin),
                'remove': (self.revoke_deputy_admin_permissions, action_security_user_revoke_deputy_admin),
            }
        }
        if old_user['role'] != new_role:
            if func_map.get(old_user['role']):
                f, action = func_map[old_user['role']]['remove']
                f(org_id, user_id)
                action(
                    self._connection,
                    org_id=org_id,
                    author_id=admin_uid,
                    object_value=old_user
                )
            if func_map.get(new_role):
                f, action = func_map[new_role]['add']
                f(org_id, user_id)
                action(
                    self._connection,
                    org_id=org_id,
                    author_id=admin_uid,
                    object_value=old_user
                )

    @staticmethod
    def change_timezone(uid, timezone):
        """
        Меняет таймзону пользователя в паспорте
        :param uid: ID пользователя
        :type uid: int
        :param timezone: таймзона
        :type timezone: string
        """
        user_data = {
            'uid': uid,
            'timezone': timezone
        }
        app.passport.account_edit(user_data)

    @staticmethod
    def get_batch_userinfo(uids):
        """
        Получает информацию о пользователях из ЧЯ
        :param uids: ID пользователя
        :type uids: list
        :rtype: dict[dict]
        """
        if getattr(g, 'user', None) and g.user.ip:
            user_ip = g.user.ip
        else:
            user_ip = get_localhost_ip_address()

        user_info = app.blackbox_instance.batch_userinfo(
            uids=uids,
            userip=user_ip,
        )
        # Если в Паспорте учётки уже нет, то blackbox отдаёт
        # словарь в котором uid это пустая строка.
        # Такие записи надо исключить из результатов get_batch_userinfo.
        user_info = (
            item
            for item in user_info
            if item['uid']
        )
        return objects_map_by_id(user_info, key='uid', remove_key=True, transform_key=int)

    @staticmethod
    def change_language(uid, language):
        """
        Меняет язык пользователя в паспорте
        :param uid: ID пользователя
        :type uid: int
        :param language: язык
        :type language: string
        """
        user_data = {
            'uid': uid,
            'language': language,
        }
        app.passport.account_edit(user_data)

    def get_uids_by_resource(self, org_id, resource_service, resource_external_id, resource_relation_name=None):
        params = dict(
            org_id=org_id,
            resource_external_id=resource_external_id,
            resource_service=resource_service,
        )

        resource_relation_name_filter = ''
        if resource_relation_name:
            resource_relation_name_filter = 'and resource_relations.name = %(resource_relation_name)s'
            params['resource_relation_name'] = resource_relation_name

        sql = f"""
            with resource_relations_prepared as (
                select resource_relations.*
                from resources
                inner join resource_relations on (
                    resources.org_id = resource_relations.org_id
                    and resources.id = resource_relations.resource_id
                    {resource_relation_name_filter}
                )
                where resources.service = %(resource_service)s
                  and resources.org_id = %(org_id)s
                  and resources.external_id = %(resource_external_id)s
            ),
            users_prepared as (
                select *
                from users
                where users.org_id = %(org_id)s
                and users.is_dismissed = false
            )
            select up1.id
            from users_prepared as up1
            inner join resource_relations_prepared as rrp1 on (
               rrp1.org_id = up1.org_id
               and (rrp1.user_id = up1.id)
            )

            union
            select up2.id
            from users_prepared as up2
            inner join departments on (up2.org_id = departments.org_id AND up2.department_id = departments.id)
            inner join resource_relations_prepared as rrp2 on (
                rrp2.org_id = up2.org_id
                and (
                        rrp2.department_id IS NOT NULL
                        AND departments.path ~
                            CAST(concat('*.', rrp2.department_id, '.*') AS lquery)
                )
            )

            union
            select up3.id
            from users_prepared as up3
            inner join user_group_membership on (up3.org_id = user_group_membership.org_id
                                                    AND up3.id = user_group_membership.user_id)
            inner join resource_relations_prepared as rrp3 on (
                rrp3.org_id = up3.org_id
                and (
                    rrp3.group_id IS NOT NULL
                    AND rrp3.group_id = user_group_membership.group_id
                )
            )
        """
        data = self._connection.execute(sql, **params).fetchall()
        return set(only_ids(data))

    def sync_with_passport(self, uid, org_id):
        if not is_domain_uid(uid):
            return

        user = self.find(
            filter_data={'id': uid, 'org_id': org_id, 'is_dismissed': Ignore},
            fields=['name', 'gender', 'birthday', 'is_sso'],
            one=True
        )

        def prepare_name(name):
            if isinstance(name, dict):
                for x in name:
                    if name[x]:
                        return name[x]
                return ''
            return name

        if user is not None and not user['is_sso']:
            firstname = prepare_name(user.get('name', {}).get('first', ''))
            lastname = prepare_name(user.get('name', {}).get('last', ''))
            if not firstname or not lastname:
                return

            app.passport.account_edit({
                'uid': uid,
                'firstname': firstname,
                'lastname': lastname,
                'gender': convert_gender_to_passport_format(user.get('gender')),
                'birthday': user.get('birthday').strftime('%Y-%m-%d') if user.get('birthday') else '',
            })


class UserMetaModel(BaseModel):
    table = 'users'
    db_alias = 'meta'

    all_fields = [
        'id',
        'org_id',
        'is_outer',
        'user_type',
        'is_dismissed',
        'created',
        'cloud_uid',
        # not from db
        'organization',
    ]
    select_related_fields = {
        'organization': 'OrganizationMetaModel',
    }

    def create(self, id, org_id, user_type='inner_user', is_outer=None, cloud_uid=None, is_dismissed=False):
        if is_outer is None:
            is_outer = is_outer_uid(id)
        if is_outer and user_type == 'inner_user':
            user_type = 'outer_admin'
        return dict(
            self._connection.execute(
                queries.CREATE_META_USER['query'],
                self.prepare_dict_for_db({
                    'id': id,
                    'org_id': org_id,
                    'is_outer': is_outer,
                    'user_type': user_type,
                    'is_dismissed': is_dismissed,
                    'cloud_uid': cloud_uid,
                })
            ).fetchone()
        )

    def get_filters_data(self, filter_data):
        distinct = False

        # todo: test me
        if not filter_data:
            return distinct, [], [], []

        filter_parts = []
        joins = []
        used_filters = []

        self.filter_by(filter_data, filter_parts, used_filters) \
            ('id', can_be_list=True) \
            ('org_id', can_be_list=True) \
            ('is_outer') \
            ('is_dismissed') \
            ('user_type', can_be_list=True) \
            ('cloud_uid')

        return distinct, filter_parts, joins, used_filters

    def get_select_related_data(self, select_related):
        # todo: test me
        if not select_related:
            return [self.default_all_projection], [], []

        select_related = select_related or []
        projections = []
        joins = []
        processors = []

        if 'organization' in select_related:
            projections += [
                'users.org_id',
                'organization.id AS "organization.id"',
                'organization.label AS "organization.label"',
                'organization.shard AS "organization.shard"',
                'organization.ready AS "organization.ready"',
            ]
            joins.append("""
            LEFT OUTER JOIN organizations as organization ON (
                users.org_id = organization.id
            )
            """)

        return projections, joins, processors

    def get(self,
            user_id,
            is_outer=False,
            org_id=None,
            fields=None,
            user_type=None,
            is_cloud=False,
            ):

        find_data = {}
        if is_cloud:
            find_data['cloud_uid'] = user_id
        else:
            find_data['id'] = user_id

        if is_outer is not Ignore:
            find_data['is_outer'] = is_outer

        if org_id:
            find_data['org_id'] = org_id
        if user_type:
            find_data['user_type'] = user_type

        return self.find(
            find_data,
            fields=fields,
            one=True,
        )

    def get_outer_admins(self, org_id=None, uid=None, fields=None, one=False):
        find_data = {
            'user_type': 'outer_admin',
            'is_dismissed': False,
        }
        if org_id:
            find_data['org_id'] = org_id
        if uid:
            find_data['id'] = uid
        return self.find(
            find_data,
            fields=fields,
            one=one,
        )

    def get_outer_deputy_admins(self, org_id=None, uid=None, fields=None):
        find_data = {
            'user_type': 'deputy_admin',
        }
        if org_id:
            find_data['org_id'] = org_id
        if uid:
            find_data['id'] = uid
        return self.find(
            find_data,
            fields=fields,
        )


class UserDismissedModel(BaseModel):
    """
    Информация по уволенному сотруднику
    """
    db_alias = 'main'
    table = 'users_dismissed_info'
    order_by = 'user_id'
    primary_key = 'user_id'

    all_fields = [
        'org_id',
        'user_id',
        'dismissed_date',
        'department',
        'groups',
        'groups_admin',
    ]
    json_fields = ['department', 'groups', 'groups_admin']

    def create(self, org_id, uid, department, groups, groups_admin, dismissed_date=None):
        query = """
          INSERT INTO
            {table}
            (org_id, user_id, dismissed_date, department, groups, groups_admin)
          VALUES
            (%(org_id)s, %(user_id)s, %(dismissed_date)s, %(department)s, %(groups)s, %(groups_admin)s)
          ON CONFLICT (org_id, user_id) DO UPDATE SET
            dismissed_date = %(dismissed_date)s,
            department = %(department)s,
            groups = %(groups)s,
            groups_admin = %(groups_admin)s
          RETURNING *
        """.format(
            table=self.table,
        )

        result = self._connection.execute(
            query,
            self.prepare_dict_for_db({
                'org_id': org_id,
                'user_id': uid,
                'dismissed_date': dismissed_date or utcnow(),
                'department': department or {},
                'groups': groups or [],
                'groups_admin': groups_admin or [],
            })
        ).fetchone()
        return self.prepare_dict_from_db(dict(result))

    def get(self, user_id, org_id=None):
        filter_data = {
            'user_id': user_id,
        }
        if org_id:
            filter_data['org_id'] = org_id
        return self.find(
            filter_data=filter_data,
            limit=1,
            one=True,
        )

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts, joins, used_filters = [], [], []

        if 'user_id' in filter_data:
            filter_parts.append(
                self.mogrify(
                    'users_dismissed_info.user_id = %(user_id)s',
                    {
                        'user_id': filter_data.get('user_id')
                    }
                )
            )
            used_filters.append('user_id')

        if 'org_id' in filter_data:
            filter_parts.append(
                self.mogrify(
                    'users_dismissed_info.org_id = %(org_id)s',
                    {
                        'org_id': filter_data.get('org_id')
                    }
                )
            )
            used_filters.append('org_id')

        return distinct, filter_parts, joins, used_filters


class UsersAnalyticsInfoModel(BaseAnalyticsModel):
    db_alias = 'main'
    table = 'users_analytics_info'
    primary_key = 'org_id'
    order_by = 'org_id'

    simple_fields = set([
        'uid',
        'org_id',
        'email',
        'created',
        'licensed_services',
        'for_date'
    ])

    all_fields = simple_fields
    date_fields = ['for_date', 'created']

    def save(self, org_id=None, rewrite=False):
        # сохраняем информацию о пользователях на сегодня
        for_date = utcnow().date()

        if rewrite:
            self.delete({'for_date': for_date})

        if org_id:
            org_id_filter = 'AND users.org_id=%(org_id)s'
        else:
            org_id_filter = ''

        query = '''
                   INSERT INTO users_analytics_info(uid, org_id, email, created, licensed_services, for_date)
                       SELECT
                           users.id,
                           users.org_id,
                           users.email,
                           (users.created at time zone 'UTC')::date,
                           ARRAY(
                               SELECT services.slug
                               FROM user_service_licenses
                                   JOIN services
                                   ON services.id = user_service_licenses.service_id
                                   JOIN organization_services
                                   ON organization_services.org_id = user_service_licenses.org_id
                                   AND organization_services.service_id = user_service_licenses.service_id
                               WHERE
                                   user_service_licenses.user_id = users.id
                                   AND user_service_licenses.org_id = users.org_id
                                   AND organization_services.enabled
                           ),
                           %(for_date)s
                       FROM users
                       WHERE
                           NOT users.is_dismissed
                           AND users.user_type = 'user'
                           {org_id_filter}
                   ON CONFLICT DO NOTHING;
        '''

        query = query.format(
            org_id_filter=org_id_filter,
        )
        query = self.mogrify(
            query,
            {
                'for_date': for_date,
                'org_id': org_id,
            }
        )

        with log.name_and_fields('analytics', org_id=org_id, for_date=for_date):
            log.info('Saving users analytics data...')
            self._connection.execute(query)
            log.info('Users analytics data has been saved')


class AdminContactsAnalyticsInfoModel(BaseAnalyticsModel):
    db_alias = 'main'
    table = 'admin_contacts_analytics_info'
    primary_key = 'uid'
    order_by = 'uid'

    simple_fields = set([
        'uid',
        'org_id',
        'phones',
        'utc',
        'first_name',
        'last_name',
        'for_date',
    ])

    all_fields = simple_fields
    date_fields = ['for_date']

    batch_size = 1000
    temp_admins_table = 'temp_admins'

    def delete_old_data(self, days=5):
        self.delete(force_remove_all=True)

    def save(self):
        log.info('Saving admins contacts analytics data...')
        self.create_temp_admins_table()

        def batcher(connection, table=self.temp_admins_table, batch_size=self.batch_size):
            cursor_name = str(uuid.uuid4())
            with connection.connection.cursor(cursor_name) as cursor:
                query = '''
                            SELECT uid, org_id
                            FROM {table};
                        '''.format(table=table)
                cursor.execute(query)

                while True:
                    rows = cursor.fetchmany(batch_size)
                    if rows:
                        yield rows
                    else:
                        break

        batches = batcher(self._connection)
        for batch in batches:
            data_for_db = self.get_batch_admin_info(batch)
            if data_for_db:
                self.bulk_create(data_for_db, strategy=Values(on_conflict=Values.do_nothing))

        log.info('Admins contacts analytics saved')

    def create_temp_admins_table(self):
        # сохраняем всех админов во временную таблицу
        log.info('Create temp admins table')
        with get_meta_connection() as meta_connection:
            tracker_id = ServiceModel(meta_connection).get_by_slug('tracker')['id']
        self._connection.execute('CREATE TEMP TABLE {} (org_id BIGINT, uid BIGINT)'.format(self.temp_admins_table))
        query = '''
               INSERT INTO {table}(uid, org_id)
                   SELECT u.id, u.org_id
                   FROM users as u JOIN organization_services as os
                   ON u.org_id = os.org_id
                   WHERE role='admin' AND os.service_id = {service_id}

                   UNION
                   SELECT o.admin_uid, o.id
                   FROM organizations as o JOIN organization_services as os
                   ON o.id = os.org_id
                   WHERE os.service_id = {service_id}
                   AND o.organization_type != '{cloud_type}';
                '''.format(
            table=self.temp_admins_table,
            service_id=tracker_id,
            cloud_type=settings.CLOUD_ORGANIZATION_TYPE,
        )

        self._connection.execute(query)

    def tz2offset(self, tz):
        try:
            offset = datetime.now(pytz.timezone(tz)).strftime('%z')
            return '{}:{}'.format(offset[:3], offset[3:])
        except Exception:
            with log.fields(tz=tz):
                log.trace().warning('Error during calc offset by tz name')
            return tz

    def get_batch_admin_info(self, uid_org_id):
        uids = [u[0] for u in uid_org_id]
        users_info = app.blackbox_instance.batch_userinfo(
            uids=uids,
            userip=get_localhost_ip_address(),
            chunk_size=200,  # требование паспорта
            dbfields=app.config['BLACKBOX']['dbfields'],
            attributes=TIMEZONE_ATTRIBUTE,
            getphones='bound',
            phone_attributes='1',
        )
        users_info = objects_map_by_id(users_info, key='uid', remove_key=True)
        result_info = []
        for item in uid_org_id:
            u_info = users_info.get(str(item[0]))
            if u_info:
                result_info.append({
                    'uid': item[0],
                    'org_id': item[1],
                    'utc': self.tz2offset(
                        str(u_info.get('attributes', {}).get(TIMEZONE_ATTRIBUTE))
                    ),
                    'first_name': u_info['fields'].get('first_name'),
                    'last_name': u_info['fields'].get('last_name'),
                    'phones': u_info['fields'].get('phones'),
                })
        return result_info


def get_external_org_ids(meta_connection, user_id):
    from intranet.yandex_directory.src.yandex_directory.core.models import (
        UserModel,
    )
    from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
        get_domains_from_db_or_domenator,
        DomainFilter,
    )

    params = (user_id,)
    results = meta_connection.execute("""
    SELECT DISTINCT shard
      FROM organizations
      LEFT JOIN users ON organizations.id = users.org_id
     WHERE users.id = %s AND users.user_type != 'outer_admin'
    """, params)
    all_shards = (item[0] for item in results.fetchall())

    org_ids = []
    user_domain = get_user_domain_from_blackbox(user_id)
    for shard in all_shards:
        with get_main_connection(shard) as shard_connection:
            shard_org_ids = UserModel(shard_connection) \
                            .filter(id=user_id, is_dismissed=False, is_robot=False) \
                            .fields('org_id') \
                            .scalar()
            # Если пользователь доменный, надо исключить из списка
            # организацию, которой принадлежит домен.
            if user_domain and shard_org_ids:
                native_org_id = get_domains_from_db_or_domenator(
                    meta_connection=meta_connection,
                    domain_filter=DomainFilter(org_id=shard_org_ids, master=True, owned=True, name=user_domain),
                    main_connection=shard_connection,
                    one=True,
                )
                if native_org_id:
                    native_org_id = native_org_id['org_id']
                    shard_org_ids = (
                        id
                        for id in shard_org_ids
                        if id != native_org_id
                    )

            org_ids.extend(shard_org_ids)

    return org_ids
