# -*- coding: utf-8 -*-
import re
from collections import defaultdict
from datetime import timedelta

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common import backpressure
from intranet.yandex_directory.src.yandex_directory.common.utils import utcnow
from intranet.yandex_directory.src.yandex_directory.common.cache import cached_in_memory_with_ttl
from intranet.yandex_directory.src.yandex_directory.common.db import get_meta_connection
from intranet.yandex_directory.src.yandex_directory.core.models import ScheduledTasksResultModel
from intranet.yandex_directory.src.yandex_directory.core.models import CallbackEventsModel, WebHookModel
from intranet.yandex_directory.src.yandex_directory.core.models.organization import OrganizationSsoSettingsModel

"""
Функции, собирающие метрики для голована/мониторинга.
Должны возвращать список пар "название метрики"-значение вида [str, int]
Имя метрики должно соотвествовать регэкспу r'^([a-zA-Z0-9\.\-]+_)+([ad][vehmntx]{3}|summ|hgram|max)$'
https://wiki.yandex-team.ru/jandekspoisk/sepe/monitoring/stat-handle/#defoltnyesigopt-suffiksy
"""


def build_closed_service_stats():
    """
    Отдаем 0 или 1 в зависимости от того, закрыт ли сервис
    по итогам проверки smoke-тестами жизненно важных сервисов
    """
    return [
        ['closed_backends_summ', int(backpressure.is_need_to_close_service())],
        ['uwsgi_workers_summ', 1],
    ]


@cached_in_memory_with_ttl(60)
def build_is_analytics_data_saved_to_yt():
    """
    Проверяем, сохранены ли данные для аналитики в YT.
    До момента срабатывания таска (от 4 до 10 часов утра МСК) данных за вчера нет и это нормально.
    Поэтому, просто возвращаем "1" до 6 утра по UTC, когда имеет смысл запускать срабатывание мониторинга.

    При изменении этой логики нужно соответствующим образом поправить время запуска таска
    yandex_directory.core.scheduler.cron.save_stats_data_to_yt_from_all_shards_queue_task!
    """
    from intranet.yandex_directory.src.yandex_directory.core.utils.analytics import AnalyticsToYtSaver

    if utcnow().hour < 7 or not AnalyticsToYtSaver().get_yt_folders_without_data():
        result = 1
    else:
        result = 0

    return [['is_analytics_data_saved_to_yt_annn', result]]


def build_task_queue_stats():
    """
    Определяем показатели очереди задач
    """
    from intranet.yandex_directory.src.yandex_directory.core.models import TaskModel
    from intranet.yandex_directory.src.yandex_directory.common.db import get_shard_numbers, get_main_connection

    def convert_task_name_to_stat_key(task_name):
        """
        my.module.name.TASKCamelCase -> task-camel-case
        """
        class_name = task_name.split('.')[-1]
        s1 = re.sub('(.)([A-Z][a-z]+)', r'\1-\2', class_name)
        return re.sub('([a-z0-9])([A-Z])', r'\1-\2', s1).lower()

    with app.app_context():
        statuses = defaultdict(int)
        for shard in get_shard_numbers():
            with get_main_connection(shard=shard) as main_connection:
                task_state_on_shard = TaskModel(main_connection).count_state()
                count_state_created_recently_on_shard = TaskModel(main_connection).count_state_created_recently()
            for row in task_state_on_shard:
                dashed_task_name = convert_task_name_to_stat_key(row['task_name'])
                key = 'task_queue_states_{}_{}_total_axxx'.format(row['queue'], row['state'])
                statuses[key] += row['count']
                key = 'task_queue_states_{}_{}_{}_axxx'.format(row['queue'], row['state'], dashed_task_name)
                statuses[key] += row['count']
                key = 'task_queue_states_{}_{}_{}_{}_axxx'.format(row['queue'], row['state'], dashed_task_name, shard)
                statuses[key] += row['count']
            for row in count_state_created_recently_on_shard:
                dashed_task_name = convert_task_name_to_stat_key(row['task_name'])
                key = 'task_queue_states_created_last_60_minute_{}_{}_total_axxx'.format(row['queue'], row['state'])
                statuses[key] += row['count']
                key = 'task_queue_states_created_last_60_minute_{}_{}_{}_axxx'.format(row['queue'], row['state'], dashed_task_name)
                statuses[key] += row['count']
                key = 'task_queue_states_created_last_60_minute_{}_{}_{}_{}_axxx'.format(row['queue'], row['state'], dashed_task_name, shard)
                statuses[key] += row['count']

    return list(statuses.items())


def build_event_stats():
    from intranet.yandex_directory.src.yandex_directory.common.db import get_shard_numbers, get_main_connection, get_meta_connection

    with get_meta_connection() as meta_connection:
        callbacks = {x['url']: x['id'] for x in WebHookModel(meta_connection).filter(environment=app.config['ENVIRONMENT'])}

    result = defaultdict(int)
    for shard in get_shard_numbers():
        key_delay = 'event_send_delay_{shard}_{id}_axxx'
        key_count = 'event_unsend_count_{shard}_{id}_axxx'
        key_delay_total = 'event_send_delay_total_axxx'
        key_count_total = 'event_unsend_count_total_axxx'
        with get_main_connection(shard=shard) as main_connection:
            for cnt, delay, callback in CallbackEventsModel(main_connection).get_delays():
                callback_id = callbacks.get(callback, 0)

                result[key_delay.format(shard=shard, id=callback_id)] = delay.total_seconds()
                result[key_count.format(shard=shard, id=callback_id)] += cnt

                result[key_delay_total] = max(result[key_delay_total], delay.total_seconds())
                result[key_count_total] += cnt

    return list(result.items())


def build_sso_delay():
    from intranet.yandex_directory.src.yandex_directory.common.db import get_shard_numbers, get_main_connection

    result = defaultdict(int)

    for shard in get_shard_numbers():
        key_delay = 'sso_sync_delay_{}_axxx'.format(shard)
        key_total_delay = 'sso_sync_total_delay_axxx'

        with get_main_connection(shard=shard) as main_connection:
            delay = OrganizationSsoSettingsModel(main_connection).get_max_delay()
            delay_in_seconds = delay.total_seconds() if delay is not None else 0

            result[key_delay] = delay_in_seconds
            result[key_total_delay] = max(result[key_total_delay], delay_in_seconds)

    return list(result.items())


def build_cron_results():
    with get_meta_connection() as meta_connection:
        signals = []
        for (name, success, count) in ScheduledTasksResultModel(meta_connection).get_for_stats(24 * 60 * 60):
            signal_name = 'scheduler-task_%s_%s_last_day_max' % (name, str(success).lower())
            signals.append(
                [signal_name, count]
            )
        for (name, success, count) in ScheduledTasksResultModel(meta_connection).get_for_stats(1 * 60 * 60):
            signal_name = 'scheduler-task_%s_%s_last_hour_max' % (name, str(success).lower())
            signals.append(
                [signal_name, count]
            )

    return signals
