# -*- coding: utf-8 -*-
from flask import g

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.common.models.types import (
    TYPE_GROUP,
    TYPE_USER,
    MODEL_TYPES,
)
from intranet.yandex_directory.src.yandex_directory.core.features import (
    CAN_WORK_WITHOUT_OWNED_DOMAIN,
    CHANGE_ORGANIZATION_OWNER,
)
from intranet.yandex_directory.src.yandex_directory.core.features.utils import is_feature_enabled
from intranet.yandex_directory.src.yandex_directory.core.models import (
    GroupModel,
    RobotServiceModel,
    UserModel,
    OrganizationServiceModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models.organization import organization_type
from intranet.yandex_directory.src.yandex_directory.core.models.user import UserRoles
from intranet.yandex_directory.src.yandex_directory.core.permission.permissions import (
    global_permissions,
    organization_permissions,
    group_permissions,
    department_permissions,
    user_permissions,
    service_permissions, sso_permissions,
)
from intranet.yandex_directory.src.yandex_directory.core.predicates import And, Not, Or
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    is_outer_org_admin,
    is_deputy_admin,
    is_org_admin,
    is_outer_uid,
    get_user_role,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.users.base import is_organization_owner
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    AuthorizationError,
    ServiceNotFound,
)

from intranet.yandex_directory.src.yandex_directory.connect_services import idm
from intranet.yandex_directory.src import settings
from intranet.yandex_directory.src.yandex_directory.sso.utils import is_sso_available
from intranet.yandex_directory.src.yandex_directory.core.models import DomainModel


def is_outer_admin(user_role, **kwargs):
    return is_outer_org_admin(user_role)


def is_deputy(user_role, **kwargs):
    return is_deputy_admin(role=user_role)


def is_inner_admin(user_role, **kwargs):
    return user_role == UserRoles.admin


def is_admin(user_role, **kwargs):
    return is_org_admin(role=user_role)


def has_owned_domain(has_owned_domains, **kwargs):
    return has_owned_domains


def has_alias_domains(main_connection, org_id, **kwargs):
    return len(DomainModel(main_connection).get_aliases(org_id, owned=None)) > 0


def is_yandex_organization(meta_connection, org_id, **kwargs):
    return is_feature_enabled(meta_connection, org_id, CAN_WORK_WITHOUT_OWNED_DOMAIN)


def is_change_owner_feature_enabled(meta_connection, org_id, **kwargs):
    return is_feature_enabled(meta_connection, org_id, CHANGE_ORGANIZATION_OWNER)


def is_partner_organization(org_type, **kwargs):
    return org_type in organization_type.partner_types


def is_cloud_organization(org_type, **kwargs):
    return org_type in organization_type.cloud_types


def is_free_organization(org_is_free, **kwargs):
    return org_is_free


def is_yandex_project(org_type, **kwargs):
    return org_type == organization_type.yandex_project


def is_education_organization(org_type, **kwargs):
    return org_type == organization_type.education


def is_charity_organization(org_type, **kwargs):
    return org_type == organization_type.charity


def is_portal(org_type, **kwargs):
    return org_type == organization_type.portal


def is_ya_team(org_id, **kwargs):
    return org_id in settings.YA_TEAM_ORG_IDS


def is_group_admin(main_connection, uid, org_id, object_type, object_id, **kwargs):
    if object_type == TYPE_GROUP and object_id:
        group = GroupModel(main_connection).get(
            org_id=org_id,
            group_id=object_id,
            fields=[
                'admins.id'
            ],
        ) or {}
        admin_uids = set(u['id'] for u in group.get('admins', []))
        if uid in admin_uids:
            return True
    else:
        return False


def object_is_admin(meta_connection, main_connection, org_id, object_type, object_id, **kwargs):
    if object_type == TYPE_USER and is_org_admin(
            role=get_user_role(
                meta_connection,
                main_connection,
                org_id,
                object_id,
            )
    ):
        return True
    else:
        return False


def is_self(uid, object_type, object_id, **kwargs):
    return object_type == TYPE_USER and uid == object_id


def is_portal_user(uid, is_cloud=False, **kwargs):
    return is_outer_uid(uid, is_cloud)


def is_robot(main_connection, org_id, uid, is_cloud=False, **kwargs):
    if is_cloud:
        return False
    return RobotServiceModel(main_connection) \
        .filter(org_id=org_id, uid=uid) \
        .count() > 0


def user_object_created_by_directory(object_type, object_id, is_cloud=False, **kwargs):
    if object_type != TYPE_USER or object_id is None:
        # это место а) нелогичное б) фронт его никогда не использует без объекта пользователя
        # но тесты сейчас требуют такого поведения
        # TODO: https://st.yandex-team.ru/DIR-6085
        return True
    return not is_outer_uid(object_id, is_cloud)


def object_is_robot(main_connection, org_id, object_type, object_id, is_cloud=False, **kwargs):
    if object_type == TYPE_USER and object_id and not is_cloud:
        try:
            object_id = int(object_id)
            # роботного пользователя нельзя изменить и удалить
            if is_robot(main_connection, org_id, object_id):
                return True
        except ValueError:
            return False
    else:
        return False


def object_is_outstaff(main_connection, org_id, object_type, object_id, **kwargs):
    if object_type == TYPE_USER and object_id:
        try:
            object_id = int(object_id)
        except ValueError:
            with log.fields(user_id=object_id):
                log.warning('User id should be integer')
            return False

        user = UserModel(main_connection) \
               .filter(org_id=org_id, id=object_id) \
               .fields('is_outstaff') \
               .one()
        if user:
            return user['is_outstaff']
        else:
            with log.fields(user_id=object_id, org_id=org_id):
                log.warning('User not found')

    return False


def has_billing_info(org_with_billing_info, **kwargs):
    return org_with_billing_info


def object_is_portal_account(object_type, object_id, is_cloud=False, **kwargs):
    if object_type == TYPE_USER:
        return is_outer_uid(object_id, is_cloud)
    else:
        return False


def object_is_org_owner(main_connection, org_id, object_type, object_id, **kwargs):
    if object_type == TYPE_USER:
        return is_organization_owner(
            main_connection,
            org_id,
            object_id
        )
    else:
        return False


def sso_available(meta_connection, org_id, **kwargs):
    return is_sso_available(meta_connection, org_id)


def no_pdd_users(main_connection, org_id, **kwargs):
    return UserModel(main_connection).filter(org_id=org_id, is_pdd=True, is_sso=False).count() == 0


def is_responsible_for_service(main_connection, org_id, object_type, **kwargs):
    if object_type not in MODEL_TYPES:
        if getattr(g, 'user', None):
            try:
                object_type = object_type or (g.service and g.service.identity.lower())
                if not object_type:
                    return False
                service = OrganizationServiceModel(main_connection).get_by_slug(
                    org_id=org_id, service_slug=object_type,
                    fields=['id', 'responsible_id'],
                )
            except (AuthorizationError, ServiceNotFound):
                return False
            return service.get('responsible_id') == g.user.passport_uid
    return False


rules = {
    global_permissions.add_domains: And(
        is_admin,
        Not(is_ya_team),
    ),
    global_permissions.add_groups: Or(
        has_owned_domain,
        is_yandex_organization
    ),
    global_permissions.add_departments: And(
        Or(is_admin, is_deputy),
        Or(has_owned_domain, is_yandex_organization),
        Not(is_ya_team)
    ),
    global_permissions.invite_users: And(
        is_yandex_organization,
        Not(is_ya_team)
    ),
    global_permissions.use_connect: And(
        Not(is_cloud_organization),
    ),
    # Пока фронт не смотрит на это право, и запрещает удалять мастер и технический домен.
    # Когда фронт перейдет наиспользование права remove_domain, это правило надо будет дополнить:
    # нельзя удалять мастер-домен, технический домен
    # Так же это будет не глобальное, а объектное право
    # domain_permissions.remove_domain: And(
    #     is_admin,
    #     domain_can_be_removed,
    #     Not(is_ya_team),
    # ),
    global_permissions.remove_domain: And(
        is_admin,
        Not(is_ya_team),
    ),
    global_permissions.add_users: And(
        Or(is_admin, is_deputy),
        has_owned_domain,
    ),
    global_permissions.can_pay: And(
        is_inner_admin,
        Or(has_owned_domain, is_yandex_organization),
        Not(is_partner_organization),
        Not(is_ya_team),
    ),
    # Смена на платный Коннект недоступна для
    # всех организаций с некоторых пор,
    # та как они все теперь с ЯОрг режимом.
    # На это указывает is_yandex_organization
    # предикат.
    global_permissions.change_subscription_plan: And(
        is_inner_admin,
        has_owned_domain,
        Not(is_partner_organization),
        Not(is_yandex_organization),
        Not(is_ya_team),
    ),
    global_permissions.remove_departments: And(
        Or(is_admin, is_deputy),
        Or(has_owned_domain, is_yandex_organization),
        Not(is_ya_team),
    ),
    global_permissions.change_master_domain: And(
        is_admin,
        has_owned_domain,
        Not(is_ya_team),
    ),
    global_permissions.manage_services: And(
        is_admin,
        Not(is_portal),
        Not(is_ya_team),
    ),
    global_permissions.manage_cloud_services_enable: And(
        is_cloud_organization,
        Not(is_portal),
        Not(is_ya_team),
    ),
    # Когда фронт полностью перейдет на manage_tracker, это право надо убрать
    global_permissions.manage_licenses: And(
        is_inner_admin,
        Or(
            has_owned_domain,
            is_yandex_organization,
        ),
        Not(is_ya_team),
    ),
    global_permissions.manage_tracker: And(
        is_inner_admin,
        Or(
            has_owned_domain,
            is_yandex_organization,
        ),
        Not(is_ya_team),
        Not(is_portal),
    ),
    global_permissions.change_logo: And(
        is_admin,
        Not(is_ya_team),
    ),
    global_permissions.activate_promocode: And(
        is_admin,
        Or(has_owned_domain, is_yandex_organization),
        Not(is_partner_organization),
        Not(is_education_organization),
        Not(is_charity_organization),
        Not(is_yandex_project),
        Not(is_ya_team),
    ),
    global_permissions.migrate_emails: And(
        is_admin,
        has_owned_domain,
        Not(is_ya_team),
        Not(is_portal),
    ),
    global_permissions.edit_dns: And(
        is_admin,
        has_owned_domain,
        Not(is_ya_team),
    ),
    global_permissions.manage_mail: And(
        is_admin,
        has_owned_domain,
        Not(is_ya_team),
    ),
    organization_permissions.edit: And(
        is_admin,
        Or(
            has_owned_domain,
            is_yandex_organization,
        ),
        Not(is_ya_team),
    ),
    organization_permissions.add_organization: And(
        is_outer_admin,
        has_owned_domain,
    ),
    organization_permissions.delete: And(
        Or(
            is_outer_admin,
            And(is_inner_admin, is_yandex_organization),
        ),
        Not(is_partner_organization),
        Not(is_ya_team),
    ),
    organization_permissions.change_owner: And(
        is_change_owner_feature_enabled,
        is_organization_owner,
    ),
    group_permissions.edit: And(
        Or(
            is_admin,
            is_deputy,
            is_group_admin,
        ),
        Or(has_owned_domain, is_yandex_organization),
    ),
    department_permissions.edit: And(
        Or(is_admin, is_deputy),
        Or(has_owned_domain, is_yandex_organization),
    ),
    user_permissions.change_alias: And(
        user_object_created_by_directory,
        Or(
            is_admin,
            And(is_deputy, Not(object_is_admin))
        ),
        Or(has_owned_domain, is_yandex_organization),
        Not(is_ya_team),
    ),
    user_permissions.make_admin: And(
        Or(
            has_owned_domain,
            is_yandex_organization,
        ),
        is_admin,
        Not(object_is_robot),
        Not(is_self),
        Not(is_ya_team),
    ),
    user_permissions.edit: And(
        Or(
            is_admin,
            And(is_deputy, Not(object_is_admin))
        ),
        Or(
            has_owned_domain,
            is_yandex_organization,
        ),
        Not(is_ya_team),
        Not(object_is_robot),
    ),
    user_permissions.edit_info: And(
        Or(
            is_admin,
            And(is_deputy, Not(object_is_admin)),
            is_self,
        ),
        Not(object_is_robot),
        has_owned_domain,
        Not(is_ya_team),
        Not(object_is_portal_account),
    ),
    user_permissions.block: And(
        user_object_created_by_directory,
        Or(
            is_admin,
            And(is_deputy, Not(object_is_admin))
        ),
        Not(object_is_robot),
        Not(is_self),
        has_owned_domain,
        Not(is_ya_team),
        Not(object_is_portal_account),
        Not(object_is_org_owner)
    ),
    user_permissions.change_avatar: And(
        Or(
            is_admin,
            And(is_deputy, Not(object_is_admin)),
            is_self,
        ),
        has_owned_domain,
        Not(is_ya_team),
        Not(object_is_robot),
        Not(object_is_portal_account),
    ),
    user_permissions.change_password: And(
        Or(
            is_admin,
            And(is_deputy, Not(object_is_admin)),
        ),
        Not(object_is_robot),
        has_owned_domain,
        Not(is_ya_team),
        Not(object_is_portal_account),
    ),
    user_permissions.dismiss: And(
        Or(
            is_admin,
            And(is_deputy, Not(object_is_admin)),
        ),
        Not(object_is_robot),
        Or(has_owned_domain, is_yandex_organization),
        Not(is_ya_team),
    ),
    global_permissions.leave_organization: And(
        is_portal_user,
    ),
    user_permissions.edit_birthday: And(
        Or(
            is_admin,
            And(is_deputy, Not(object_is_admin)),
            is_self,
        ),
        Not(object_is_robot),
        has_owned_domain,
        Not(is_ya_team),
        Not(object_is_portal_account),
    ),
    user_permissions.edit_contacts: And(
        Or(
            is_admin,
            And(is_deputy, Not(object_is_admin)),
            is_self,
        ),
        Not(object_is_robot),
        has_owned_domain,
        Not(is_ya_team),
        Not(object_is_portal_account),
    ),
    user_permissions.change_role: And(
        Or(
            has_owned_domain,
            is_yandex_organization,
        ),
        is_admin,
        Not(object_is_robot),
        Not(object_is_org_owner),
        Not(is_ya_team),
    ),
    user_permissions.move_to_staff: And(
        is_admin,
        object_is_outstaff,
        Not(object_is_robot),
        Not(is_ya_team),
    ),
    user_permissions.can_change_relations: Or(
        is_admin,
        is_responsible_for_service,
        idm.permissions_for_resource_access,
    ),
    service_permissions.update_service_data: Or(
        is_admin,
        is_responsible_for_service,
    ),
    sso_permissions.read_sso_settings: And(
        is_admin,
    ),
    sso_permissions.write_sso_settings: And(
        is_admin,
        has_owned_domain,
        Not(has_alias_domains),
        sso_available,
        no_pdd_users
    )
}
