import collections.abc
# -*- coding: utf-8 -*-


class Predicate(object):
    def evaluate(self, cache, *args, **kwargs):
        raise NotImplementedError


class ConstPredicate(Predicate):
    def __init__(self, value):
        self.value = value

    def evaluate(self, cache, *args, **kwargs):
        return self.value


class CacheablePredicate(Predicate):
    def _get_cache_key(self, *args, **kwargs):
        raise NotImplementedError

    def _do_evaluate(self, cache, *args, **kwargs):
        raise NotImplementedError

    def evaluate(self, cache, *args, **kwargs):
        key = self._get_cache_key(*args, **kwargs)
        if key not in cache:
            cache[key] = self._do_evaluate(cache, *args, **kwargs)
        return cache[key]


class CallablePredicate(CacheablePredicate):
    def __init__(self, callable):
        self.callable = callable

    def _get_cache_key(self, *args, **kwargs):
        return self.callable.__name__

    def _do_evaluate(self, cache, *args, **kwargs):
        return self.callable(*args, **kwargs)


def maybe_predicate(value):
    if isinstance(value, Predicate):
        return value
    elif isinstance(value, bool):
        return ConstPredicate(value)
    elif isinstance(value, collections.abc.Callable):
        return CallablePredicate(value)
    else:
        raise TypeError('Is not predicate.')


class Always(Predicate):
    """Этот предикат всегда возвращает True.
    """
    def evaluate(self, cache, *args, **kwargs):
        return True


class And(Predicate):
    def __init__(self, predicate1, *predicates):
        self.predicates = list(map(
            maybe_predicate,
            [predicate1] + list(predicates),
        ))

    def evaluate(self, cache, *args, **kwargs):
        for predicate in self.predicates:
            if not predicate.evaluate(cache, *args, **kwargs):
                return False
        return True


class Or(Predicate):
    def __init__(self, predicate1, *predicates):
        self.predicates = list(map(
            maybe_predicate,
            [predicate1] + list(predicates),
        ))

    def evaluate(self, cache, *args, **kwargs):
        for predicate in self.predicates:
            if predicate.evaluate(cache, *args, **kwargs):
                return True
        return False


class Not(Predicate):
    def __init__(self, predicate):
        self.predicate = maybe_predicate(predicate)

    def evaluate(self, cache, *args, **kwargs):
        return not self.predicate.evaluate(cache, *args, **kwargs)


class Is(Predicate):
    """
    Этот предикат нужен, если в правиле используется только одно условие.
    Пример:
    rules = {
        ...,
        some_permission: Is(is_outer_admin),
        ...
    }
    """
    def __init__(self, predicate):
        self.predicate = maybe_predicate(predicate)

    def evaluate(self, cache, *args, **kwargs):
        return self.predicate.evaluate(cache, *args, **kwargs)
