# encoding: UTF-8

import hmac
import hashlib
import base64
import binascii
import datetime as dt

from Crypto import Random
from Crypto.Cipher import AES
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log


class BaseToken(object):

    def __init__(self, registry):
        self.gen_secret = app.config['REGISTRAR_GEN_TOKEN_SECRET'].encode()
        self.store_secret = app.config['REGISTRAR_STORE_TOKEN_SECRET'].encode()
        self.registry = registry

    def _generate_raw_token(self, sha, *args):
        messstr = ''
        for i, arg in enumerate(args, 1):
            messstr += str(arg) + '{} |{}|'.format(args, i)
        messstr += str(dt.datetime.now())
        messstr = messstr.encode('utf-8')
        return hmac.new(self.gen_secret, messstr, sha)

    def serialize_token(self, token):
        return token

    def unserialize_token(self, token):
        return token

    def generate_token(self, *args):
        raise NotImplemented

    def set_token(self, entity, token):
        entity.token = token
        self.registry.save(entity)

    def get_or_create_entity_token(self, entity_id, *generate):
        """
        Получить токен для сущности
        """
        entity = self.registry.get(entity_id)

        if not entity:
            return
        if entity.token:
            return self.unserialize_token(entity.token)
        token = self.generate_token(entity_id, *generate)
        self.set_token(entity, self.serialize_token(token))
        return token

    def del_entity_token(self, entity_id):
        """
        Сбросить токен
        """
        entity = self.registry.get(entity_id)
        if not entity:
            return
        self.set_token(entity, None)


class PlainToken(BaseToken):
    """
    Для протсых токенов
    Это токены доменов для v1 апи ПДД
    """

    def generate_token(self, *args):
        return self._generate_raw_token(hashlib.sha224, *args).hexdigest()


class CryptToken(BaseToken):
    """
    Для шифрованных токенов
    Это токены регистраторов и доменов для v2 апи ПДД
    """

    def generate_token(self, *args):
        raw_token = self._generate_raw_token(hashlib.sha256, *args).digest()
        return base64.b32encode(raw_token).rstrip(b'=')

    def serialize_token(self, token):
        cipher = AES.new(self.store_secret, AES.MODE_ECB)
        ljust_value = b'='
        if isinstance(token, str):
            ljust_value = '='
        raw_token = base64.b32decode(token.upper().ljust(56, ljust_value))
        return base64.b32encode(cipher.encrypt(raw_token)).decode('utf-8')

    def unserialize_token(self, token):
        cipher = AES.new(self.store_secret, AES.MODE_ECB)
        dec_token = cipher.decrypt(base64.b32decode(token))
        return base64.b32encode(dec_token).rstrip(b'=')

    def get_entity_by_token(self, token):
        try:
            crypt_token = self.serialize_token(token)
        except binascii.Error:
            log.warning('Got error while parsing token')
            return
        return self.registry.find_one_by_token(crypt_token)


class RegistrarPassword(object):

    def __init__(self):
        self.store_secret = app.config['REGISTRAR_STORE_PASSWORD_SECRET'].encode()

    def decrypt_password(self, registrar):
        password = registrar.password
        if registrar.pdd_version == 'old':
            return password
        if not registrar.iv:
            return ''
        iv_dec = base64.b32decode(registrar.iv)
        cipher = AES.new(self.store_secret, AES.MODE_CFB, iv_dec)

        return cipher.decrypt(
            base64.b32decode(password)
        )[:registrar.plength]

    def encrypt_password(self, password, registrar):
        if registrar.pdd_version == 'old':
            registrar.password = password
            return registrar

        iv = Random.new().read(AES.block_size)
        iv_enc = base64.b32encode(iv)

        cipher = AES.new(self.store_secret, AES.MODE_CFB, iv)
        plength = len(password)
        password_prepared = password
        if len(password) % 16:
            password_prepared = password + '0' * (16 - len(password) % 16)
        registrar.plength = plength
        registrar.iv = iv_enc.decode('utf-8')
        registrar.password = base64.b32encode(
            cipher.encrypt(password_prepared)
        ).decode('utf-8')
        return registrar
