# -*- coding: utf-8 -*-
import hashlib

from lxml import etree
from lxml.etree import LxmlSyntaxError
from requests import Response, RequestException

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.meta.models import Registrar
from intranet.yandex_directory.src.yandex_directory.zora.exceptions import ZoraError
from intranet.yandex_directory.src.yandex_directory.common.utils import to_punycode
from .exceptions import RegistrarInteractionException


class RegistrarClient:
    def __init__(self, registrar):
        # type: (Registrar) -> None
        self.registrar = registrar
        self.password = registrar['password']

    def __request(self, url, params=None):
        # type: (str, dict or None) -> Response
        if params is None:
            params = {}

        if 'domain' in params:
            params['domain'] = to_punycode(params['domain'])

        if not url.startswith('http'):
            url = 'http://' + url
        if url.startswith('https') and self.registrar['pdd_version'] == 'old':
            params['password'] = self.password
        else:
            sign = b''
            for param in params.values():
                if not isinstance(param, bytes):
                    param = str(param).encode('utf-8')

                sign += param
            password = self.password
            if isinstance(password, str):
                password = password.encode('utf-8')
            sign += password
            sign = hashlib.sha1(sign).hexdigest()
            params["sign"] = sign

        try:
            return app.zora_client.get(url, params=params)
        except (ZoraError, RequestException) as e:
            with log.fields(url=url):
                log.error('Error during request to registrar: %s', str(e))
            raise RegistrarInteractionException()

    def call_check_payed_callback(self, domain_name, service_id):
        # type: (str, int) -> bool
        url = self.registrar['validate_domain_url']
        # колбек обязательный для регистраторов v1, дергаем его в любом случае

        response = self.__request(url, {
            'domain': domain_name,
            'service_id': service_id,
        })
        try:
            response_text = response.text
            if isinstance(response_text, str):
                response_text = response_text.encode('utf-8')
            response_xml = etree.fromstring(response_text)
            if response_xml.find("error").text == "ok":
                if response_xml.find("value").text == "True":
                    return True
        except LxmlSyntaxError:
            log.warning('Registrar response parsing error')

        return False

    def call_added_init_registrar_callback(self, domain_name, secret_value):
        # type: (str, str) -> bool
        url = self.registrar['domain_added_callback_url']
        if not url:
            return True

        response = self.__request(url, {
            'domain': domain_name,
            'secret_name': secret_value,
            'secret_value': secret_value,
        })

        try:
            response_text = response.text
            if isinstance(response_text, str):
                response_text = response_text.encode('utf-8')
            response_xml = etree.fromstring(response_text)
            if response_xml.find("error").text == "ok":
                return True
        except LxmlSyntaxError:
            log.warning('Registrar response parsing error')

        return False

    def call_domain_verified_callback(self, domain_name):
        # type: (str) -> None
        url = self.registrar['domain_verified_callback_url']
        if not url:
            return

        self.__request(url, {
            'domain': domain_name,
        })

    def call_domain_deleted_callback(self, domain_name):
        # type: (str) -> None
        url = self.registrar['domain_deleted_callback_url']
        if not url:
            return

        self.__request(url, {
            'domain': domain_name,
        })
