# -*- coding: utf-8 -*-
import logging

from datetime import timedelta

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.core.cloud.tasks import SyncCloudOrgsTask
from intranet.yandex_directory.src.yandex_directory.core.models import ScheduledTasksResultModel

from intranet.yandex_directory.src.yandex_directory.core.scheduler.utils import (
    scheduled_job,
    short_scheduled_job,
)
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_shard_numbers,
    get_main_connection, get_meta_connection,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import utcnow
from intranet.yandex_directory.src.yandex_directory.core.db.utils import by_shards
from intranet.yandex_directory.src.yandex_directory.core.events.utils import notify_about_new_events

from intranet.yandex_directory.src.yandex_directory.core.commands.create_robots import Command as CreateRobotsCommand
from intranet.yandex_directory.src.yandex_directory.core.commands.check_inactive_contracts import Command as CheckInactiveContractsCommand
from intranet.yandex_directory.src.yandex_directory.core.commands.check_balance_and_debt import Command as CheckBalanceDebtCommand
from intranet.yandex_directory.src.yandex_directory.core.commands.notify_about_trial_expiration import Command as NotifyAboutTrialExpirationCommand
from intranet.yandex_directory.src.yandex_directory.core.commands.disable_expired_promocodes import DisableExpiredPromocodesCommand
from intranet.yandex_directory.src.yandex_directory.core.commands.disable_service_trial_expired import Command as DisableServiceTrialExpiredCommand
from intranet.yandex_directory.src.yandex_directory.core.billing.tasks import SaveBillingInfoToYTTask
from intranet.yandex_directory.src.yandex_directory.core.utils import user_inconsistencies
from intranet.yandex_directory.src.yandex_directory.core.utils.tasks import SaveAnalyticsMetaTask
from intranet.yandex_directory.src.yandex_directory.core.maillist_check.tasks import MaillistsCheckTask
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask
from intranet.yandex_directory.src.yandex_directory.access_restore.utils import init_access_restore_task
from intranet.yandex_directory.src.yandex_directory.core.utils.organization.vip import update_vip_reasons_for_all_orgs
from intranet.yandex_directory.src.yandex_directory.core.tasks.tracker import CheckActivityInTrackerTask
from intranet.yandex_directory.src.yandex_directory.sso.tasks import SyncSsoOrgsTask

logger = logging.getLogger('core.scheduler.cron')


def setup_cron(app):
    if not app.config['SCHEDULER_MODE']:
        return

    @scheduled_job(seconds=app.config['ORG_IDS_INCONSISTENCIES_FREQUENCY'])
    def update_org_ids_inconsistencies_count():
        user_inconsistencies.update_cache()

    @scheduled_job(minutes=15)
    def access_restore():
        init_access_restore_task()

    @scheduled_job(minutes=15)
    @by_shards(meta_for_write=False)
    def update_vip_reasons_job(meta_connection, main_connection):
        update_vip_reasons_for_all_orgs(main_connection)

    @scheduled_job(minutes=30)
    def create_robots_task():
        CreateRobotsCommand()(app=app)

    # в отличие от остальных задач, где используется trigger=interval,
    # здесь мы используем cron, и для него параметр указывается в
    # единственном числе
    @scheduled_job(trigger='cron', hour=4)
    def truncate_events():
        """
        Усекаем таблицу events
        Удаляем записи старше `EVENTS_LOG_SIZE_DAYS` дней
        """
        from intranet.yandex_directory.src.yandex_directory.core.models.event import EventModel
        shards = get_shard_numbers()

        for shard in shards:
            with get_main_connection(shard, for_write=True) as connection:
                with log.fields(shard=shard):
                    log.debug('Truncating events')
                    EventModel(connection).truncate(days=app.config['EVENTS_LOG_SIZE_DAYS'])

    @scheduled_job(trigger='cron', hour='1')
    def save_analytics_data():
        """
        Сохраняем аналитические данные в базу
        Запускаем задачу в час ночи UTC.

        Сначала запускаем таск на создание тасков локального сохранения данных во всех таблицах по шардам и
        ставим на выполение таск копирования данных в YT.
        Таск копирования запускается в случае успешного сохранения всех данных локально.
        """
        if not app.config['YT_ANALYTICS_SAVING_ENABLED']:
            log.debug('Skip saving analytics')
            return

        shards = get_shard_numbers()
        # запускаем на одном шарде, потому что сам таск внутри ходит по всем шардам
        with get_main_connection(shard=shards[0], for_write=True) as main_connection:
            try:
                SaveAnalyticsMetaTask(main_connection).delay()
            except DuplicatedTask:
                # если таск уже есть - не нужно пробовать создать его снова
                log.info('DuplicatedTask: SaveAnalyticsMetaTask already in queue')

    @scheduled_job(minutes=15)
    def check_organizations_contracts():
        """
        Проверяем статусы подписания договоров у организаций.
        """
        CheckInactiveContractsCommand()(app=app, org_id=None)

    @scheduled_job(minutes=15)
    def check_organizations_balance_and_debt():
        """
        Проверяем баланс и дату задолженности организаций каждые 15 минут
        """
        if not app.config['CRON_CHECK_BALANCE']:
            log.info('Skip check balance')
            return

        CheckBalanceDebtCommand()(app=app, org_id=None, force_update=False)

    @scheduled_job(minutes=30)
    def notify_about_trial_services():
        """
        Отправляем письма админам организации, у которых срок сервисов скоро истечет
        """
        NotifyAboutTrialExpirationCommand()(app=app, org_id=None)

    @scheduled_job(trigger='cron', hour='2', minute='*/15')
    def disable_organizations_trial_expired_services():
        """
        Каждые 15 минут 2 часа суток выключаем лицензионные сервисы, у которых закончился триал и нет лицензий
        """
        # должно запускаться до create_event_about_trail_ended
        DisableServiceTrialExpiredCommand()(app=app, org_id=None)

    @scheduled_job(trigger='cron', hour='4')
    @by_shards(start_transaction=False)
    def create_event_about_trail_ended(meta_connection, main_connection):
        """
        Создаем события об окончании триала
        только для организаций где сервис включен
        """
        # должно запускаться после disable_organizations_trial_expired_services
        from intranet.yandex_directory.src.yandex_directory.core.models.service import notify_about_trail_ended
        notify_about_trail_ended(main_connection)


    @scheduled_job(trigger='cron', hour=1, minute='*/15')
    def disable_all_expired_promocodes():
        """
        Выключаем все активные, но просроченные промокоды
        """
        DisableExpiredPromocodesCommand()(app=app, org_id=None)

    @short_scheduled_job(seconds=5)
    @by_shards(meta_for_write=False, start_transaction=False)
    def notify_event_subscribers(meta_connection, main_connection):
        """
        Оповещаем подписчиков событий
        """
        with log.name_and_fields('notifications'):
            notify_about_new_events(meta_connection, main_connection)

    @scheduled_job(trigger='cron', hour='1', minute='*/15')
    @by_shards(start_transaction=False)
    def change_service_trial_status_to_expired(meta_connection, main_connection):
        """
        Меняем 'trial_status' на expired, если кончился триал
        """
        from intranet.yandex_directory.src.yandex_directory.core.models.service import set_trial_status_to_expired
        set_trial_status_to_expired(main_connection)

    @scheduled_job(minutes=15)
    @by_shards(meta_for_write=False)
    def unblock_domains(meta_connection, main_connection):
        """
        Разблокируем домены, которые висят заблокированными дольше 10 минут

        """
        query = """UPDATE domains SET blocked_at=NULL
                   WHERE blocked_at IS NOT NULL
                   AND blocked_at < NOW() - INTERVAL '10 minutes'"""
        main_connection.execute(query)

    @scheduled_job(minutes=15)
    @by_shards(meta_for_write=False)
    def truncate_capped_tables(meta_connection, main_connection):
        """
        Удалим записи о события в ProcessedEventsModel старше 4 часов
        На самом деле мы используем данные за час, но оставляем хвост побольше,
        чтобы можно было обработку дыр подебажить.

        Так же, подчищаем успешно выполненные задачи.
        """
        from intranet.yandex_directory.src.yandex_directory.core.models import ProcessedEventsModel
        dt = utcnow() - timedelta(hours=4)
        environment = app.config['ENVIRONMENT']

        ProcessedEventsModel(main_connection).delete(
            filter_data={'created_at__lt': dt, 'environment': environment}
        )
        # Почему-то для TaskModel у меня не сработало использование ORM
        # Выдавалась ошибка:
        # NotImplementedError: "status","finished_at__lt" filters have not been used while filtering
        queues = ['default']
        for queue in queues:
            query = """DELETE FROM tasks
                        WHERE queue = '{env}-{queue}'
                          AND state='success'
                          AND finished_at < NOW() - INTERVAL '4 HOUR'""".format(
                              env=environment,
                              queue=queue)
            main_connection.execute(query)
            query = """DELETE FROM tasks
                        WHERE queue = '{env}-{queue}'
                          AND state='failed'
                          AND created_at < NOW() - INTERVAL '7 DAY'""".format(
                              env=environment,
                              queue=queue)
            main_connection.execute(query)

        query = """DELETE FROM callback_events
                    WHERE done = True
                      AND created_at < NOW() - INTERVAL '48 HOUR'"""
        main_connection.execute(query)

    @scheduled_job(minutes=30)
    @by_shards(meta_for_write=False, start_transaction=False)
    def check_maillists(meta_connection, main_connection):
        """
         В 2:00 по UTC берём все организации, у которых подключен сервис рассылок, и ревизия больше,
         чем в таблице maillist_checks, и создаем таск MaillistsCheckTask,
         который сравнит состояние рассылок в директории и BigML

        """
        from intranet.yandex_directory.src.yandex_directory.core.models import MaillistCheckModel
        orgs = MaillistCheckModel(main_connection).get_organizations_for_sync()
        for org in orgs:
            with main_connection.begin_nested():
                try:
                    MaillistsCheckTask(main_connection).delay(
                        org_id=org['org_id'],
                    )
                except DuplicatedTask:
                    # если таск уже есть - не нужно пробовать создать его снова
                    log.info('DuplicatedTask: MaillistsCheckTask already in queue')


    @scheduled_job(minutes=5)
    def sync_cloud_orgs():
        if app.config['ENVIRONMENT'] not in ('production', 'testing', 'integration_qa'):
            return
        try:
            shards = get_shard_numbers()
            for shard in shards:
                with get_main_connection(shard=shard, for_write=True) as main_connection:
                    SyncCloudOrgsTask(main_connection).delay()
        except DuplicatedTask:
            log.info('DuplicatedTask: SyncCloudOrgsTask already in queue')

    @scheduled_job(trigger='cron', hour='*/4')
    def sync_sso_orgs():
        if app.config['ENVIRONMENT'] not in ('production', 'testing', 'integration_qa'):
            return
        shards = get_shard_numbers()
        for shard in shards:
            with get_main_connection(shard=shard, for_write=True) as main_connection:
                try:
                    SyncSsoOrgsTask(main_connection).delay()
                except DuplicatedTask:
                    log.info('DuplicatedTask: SyncSsoOrgsTask already in queue')

    @scheduled_job(trigger='cron', hour=3)
    def check_activity_in_tracker():
        if app.config['ENVIRONMENT'] not in ('production', 'testing', 'integration_qa'):
            return

        shards = get_shard_numbers()
        with get_main_connection(shard=shards[0], for_write=True) as main_connection:
            CheckActivityInTrackerTask(main_connection).delay()

    @scheduled_job(trigger='cron', hour=1)
    def remove_old_objects():
        """Удаляет старые объекты"""
        with get_meta_connection(for_write=True) as meta_connection:
            ScheduledTasksResultModel(meta_connection).truncate_old()
